-- mods/global_patterns/init.lua

local MODNAME = minetest.get_current_modname() or "global_patterns"



-- active configuration
local active_pattern = "copy_offset"
local dx, dy, dz = 0, 0, 0   -- default offset

local offset = {x=0, y=0, z=0}

local char = nil



-- pattern definitions
local patterns = {}


local function isalpha(ch)
    return ch:match("%a") ~= nil
end


patterns["z"] = function(pos, node_name)
    -- max expansion length
    local max_len = 500

    -- voxelmanip extents (covering a square area around pos)
    local minp = {x = pos.x, y = pos.y, z = pos.z}
    local maxp = {x = pos.x, y = pos.y, z = pos.z + max_len}

    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- fast lookup
    local cid_bridge = minetest.get_content_id(node_name)

    -- allowed nodes to overwrite
    local replaceable = {
        ["air"] = true,
        ["default:water_source"] = true,
        ["default:water_flowing"] = true,
		["default:river_water_source"] = true,
		["default:river_water_flowing"] = true,
		
        ["default:lava_source"] = true,
        ["default:lava_flowing"] = true,
    }

    -- helper to check if node is replaceable
    local function can_replace(vi)
        local nid = data[vi]
        local nname = minetest.get_name_from_content_id(nid)
        return replaceable[nname] or false
    end

 

-- +Z
do
    local lastpos = nil
    for dz = 1, max_len do
        local p = {x = pos.x, y = pos.y, z = pos.z + dz}
        local vi = area:indexp(p)
        if can_replace(vi) then
            data[vi] = cid_bridge
            lastpos = p
        else
            break
        end
    end
    --if lastpos then
       -- minetest.set_node({x=lastpos.x, y=lastpos.y + 1, z=lastpos.z}, {name=node_name})
    --end
end




    -- write back changes
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end


for _, alias in ipairs({"Z", "Z+", "z+", "+Z", "+z"}) do
    patterns[alias] = patterns["z"]
end





patterns["-z"] = function(pos, node_name)
    -- max expansion length
    local max_len = 500

    -- voxelmanip extents (covering a square area around pos)
    local minp = {x = pos.x, y = pos.y, z = pos.z - max_len}
    local maxp = {x = pos.x, y = pos.y, z = pos.z}

    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- fast lookup
    local cid_bridge = minetest.get_content_id(node_name)

    -- allowed nodes to overwrite
    local replaceable = {
        ["air"] = true,
        ["default:water_source"] = true,
        ["default:water_flowing"] = true,
		["default:river_water_source"] = true,
		["default:river_water_flowing"] = true,
		
        ["default:lava_source"] = true,
        ["default:lava_flowing"] = true,
    }

    -- helper to check if node is replaceable
    local function can_replace(vi)
        local nid = data[vi]
        local nname = minetest.get_name_from_content_id(nid)
        return replaceable[nname] or false
    end

   
-- -Z
do
    local lastpos = nil
    for dz = -1, -max_len, -1 do
        local p = {x = pos.x, y = pos.y, z = pos.z + dz}
        local vi = area:indexp(p)
        if can_replace(vi) then
            data[vi] = cid_bridge
            lastpos = p
        else
            break
        end
    end
    --if lastpos then
      --  minetest.set_node({x=lastpos.x, y=lastpos.y + 1, z=lastpos.z}, {name=node_name})
    --end
end


    -- write back changes
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end


for _, alias in ipairs({"Z-", "z-", "-Z"}) do
    patterns[alias] = patterns["-z"]
end



patterns["x"] = function(pos, node_name)
    -- max expansion length
    local max_len = 500

    -- voxelmanip extents (covering a square area around pos)
    local minp = {x = pos.x, y = pos.y, z = pos.z}
    local maxp = {x = pos.x + max_len, y = pos.y, z = pos.z}

    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- fast lookup
    local cid_bridge = minetest.get_content_id(node_name)

    -- allowed nodes to overwrite
    local replaceable = {
        ["air"] = true,
        ["default:water_source"] = true,
        ["default:water_flowing"] = true,
		["default:river_water_source"] = true,
		["default:river_water_flowing"] = true,
		
        ["default:lava_source"] = true,
        ["default:lava_flowing"] = true,
    }

    -- helper to check if node is replaceable
    local function can_replace(vi)
        local nid = data[vi]
        local nname = minetest.get_name_from_content_id(nid)
        return replaceable[nname] or false
    end

    -- expand in +X
    -- +X
do
    local lastpos = nil
    for dx = 1, max_len do
        local p = {x = pos.x + dx, y = pos.y, z = pos.z}
        local vi = area:indexp(p)
        if can_replace(vi) then
            data[vi] = cid_bridge
            lastpos = p
        else
            break
        end
    end
    --if lastpos then
      --  minetest.set_node({x=lastpos.x, y=lastpos.y + 1, z=lastpos.z}, {name = node_name})
    --end
end



    -- write back changes
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end

for _, alias in ipairs({"X" , "X+", "x+", "+X", "+x"}) do
    patterns[alias] = patterns["x"]
end


patterns["x-"] = function(pos, node_name)
    -- max expansion length
    local max_len = 500

    -- voxelmanip extents (covering a square area around pos)
    local minp = {x = pos.x - max_len, y = pos.y, z = pos.z}
    local maxp = {x = pos.x, y = pos.y, z = pos.z}

    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- fast lookup
    local cid_bridge = minetest.get_content_id(node_name)

    -- allowed nodes to overwrite
    local replaceable = {
        ["air"] = true,
        ["default:water_source"] = true,
        ["default:water_flowing"] = true,
		["default:river_water_source"] = true,
		["default:river_water_flowing"] = true,
		
        ["default:lava_source"] = true,
        ["default:lava_flowing"] = true,
    }

    -- helper to check if node is replaceable
    local function can_replace(vi)
        local nid = data[vi]
        local nname = minetest.get_name_from_content_id(nid)
        return replaceable[nname] or false
    end


-- -X
do
    local lastpos = nil
    for dx = -1, -max_len, -1 do
        local p = {x = pos.x + dx, y = pos.y, z = pos.z}
        local vi = area:indexp(p)
        if can_replace(vi) then
            data[vi] = cid_bridge
            lastpos = p
        else
            break
        end
    end
    --if lastpos then
     --   minetest.set_node({x=lastpos.x, y=lastpos.y + 1, z=lastpos.z}, {name=node_name})
    --end
end



    -- write back changes
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end

for _, alias in ipairs({ "X-", "-X", "-x"}) do
    patterns[alias] = patterns["x-"]
end


patterns["y-"] = function(pos, node_name)
    -- max expansion length
    local max_len = 500

    -- voxelmanip extents (covering a square area around pos)
    local minp = {x = pos.x, y = pos.y - max_len, z = pos.z}
    local maxp = {x = pos.x, y = pos.y, z = pos.z}

    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- fast lookup
    local cid_bridge = minetest.get_content_id(node_name)

    -- allowed nodes to overwrite
    local replaceable = {
        ["air"] = true,
        ["default:water_source"] = true,
        ["default:water_flowing"] = true,
		["default:river_water_source"] = true,
		["default:river_water_flowing"] = true,
		
        ["default:lava_source"] = true,
        ["default:lava_flowing"] = true,
    }

    -- helper to check if node is replaceable
    local function can_replace(vi)
        local nid = data[vi]
        local nname = minetest.get_name_from_content_id(nid)
        return replaceable[nname] or false
    end


-- -Y
do
    local lastpos = nil
    for dy = -1, -max_len, -1 do
        local p = {x = pos.x, y = pos.y + dy, z = pos.z}
        local vi = area:indexp(p)
        if can_replace(vi) then
            data[vi] = cid_bridge
            lastpos = p
        else
            break
        end
    end
    --if lastpos then
     --   minetest.set_node({x=lastpos.x, y=lastpos.y + 1, z=lastpos.z}, {name=node_name})
    --end
end
    -- write back changes
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end


for _, alias in ipairs({ "Y-", "-Y", "-y"}) do
    patterns[alias] = patterns["y-"]
end




patterns["wallz+"] = function(pos, node_name)
    -- max expansion length
    local max_len = 500

    -- voxelmanip extents (covering a square area around pos)
    local minp = {x = pos.x, y = pos.y - max_len, z = pos.z}
    local maxp = {x = pos.x, y = pos.y, z = pos.z + max_len}

    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- fast lookup
    local cid_bridge = minetest.get_content_id(node_name)

    -- allowed nodes to overwrite
    local replaceable = {
        ["air"] = true,
        ["default:water_source"] = true,
        ["default:water_flowing"] = true,
		["default:river_water_source"] = true,
		["default:river_water_flowing"] = true,
		
        ["default:lava_source"] = true,
        ["default:lava_flowing"] = true,
    }

    -- helper to check if node is replaceable
    local function can_replace(vi)
        local nid = data[vi]
        local nname = minetest.get_name_from_content_id(nid)
        return replaceable[nname] or false
    end

                    
 

-- +Z
do
    local lastpos = nil
    for dz = 1, max_len do
        local p = {x = pos.x, y = pos.y, z = pos.z + dz}
        local vi = area:indexp(p)
        if can_replace(vi) then
            data[vi] = cid_bridge
            lastpos = p
			--patterns["y-"](p, node_name)
			for dy = -1, -max_len, -1 do
				local py = {x = p.x, y = p.y + dy, z = p.z}
				local vpi = area:indexp(py)
				if can_replace(vpi) then
					data[vpi] = cid_bridge
				else
					break
				end
			end 
        else
			for dy = -1, -max_len, -1 do
				local py = {x = p.x, y = p.y + dy, z = p.z - dz}
				local vpi = area:indexp(py)
				if can_replace(vpi) then
					data[vpi] = cid_bridge
				else
					break
				end
			end 
            break
        end
    end
    --if lastpos then
       -- minetest.set_node({x=lastpos.x, y=lastpos.y + 1, z=lastpos.z}, {name=node_name})
    --end
end




    -- write back changes
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end

patterns["wallz"] = patterns["wallz+"]

patterns["wallz-"] = function(pos, node_name)
    -- max expansion length
    local max_len = 500

    -- voxelmanip extents (covering a square area around pos)
    local minp = {x = pos.x, y = pos.y - max_len, z = pos.z - max_len}
    local maxp = {x = pos.x, y = pos.y, z = pos.z}

    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- fast lookup
    local cid_bridge = minetest.get_content_id(node_name)

    -- allowed nodes to overwrite
    local replaceable = {
        ["air"] = true,
        ["default:water_source"] = true,
        ["default:water_flowing"] = true,
		["default:river_water_source"] = true,
		["default:river_water_flowing"] = true,
		
        ["default:lava_source"] = true,
        ["default:lava_flowing"] = true,
    }

    -- helper to check if node is replaceable
    local function can_replace(vi)
        local nid = data[vi]
        local nname = minetest.get_name_from_content_id(nid)
        return replaceable[nname] or false
    end

   
-- -Z
do
    local lastpos = nil
    for dz = -1, -max_len, -1 do
        local p = {x = pos.x, y = pos.y, z = pos.z + dz}
        local vi = area:indexp(p)
        if can_replace(vi) then
            data[vi] = cid_bridge
            lastpos = p
			--patterns["y-"](p, node_name)	
			for dy = -1, -max_len, -1 do
				local py = {x = p.x, y = p.y + dy, z = p.z}
				local vpi = area:indexp(py)
				if can_replace(vpi) then
					data[vpi] = cid_bridge
				else
					break
				end
			end 			
        else
			for dy = -1, -max_len, -1 do
				local py = {x = p.x, y = p.y + dy, z = p.z - dz}
				local vpi = area:indexp(py)
				if can_replace(vpi) then
					data[vpi] = cid_bridge
				else
					break
				end
			end 
            break
        end
    end
    --if lastpos then
      --  minetest.set_node({x=lastpos.x, y=lastpos.y + 1, z=lastpos.z}, {name=node_name})
    --end
end


    -- write back changes
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end


patterns["wallx+"] = function(pos, node_name)
    -- max expansion length
    local max_len = 500

    -- voxelmanip extents (covering a square area around pos)
    local minp = {x = pos.x, y = pos.y - max_len, z = pos.z}
    local maxp = {x = pos.x + max_len, y = pos.y, z = pos.z}

    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- fast lookup
    local cid_bridge = minetest.get_content_id(node_name)

    -- allowed nodes to overwrite
    local replaceable = {
        ["air"] = true,
        ["default:water_source"] = true,
        ["default:water_flowing"] = true,
		["default:river_water_source"] = true,
		["default:river_water_flowing"] = true,
		
        ["default:lava_source"] = true,
        ["default:lava_flowing"] = true,
    }

    -- helper to check if node is replaceable
    local function can_replace(vi)
        local nid = data[vi]
        local nname = minetest.get_name_from_content_id(nid)
        return replaceable[nname] or false
    end

    -- expand in +X
    -- +X
do
    local lastpos = nil
    for dx = 1, max_len do
        local p = {x = pos.x + dx, y = pos.y, z = pos.z}
        local vi = area:indexp(p)
        if can_replace(vi) then
            data[vi] = cid_bridge
            lastpos = p
			--patterns["y-"](p, node_name)
			for dy = -1, -max_len, -1 do
				local py = {x = p.x, y = p.y + dy, z = p.z}
				local vpi = area:indexp(py)
				if can_replace(vpi) then
					data[vpi] = cid_bridge
				else
					break
				end
			end 
        else
			for dy = -1, -max_len, -1 do
				local py = {x = p.x - dx, y = p.y + dy, z = p.z}
				local vpi = area:indexp(py)
				if can_replace(vpi) then
					data[vpi] = cid_bridge
				else
					break
				end
			end 
            break
        end
    end
    --if lastpos then
      --  minetest.set_node({x=lastpos.x, y=lastpos.y + 1, z=lastpos.z}, {name = node_name})
    --end
end



    -- write back changes
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end

patterns["wallx"] = patterns["wallx+"]

patterns["wallx-"] = function(pos, node_name)
    -- max expansion length
    local max_len = 500

    -- voxelmanip extents (covering a square area around pos)
    local minp = {x = pos.x - max_len, y = pos.y - max_len, z = pos.z}
    local maxp = {x = pos.x, y = pos.y, z = pos.z}

    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- fast lookup
    local cid_bridge = minetest.get_content_id(node_name)

    -- allowed nodes to overwrite
    local replaceable = {
        ["air"] = true,
        ["default:water_source"] = true,
        ["default:water_flowing"] = true,
		["default:river_water_source"] = true,
		["default:river_water_flowing"] = true,
		
        ["default:lava_source"] = true,
        ["default:lava_flowing"] = true,
    }

    -- helper to check if node is replaceable
    local function can_replace(vi)
        local nid = data[vi]
        local nname = minetest.get_name_from_content_id(nid)
        return replaceable[nname] or false
    end


-- -X
do
    local lastpos = nil
    for dx = -1, -max_len, -1 do
        local p = {x = pos.x + dx, y = pos.y, z = pos.z}
        local vi = area:indexp(p)
        if can_replace(vi) then
            data[vi] = cid_bridge
            lastpos = p
			--patterns["y-"](p, node_name)
			for dy = -1, -max_len, -1 do
				local py = {x = p.x, y = p.y + dy, z = p.z}
				local vpi = area:indexp(py)
				if can_replace(vpi) then
					data[vpi] = cid_bridge
				else
					break
				end
			end 
        else
			for dy = -1, -max_len, -1 do
				local py = {x = p.x - dx, y = p.y + dy, z = p.z}
				local vpi = area:indexp(py)
				if can_replace(vpi) then
					data[vpi] = cid_bridge
				else
					break
				end
			end 
            break
        end
    end
    --if lastpos then
     --   minetest.set_node({x=lastpos.x, y=lastpos.y + 1, z=lastpos.z}, {name=node_name})
    --end
end



    -- write back changes
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end





--[[local letters = {
    A = {
       {0,1,0,1,0},
        {1,0,1,0,1},
        {1,1,1,1,1},
        {1,0,0,0,1},
        {1,0,0,0,1},
    },
    B = {
        {1,1,1,0,0},
        {1,0,0,1,0},
        {1,1,1,0,0},
        {1,0,0,1,0},
        {1,1,1,0,0},
    },
    -- Add more letters here
}

]]--

letters = {
    A = { {0,0,1,0,0}, {0,1,0,1,0}, {1,0,0,0,1}, {1,1,1,1,1}, {1,0,0,0,1}, {1,0,0,0,1} },
    B = { {1,1,1,1,0}, {1,0,0,0,1}, {1,1,1,1,0}, {1,0,0,0,1}, {1,0,0,0,1}, {1,1,1,1,0} },
    C = { {0,1,1,1,0}, {1,0,0,0,1}, {1,0,0,0,0}, {1,0,0,0,0}, {1,0,0,0,1}, {0,1,1,1,0} },
    D = { {1,1,1,1,0}, {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {1,1,1,1,0} },
    E = { {1,1,1,1,1}, {1,0,0,0,0}, {1,1,1,1,1}, {1,0,0,0,0}, {1,0,0,0,0}, {1,1,1,1,1} },
    F = { {1,1,1,1,1}, {1,0,0,0,0}, {1,1,1,1,1}, {1,0,0,0,0}, {1,0,0,0,0}, {1,0,0,0,0} },
    G = { {0,1,1,1,0}, {1,0,0,0,1}, {1,0,0,0,0}, {1,0,1,1,1}, {1,0,0,0,1}, {0,1,1,1,0} },
    H = { {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {1,1,1,1,1}, {1,0,0,0,1}, {1,0,0,0,1} },
    I = { {1,1,1,1,1}, {0,0,1,0,0}, {0,0,1,0,0}, {0,0,1,0,0}, {0,0,1,0,0}, {1,1,1,1,1} },
    J = { {1,1,1,1,1}, {0,0,0,1,0}, {0,0,0,1,0}, {0,0,0,1,0}, {1,0,0,1,0}, {0,1,1,0,0} },
    K = { {1,0,0,1,0}, {1,0,1,0,0}, {1,1,0,0,0}, {1,0,1,0,0}, {1,0,0,1,0}, {1,0,0,0,1} },
    L = { {1,0,0,0,0}, {1,0,0,0,0}, {1,0,0,0,0}, {1,0,0,0,0}, {1,0,0,0,0}, {1,1,1,1,1} },
    M = { {1,0,0,0,1}, {1,1,0,1,1}, {1,0,1,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1} },
    N = { {1,0,0,0,1}, {1,1,0,0,1}, {1,0,1,0,1}, {1,0,0,1,1}, {1,0,0,0,1}, {1,0,0,0,1} },
    O = { {0,1,1,1,0}, {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {0,1,1,1,0} },
    P = { {1,1,1,1,0}, {1,0,0,0,1}, {1,1,1,1,0}, {1,0,0,0,0}, {1,0,0,0,0}, {1,0,0,0,0} },
    Q = { {0,1,1,1,0}, {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,1,0}, {0,1,1,0,1} },
    R = { {1,1,1,1,0}, {1,0,0,0,1}, {1,1,1,1,0}, {1,0,1,0,0}, {1,0,0,1,0}, {1,0,0,0,1} },
    S = { {0,1,1,1,1}, {1,0,0,0,0}, {0,1,1,1,0}, {0,0,0,0,1}, {0,0,0,0,1}, {1,1,1,1,0} },
    T = { {1,1,1,1,1}, {0,0,1,0,0}, {0,0,1,0,0}, {0,0,1,0,0}, {0,0,1,0,0}, {0,0,1,0,0} },
    U = { {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {0,1,1,1,0} },
    V = { {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {0,1,0,1,0}, {0,0,1,0,0} },
    W = { {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {1,0,0,0,1}, {1,0,1,0,1}, {0,1,0,1,0} },
    X = { {1,0,0,0,1}, {0,1,0,1,0}, {0,0,1,0,0}, {0,1,0,1,0}, {0,1,0,1,0}, {1,0,0,0,1} },
    Y = { {1,0,0,0,1}, {0,1,0,1,0}, {0,0,1,0,0}, {0,0,1,0,0}, {0,0,1,0,0}, {0,0,1,0,0} },
    Z = { {1,1,1,1,1}, {0,0,0,1,0}, {0,0,1,0,0}, {0,1,0,0,0}, {1,0,0,0,0}, {1,1,1,1,1} },
    ["0"] = { {0,1,1,1,0}, {1,0,0,1,1}, {1,0,1,0,1}, {1,0,1,0,1}, {1,1,0,0,1}, {0,1,1,1,0} },
    ["1"] = { {0,0,1,0,0}, {0,1,1,0,0}, {1,0,1,0,0}, {0,0,1,0,0}, {0,0,1,0,0}, {1,1,1,1,1} },
    ["2"] = { {0,1,1,1,0}, {1,0,0,0,1}, {0,0,0,1,0}, {0,0,1,0,0}, {0,1,0,0,0}, {1,1,1,1,1} },
    ["3"] = { {1,1,1,1,0}, {0,0,0,0,1}, {0,1,1,1,0}, {0,0,0,0,1}, {0,0,0,0,1}, {1,1,1,1,0} },
    ["4"] = { {0,0,0,1,0}, {0,0,1,1,0}, {0,1,0,1,0}, {1,0,0,1,0}, {1,1,1,1,1}, {0,0,0,1,0} },
    ["5"] = { {1,1,1,1,1}, {1,0,0,0,0}, {1,1,1,1,0}, {0,0,0,0,1}, {1,0,0,0,1}, {0,1,1,1,0} },
    ["6"] = { {0,1,1,1,1}, {1,0,0,0,0}, {1,1,1,1,0}, {1,0,0,0,1}, {1,0,0,0,1}, {0,1,1,1,0} },
    ["7"] = { {1,1,1,1,1}, {0,0,0,0,1}, {0,0,0,1,0}, {0,0,1,0,0}, {0,1,0,0,0}, {1,0,0,0,0} },
    ["8"] = { {0,1,1,1,0}, {1,0,0,0,1}, {0,1,1,1,0}, {1,0,0,0,1}, {1,0,0,0,1}, {0,1,1,1,0} },
    ["9"] = { {0,1,1,1,0}, {1,0,0,0,1}, {1,0,0,0,1}, {0,1,1,1,1}, {0,0,0,0,1}, {0,0,0,0,1} },
}





patterns["letter"] = function(pos , node_name)
    local arr = letters[char]
    if not arr then return end  -- safety: skip unknown letters

    local air = "air"
    local cid_node = minetest.get_content_id(node_name)
    local cid_air  = minetest.get_content_id(air)

    local manip = minetest.get_voxel_manip()
    local minp, maxp = manip:read_from_map(pos, pos)
    local area = VoxelArea:new{MinEdge=minp, MaxEdge=maxp}
    local data = manip:get_data()
	
	local char

    -- loop rows/cols
    for i = 1, 6 do
        for j = 1, 5 do
            local x, y, z = pos.x, pos.y, pos.z

            -- orientation: place along X or Z
            if dx ~= 0 then
                x = pos.x + j - 1
                y = pos.y + (6 - i)  -- flip so top row is top
				--char = dx
            elseif dz~= 0 then
                z = pos.z + j - 1
                y = pos.y + (6 - i)
				--char = dz
			else
				break
            end

            local vi = area:index(x, y, z)
            if arr[i][j] == 1 then
                data[vi] = cid_node
            else
                data[vi] = cid_air
            end
        end
    end

    manip:set_data(data)
    manip:write_to_map()
    manip:update_map()
end








patterns["general_cuboid"] = function(pos, node_name)
    local cid = minetest.get_content_id(node_name)

    local x_min, x_max = -dx, dx
    local z_min, z_max = -dz, dz

    local y_min, y_max
    if dy >= 0 then
        y_min, y_max = 0, dy - 1
    else
        y_min, y_max = dy + 1, 0
    end
    y_max = (dy == 0) and 0 or y_max  -- flat layer if dy = 0

    -- Determine voxelmanip boundaries
    local minp = {x = pos.x + x_min, y = pos.y + y_min, z = pos.z + z_min}
    local maxp = {x = pos.x + x_max, y = pos.y + y_max, z = pos.z + z_max}

    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- Loop through all coordinates
    for x = x_min, x_max do
        for y = y_min, y_max do
            for z = z_min, z_max do
                local vi = area:index(pos.x + x, pos.y + y, pos.z + z)
                data[vi] = cid
            end
        end
    end

    -- Commit changes once
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end



patterns["stair"] = function(pos, node_name)
    local step_count = math.max(math.abs(dx), math.abs(dz), 1)
    local y_step = dy or 1

    local x_dir, z_dir = 0, 0
    if math.abs(dx) > 1 then
        x_dir = dx > 0 and 1 or -1
    elseif math.abs(dz) > 1 then
        z_dir = dz > 0 and 1 or -1
    end

    -- Compute end position
    local end_pos = {
        x = pos.x + (step_count - 1) * x_dir,
        y = pos.y + (step_count - 1) * y_step,
        z = pos.z + (step_count - 1) * z_dir
    }

    -- Bounding box for voxelmanip
    local minp = {
        x = math.min(pos.x, end_pos.x),
        y = math.min(pos.y, end_pos.y),
        z = math.min(pos.z, end_pos.z),
    }
    local maxp = {
        x = math.max(pos.x, end_pos.x),
        y = math.max(pos.y, end_pos.y),
        z = math.max(pos.z, end_pos.z),
    }

    -- Voxelmanip setup
    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    local cid = minetest.get_content_id(node_name)

    -- Place the stair steps
    for i = 0, step_count - 1 do
        local x = pos.x + i * x_dir
        local y = pos.y + i * y_step
        local z = pos.z + i * z_dir
        local vi = area:index(x, y, z)
        data[vi] = cid
    end

    -- Commit changes
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end


patterns.tree = function(pos, node_name)
    -- Fixed tree height (20) unless overridden by /setdim dy
    local height = (dy and dy > 0) and dy or 20
    local cid = minetest.get_content_id(node_name)

    local nodes_to_place = {}

    for i = 0, height - 1 do
        local wx, wy, wz = pos.x, pos.y + i, pos.z
        table.insert(nodes_to_place, {x=wx, y=wy, z=wz})

        -- Add branches starting from 60% of height upwards
        if i > math.floor(height * 0.6) and i % 2 == 0 then
            local offsets = {
                {x=1, y=0, z=0}, {x=-1, y=0, z=0},
                {x=0, y=0, z=1}, {x=0, y=0, z=-1},
            }
            for _, off in ipairs(offsets) do
                table.insert(nodes_to_place, {x=wx+off.x, y=wy, z=wz+off.z})
            end
        end
    end

    -- Bounding box
    local minp = {x=pos.x, y=pos.y, z=pos.z}
    local maxp = {x=pos.x, y=pos.y + height - 1, z=pos.z}
    for _, p in ipairs(nodes_to_place) do
        if p.x < minp.x then minp.x = p.x end
        if p.y < minp.y then minp.y = p.y end
        if p.z < minp.z then minp.z = p.z end
        if p.x > maxp.x then maxp.x = p.x end
        if p.y > maxp.y then maxp.y = p.y end
        if p.z > maxp.z then maxp.z = p.z end
    end

    -- Load voxelmanip
    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- Place all nodes
    for _, p in ipairs(nodes_to_place) do
        local vi = area:index(p.x, p.y, p.z)
        data[vi] = cid
    end

    -- Commit
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end


patterns["o"] = function(pos, node_name)
    -- Get content ID for the node once (fast)
    local cid = minetest.get_content_id(node_name)

    -- Figure out region bounds
    local x_start = -dx
    local x_end   = dx -->= 0 and dx - 1 or 0
    local y_start = -dy -->= 0 and 0 or dy + 1
    local y_end   = dy -->= 0 and dy - 1 or 0
    local z_start = -dz-- >= 0 and 0 or dz + 1
    local z_end   = dz -->= 0 and dz - 1 or 0

    local minp = {
        x = pos.x + math.min(x_start, x_end),
        y = pos.y + math.min(y_start, y_end),
        z = pos.z + math.min(z_start, z_end),
    }
    local maxp = {
        x = pos.x + math.max(x_start, x_end),
        y = pos.y + math.max(y_start, y_end),
        z = pos.z + math.max(z_start, z_end),
    }

    -- Load voxelmanip
    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- 3D loop
	y = 0
	for x = x_start, x_end do
		
		local wx, wy, wz = pos.x + x, pos.y + y, pos.z + z_start
        local vi = area:index(wx, wy, wz)
		data[vi] = cid
		
		local wx, wy, wz = pos.x + x, pos.y + y, pos.z + z_end
        local vi = area:index(wx, wy, wz)
		data[vi] = cid
				
	end
	
	for z= z_start, z_end do
		
		local wx, wy, wz = pos.x + x_start, pos.y + y, pos.z + z
        local vi = area:index(wx, wy, wz)
		data[vi] = cid
		
		local wx, wy, wz = pos.x + x_end, pos.y + y, pos.z + z
        local vi = area:index(wx, wy, wz)
		data[vi] = cid
				
	end    

    -- Write changes back to world
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end

 patterns["O"] = patterns["o"]

patterns.cuboid = function(pos, node_name)
    -- determine extents
    local x_start = dx >= 0 and 0 or dx + 1
    local x_end   = dx >= 0 and dx - 1 or 0
    local y_start = dy >= 0 and 0 or dy + 1
    local y_end   = dy >= 0 and dy - 1 or 0
    local z_start = dz >= 0 and 0 or dz + 1
    local z_end   = dz >= 0 and dz - 1 or 0

    -- world coordinates
    local minp = {
        x = pos.x + math.min(x_start, x_end),
        y = pos.y + math.min(y_start, y_end),
        z = pos.z + math.min(z_start, z_end),
    }
    local maxp = {
        x = pos.x + math.max(x_start, x_end),
        y = pos.y + math.max(y_start, y_end),
        z = pos.z + math.max(z_start, z_end),
    }

    -- voxelmanip setup
    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- fast content ID lookup
    local cid = minetest.get_content_id(node_name)

    -- fill cuboid
    for x = x_start, x_end, 1 do
        for y = y_start, y_end, 1 do
            for z = z_start, z_end, 1 do
                local vi = area:index(pos.x + x, pos.y + y, pos.z + z)
                data[vi] = cid
            end
        end
    end

    -- write changes back
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end

patterns.floodfillxz = function(pos, node_name)
    local max_size = 5000  -- safety limit to prevent infinite fill
	local player = minetest.get_player_by_name("singleplayer")

    -- voxelmanip setup: load a generous area around pos
    local radius = 100
    local minp = {x = pos.x - radius, y = pos.y, z = pos.z - radius}
    local maxp = {x = pos.x + radius, y = pos.y, z = pos.z + radius}
    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    local cid_fill = minetest.get_content_id(node_name)

    -- replaceable nodes
    local replaceable = {
        ["air"] = true,
        ["default:water_source"] = true,
        ["default:water_flowing"] = true,
        ["default:lava_source"] = true,
        ["default:lava_flowing"] = true,
    }

    local function can_replace(vi)
        local nid = data[vi]
        local nname = minetest.get_name_from_content_id(nid)
        return replaceable[nname] or false
    end

    local inv = player:get_inventory()

    -- queue for BFS flood fill
    local queue = {}
    local visited = {}
    local function enqueue(x, z)
        local key = x .. ":" .. z
        if not visited[key] then
            table.insert(queue, {x=x, z=z})
            visited[key] = true
        end
    end

    enqueue(pos.x, pos.z)
    local vi = area:index(pos.x, pos.y, pos.z)
	data[vi] = minetest.get_content_id("air")
	

    local filled = 0
    while #queue > 0 and filled < max_size do
        local cell = table.remove(queue, 1)
        local vi = area:index(cell.x, pos.y, cell.z)

        if can_replace(vi) and inv:contains_item("main", node_name) then
            data[vi] = cid_fill
            inv:remove_item("main", node_name)
            filled = filled + 1

            -- neighbors
            enqueue(cell.x+1, cell.z)
            enqueue(cell.x-1, cell.z)
            enqueue(cell.x, cell.z+1)
            enqueue(cell.x, cell.z-1)
        end
    end

    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()

    minetest.chat_send_player(player:get_player_name(),
        "Floodfill placed " .. filled .. " nodes.")
end


patterns.bridge = function(pos, node_name)
    -- max expansion length
    local max_len = 500

    -- voxelmanip extents (covering a square area around pos)
    local minp = {x = pos.x - max_len, y = pos.y, z = pos.z - max_len}
    local maxp = {x = pos.x + max_len, y = pos.y, z = pos.z + max_len}

    local vm = minetest.get_voxel_manip()
    local emin, emax = vm:read_from_map(minp, maxp)
    local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
    local data = vm:get_data()

    -- fast lookup
    local cid_bridge = minetest.get_content_id(node_name)

    -- allowed nodes to overwrite
    local replaceable = {
        ["air"] = true,
        ["default:water_source"] = true,
        ["default:water_flowing"] = true,
		["default:river_water_source"] = true,
		["default:river_water_flowing"] = true,
		
        ["default:lava_source"] = true,
        ["default:lava_flowing"] = true,
    }

    -- helper to check if node is replaceable
    local function can_replace(vi)
        local nid = data[vi]
        local nname = minetest.get_name_from_content_id(nid)
        return replaceable[nname] or false
    end

    -- expand in +X
    -- +X
do
    local lastpos = nil
    for dx = 1, max_len do
        local p = {x = pos.x + dx, y = pos.y, z = pos.z}
        local vi = area:indexp(p)
        if can_replace(vi) then
            data[vi] = cid_bridge
            lastpos = p
        else
            break
        end
    end
    if lastpos then
        minetest.set_node({x=lastpos.x, y=lastpos.y + 1, z=lastpos.z}, {name = node_name})
    end
end

-- -X
do
    local lastpos = nil
    for dx = -1, -max_len, -1 do
        local p = {x = pos.x + dx, y = pos.y, z = pos.z}
        local vi = area:indexp(p)
        if can_replace(vi) then
            data[vi] = cid_bridge
            lastpos = p
        else
            break
        end
    end
    if lastpos then
        minetest.set_node({x=lastpos.x, y=lastpos.y + 1, z=lastpos.z}, {name=node_name})
    end
end

-- +Z
do
    local lastpos = nil
    for dz = 1, max_len do
        local p = {x = pos.x, y = pos.y, z = pos.z + dz}
        local vi = area:indexp(p)
        if can_replace(vi) then
            data[vi] = cid_bridge
            lastpos = p
        else
            break
        end
    end
    if lastpos then
        minetest.set_node({x=lastpos.x, y=lastpos.y + 1, z=lastpos.z}, {name=node_name})
    end
end

-- -Z
do
    local lastpos = nil
    for dz = -1, -max_len, -1 do
        local p = {x = pos.x, y = pos.y, z = pos.z + dz}
        local vi = area:indexp(p)
        if can_replace(vi) then
            data[vi] = cid_bridge
            lastpos = p
        else
            break
        end
    end
    if lastpos then
        minetest.set_node({x=lastpos.x, y=lastpos.y + 1, z=lastpos.z}, {name=node_name})
    end
end


    -- write back changes
    vm:set_data(data)
    vm:write_to_map(true)
    vm:update_map()
end



minetest.register_on_mods_loaded(function()
    local patched = 0

    for name, def in pairs(minetest.registered_nodes) do
        if def and not def.after_place_node and name ~= "air" and name ~= "ignore" then
            minetest.override_item(name, {
                after_place_node = function(pos, placer, itemstack, pointed_thing)
                    
					
					if offset.x ~= 0 or offset.y ~= 0 or offset.z ~= 0 then
						minetest.set_node(pos, { name = "air" })
						-- at least one offset is non-zero
					end

					
					local final_pos = {
                        x = pos.x + offset.x,
                        y = pos.y + offset.y,
                        z = pos.z + offset.z,
                    }

                    local func = patterns[active_pattern]
                    if func then
                        func(final_pos, name)
                    end
                end
            })
            patched = patched + 1
        end
    end

    minetest.log("action", ("[%s] attached after_place_node to %d nodes."):format(MODNAME, patched))
end)




-- chat command: set pattern
minetest.register_chatcommand("setpattern", {
    params = "<name>",
    description = "Set the active pattern",
    privs = { server = true },
    func = function(name, param)
        if patterns[param] then
            active_pattern = param
            return true, "Pattern set to " .. param
        else
            return false, "Unknown pattern!"
        end
    end,
})




-- chat command: set dimensions
minetest.register_chatcommand("setdim", {
    params = "<dx> <dy> <dz>",
    description = "Set dimensions for offset pattern",
    privs = { server = true },
    func = function(name, param)
        local x, y, z = param:match("^(%-?%d+)%s+(%-?%d+)%s+(%-?%d+)$")
        if not x then
            return false, "Usage: /setdim <dx> <dy> <dz>"
        end
        dx, dy, dz = tonumber(x), tonumber(y), tonumber(z)
        return true, ("Dimensions set to dx=%d dy=%d dz=%d"):format(dx, dy, dz)
    end,
})




minetest.register_chatcommand("setoffset", {
    params = "<x> <y> <z>",
    description = "Set pattern offset",
    func = function(name, param)
        local x, y, z = param:match("(%-?%d+)%s+(%-?%d+)%s+(%-?%d+)")
        x, y, z = tonumber(x), tonumber(y), tonumber(z)
        if x and y and z then
            offset.x, offset.y, offset.z = x, y, z
            return true, ("Offset set to x=%d y=%d z=%d"):format(x, y, z)
        else
            return false, "Invalid parameters"
        end
    end
})





minetest.register_chatcommand("setchar", {
    params = "<letter>",
    description = "Set the active letter pattern",
    privs = { server = true },
    func = function(name, param)
        -- Convert to uppercase to match keys in letters table
        local letter = param:upper()

        if letters[letter] then
            char = letter
            return true, "Pattern set to " .. letter
        else
            return false, "Unknown letter! Available letters: A-Z0-9"
        end
    end,
})


