-- yl_scheduler.get_task(UUID)
--
local function get_task(UUID)
    local success, tasks = yl_scheduler.list_all_tasks()
    if (success == false) then return nil end
    for key, task in ipairs(tasks) do
        if task.id == UUID then return true, task end
    end
    return nil
end

function yl_scheduler.get_task(UUID) return get_task(UUID) end


-- yl_scheduler.set_task(at, func, params, owner, notes)
--
local function set_task(at, func, params, owner, notes)

    -- Defense
    -- Those values come from files and chatcommands

    local validate_success, validate_message =
        yl_scheduler.validate(at, func, params, owner, notes)
    if (validate_success == false) then return false, validate_message end

    local task = {}

    -- UUID
    local get_uuid_success, UUID = yl_scheduler.create_uuid()
    if (get_uuid_success == false) then return false, UUID end

    task.id = UUID

    -- at
    -- Try to find what timeformat we are
    -- Caclulate utc time from this given time format
    task.at = at

    -- It's fresh? Then done = -1
    task.done = -1

    -- func
    task.func = func

    -- params, optional
    task.params = params or {}

    -- owner
    task.owner = owner

    -- notes, optional
    task.notes = notes or ""

    -- save to file
    local write_file_success = yl_scheduler.save_json(UUID, task)

    if (write_file_success == false) then return false, "Cannot write file" end

    -- store in table
    table.insert(yl_scheduler.tasks, task)

    return true, UUID
end

function yl_scheduler.set_task(at, func, params, owner, notes)
    return set_task(at, func, params, owner, notes)
end


-- yl_scheduler.list_all_tasks()
--
local function list_all_tasks()
    if next(yl_scheduler.tasks) then
        return true, yl_scheduler.tasks
    else
        return false, {}
    end
end

function yl_scheduler.list_all_tasks() return list_all_tasks() end


-- function find_ids(UUID)
--
local function find_ids(UUID)
    local _, tasks = yl_scheduler.list_all_tasks()
    local result = {}
    for i, task in ipairs(tasks) do
        if (string.find(task.id, UUID) ~= nil) then
            table.insert(result, task.id)
        end
    end
    if next(result) then
        return true, result
    else
        return false, result
    end
end

function yl_scheduler.find_ids(UUID) return find_ids(UUID) end


-- yl_scheduler.find_task(UUID)
--
local function find_task(UUID)
    local _, tasks = yl_scheduler.list_all_tasks()
    local result = {}
    for i, task in ipairs(tasks) do
        if (string.find(task.id, UUID) ~= nil) then
            table.insert(result, task)
        end
    end
    if next(result) then
        return true, result
    else
        return false, result
    end
end

function yl_scheduler.find_task(UUID) return find_task(UUID) end


-- yl_scheduler.remove_task(UUID)
--
local function remove_task(UUID)
    local _, tasks = yl_scheduler.list_all_tasks()
    for i, task in ipairs(tasks) do
        if task.id == UUID then
            table.remove(tasks, i)
            local success, errormsg = yl_scheduler.remove_file(UUID)
            if (success == false) then
                minetest.log("warning",
                             "[MOD] yl_scheduler : Cannot remove UUID " .. UUID ..
                                 ": File not found. Errormsg: " .. errormsg)
            end
            return true, task
        end
    end
    return false, "UUID not found"
end

function yl_scheduler.remove_task(UUID) return remove_task(UUID) end


-- yl_scheduler.clean_past_tasks()
--
local function clean_past_tasks()
    local _, tasks = yl_scheduler.list_all_tasks()
    local amount_deleted = 0
    local current_utc_time = os.time() -- utc

    for i = #tasks, 1, -1 do
        if ((tasks[i].at ~= nil) and (tasks[i].at < current_utc_time)) then
            local UUID = tasks[i].id
            table.remove(tasks, i)
            local success, errormsg = yl_scheduler.remove_file(UUID)
            if (success == false) then
                minetest.log("warning",
                             "[MOD] yl_scheduler : Cannot remove UUID " .. UUID ..
                                 ": File not found. Errormsg: " .. errormsg)
            end
            amount_deleted = amount_deleted + 1
        end
    end

    yl_scheduler.tasks = tasks

    local amount_remaining = #tasks
    if amount_deleted > 0 then
        return true, amount_deleted, amount_remaining
    else
        return false, "no tasks removed"
    end
end

function yl_scheduler.clean_past_tasks() return clean_past_tasks() end


-- yl_scheduler.clean_executed_tasks()
--
local function clean_executed_tasks()
    local _, tasks = yl_scheduler.list_all_tasks()
    local amount_deleted = 0

    for i = #tasks, 1, -1 do
        if ((tasks[i].done ~= nil) and (tasks[i].done > 0)) then
            local UUID = tasks[i].id
            table.remove(tasks, i)
            local success, errormsg = yl_scheduler.remove_file(UUID)
            if (success == false) then
                minetest.log("warning",
                             "[MOD] yl_scheduler : Cannot remove UUID " .. UUID ..
                                 ": File not found. Errormsg: " .. errormsg)
            end
            amount_deleted = amount_deleted + 1
        end
    end

    yl_scheduler.tasks = tasks

    local amount_remaining = #tasks
    if amount_deleted > 0 then
        return true, amount_deleted, amount_remaining
    else
        return false, "no tasks removed"
    end
end

function yl_scheduler.clean_executed_tasks() return clean_executed_tasks() end

-- yl_scheduler.execute_task(UUID)
--
local function execute_task(UUID)
    local _, task = yl_scheduler.get_task(UUID)
    local func = _G

    local f = string.split(task.func, ".")
    for _, part in ipairs(f) do
        if (part == nil) then break end
        func = func[part]
    end

    -- Execute the function
    if func and (type(func) == "function") then
        local success, message = pcall(func, unpack(task.params or {}))
        if not success then return false, "Pcall " .. message end
    else
        -- remove from list, so that it won't get exectured over and over
        return false, "Function not found"
    end

    -- Change the done value in the file
    local current_utc_time = os.time()
    task.done = current_utc_time
    yl_scheduler.save_json(UUID, task)

    -- Change the done value in the table
    for _, t_task in ipairs(yl_scheduler.tasks) do
        if t_task.id == UUID then t_task.done = current_utc_time end
    end

    return true, task
end

function yl_scheduler.execute_task(UUID) return execute_task(UUID) end
