--[[
    In case of changing data format that breaks backward compatibility, add the migration code to migrations.lua
--]]

superquest.data_processing = {}

superquest.data_processing.remove_flag_from_network = function(owner, network, pos)
    if owner == "" or network == "" then
        return
    end

    local data = superquest.storage.get_network_data(owner, network)
    if (data ~= nil and data.flags ~= nil) then
        data.flags[superquest.storage.coords_to_key(pos)] = nil
        if next(data.flags, nil) == nil then
            data.flags = nil
        end
        superquest.storage.set_network_data(owner, network, data)
    end
end

superquest.data_processing.add_flag_to_network = function(owner, network, pos)
    if owner == "" or network == "" then
        return
    end

    local data = superquest.storage.get_network_data(owner, network)
    if data == nil then
        data = {}
    end
    if data.flags == nil then
        data.flags = {}
    end
    data.flags[superquest.storage.coords_to_key(pos)] = { players_reached = {} }
    superquest.storage.set_network_data(owner, network, data)
end

superquest.data_processing.add_player_reached_flag = function(owner, network, pos, player_name)
    local data = superquest.storage.get_network_data(owner, network)
    if data == nil or data.flags == nil or data.flags[superquest.storage.coords_to_key(pos)] == nil then
        -- Could be optimized but since the situation is unlikely, it's ok
        superquest.data_processing.add_flag_to_network(owner, network, pos)

        data = superquest.storage.get_network_data(owner, network)
    elseif data.flags[superquest.storage.coords_to_key(pos)].players_reached[player_name] then
        return nil
    end

    data.flags[superquest.storage.coords_to_key(pos)].players_reached[player_name] = 1
    superquest.storage.set_network_data(owner, network, data)

    local total_flags = 0
    local reached_flags = 0
    for k, v in pairs(data.flags) do
        if v.players_reached[player_name] then
            reached_flags = reached_flags + 1
        end
        total_flags = total_flags + 1
    end

    return { total = total_flags, reached = reached_flags }
end

superquest.data_processing.remove_reward_box_from_network = function(owner, network, pos)
    if owner == "" or network == "" then
        return
    end

    local data = superquest.storage.get_network_data(owner, network)
    if (data ~= nil and data.reward_boxes ~= nil and data.reward_boxes[superquest.storage.coords_to_key(pos)] ~= nil) then
        data.reward_boxes[superquest.storage.coords_to_key(pos)] = nil
        data.reward_boxes_count = data.reward_boxes_count - 1
        if data.reward_boxes_count == 0 then
            data.reward_boxes = nil
            data.reward_boxes_count = nil
        end
        superquest.storage.set_network_data(owner, network, data)
    end
end

superquest.data_processing.add_reward_box_to_network = function(owner, network, pos)
    if owner == "" or network == "" then
        return
    end

    local data = superquest.storage.get_network_data(owner, network)
    if data == nil then
        data = {}
    end
    if data.reward_boxes == nil or data.reward_boxes_count == nil then
        data.reward_boxes = {}
        data.reward_boxes_count = 0
    end
    data.reward_boxes[superquest.storage.coords_to_key(pos)] = {
        completions = {}
    }
    data.reward_boxes_count = data.reward_boxes_count + 1
    superquest.storage.set_network_data(owner, network, data)
end

superquest.data_processing.get_reward_box_stats_for_player = function(owner, network, pos, player_name)
    if owner == "" or network == "" then
        return { total_flags = 0, reached_flags = 0, completions = 0 }
    end

    local key = superquest.storage.coords_to_key(pos)

    local data = superquest.storage.get_network_data(owner, network)
    if data == nil or data.reward_boxes == nil or data.reward_boxes[key] == nil then
        superquest.data_processing.add_reward_box_to_network(owner, network, pos)

        data = superquest.storage.get_network_data(owner, network)
    end

    local total_flags = 0;
    local reached_flags = 0;
    if data.flags then
        for k, v in pairs(data.flags) do
            if v.players_reached[player_name] then
                reached_flags = reached_flags + 1
            end
            total_flags = total_flags + 1
        end
    end

    local completions = 0
    if data.reward_boxes[key].completions[player_name] ~= nil then
        completions = data.reward_boxes[key].completions[player_name]
    end

    return {
        total_flags = total_flags,
        reached_flags = reached_flags,
        completions = completions
    }
end

superquest.data_processing.check_for_completion = function(owner, network, pos, player_name, max_completions)
    local key = superquest.storage.coords_to_key(pos)

    local data = superquest.storage.get_network_data(owner, network)
    if data == nil or data.reward_boxes == nil or data.reward_boxes[key] == nil then
        superquest.data_processing.add_reward_box_to_network(owner, network, pos)

        data = superquest.storage.get_network_data(owner, network)
    end

    local completions = 0
    if data.reward_boxes[key].completions[player_name] ~= nil then
        completions = data.reward_boxes[key].completions[player_name]
    end

    if max_completions ~= 0 and completions >= max_completions then
        return nil
    end

    if data.flags then
        for _, v in pairs(data.flags) do
            if not v.players_reached[player_name] then
                return nil
            end
        end
    end

    return completions
end

superquest.data_processing.score_completion = function(owner, network, pos, player_name)
    local key = superquest.storage.coords_to_key(pos)

    local data = superquest.storage.get_network_data(owner, network)
    if data == nil or data.reward_boxes == nil or data.reward_boxes[key] == nil then
        superquest.data_processing.add_reward_box_to_network(owner, network, pos)
        data = superquest.storage.get_network_data(owner, network)
    end

    if data.flags then
        for _, v in pairs(data.flags) do
            v.players_reached[player_name] = nil
        end
    end

    local completions = 0
    if data.reward_boxes[key].completions[player_name] ~= nil then
        completions = data.reward_boxes[key].completions[player_name]
    end
    data.reward_boxes[key].completions[player_name] = completions + 1

    superquest.storage.set_network_data(owner, network, data)
end

superquest.data_processing.get_all_completions = function(owner, network, pos)
    local key = superquest.storage.coords_to_key(pos)

    local data = superquest.storage.get_network_data(owner, network)
    if data == nil or data.reward_boxes == nil or data.reward_boxes[key] == nil then
        superquest.data_processing.add_reward_box_to_network(owner, network, pos)
        data = superquest.storage.get_network_data(owner, network)
    end

    return data.reward_boxes[key].completions
end

superquest.data_processing.reset_all_completions = function(owner, network, pos)
    local key = superquest.storage.coords_to_key(pos)

    local data = superquest.storage.get_network_data(owner, network)
    if data == nil or data.reward_boxes == nil or data.reward_boxes[key] == nil then
        superquest.data_processing.add_reward_box_to_network(owner, network, pos)
        data = superquest.storage.get_network_data(owner, network)
    end

    data.reward_boxes[key].completions = {}

    superquest.storage.set_network_data(owner, network, data)
end

superquest.data_processing.has_a_reward_box = function(owner, network)
    if owner == "" or network == "" then
        return false
    end

    local data = superquest.storage.get_network_data(owner, network)

    return data ~= nil and data.reward_boxes ~= nil
end

local S = superquest.S

superquest.data_processing.get_network_description = function(owner, network)
    local data = superquest.storage.get_network_data(owner, network)
    if (data == nil) then
        return nil
    end

    local res = S("Network") .. " " .. network .. ":\n"

    if data.flags then
        for k, v in pairs(data.flags) do
            res = res .. "\t" .. S("flag at") .. " (" .. k .. ")\n"
        end
    end

    if data.reward_boxes then
        for k, v in pairs(data.reward_boxes) do
            res = res .. "\t" .. S("reward box at") .. " (" .. k .. ")\n"
        end
    end

    return res
end