local S = superquest.S

local form_name_owner = minetest.get_current_modname()..":reward_box_owner"
local form_name_user = minetest.get_current_modname()..":reward_box_user"
local form_name_completions = minetest.get_current_modname()..":reward_box_completions"
local node_positions = {}

local function pos_to_formspec_inv_loc(pos)
    return "nodemeta:"..pos.x..","..pos.y..","..pos.z
end

local function get_formspec_for_owner(owner, network, pos, max_completions, available_rewards, total_flags, can_edit_owner)
    local formspec_data =
        "formspec_version[4]" ..
        "size[22,13.8]"
    if can_edit_owner then
        formspec_data = formspec_data ..
            "field[0.5,0.5;9.75,0.8;owner;"..S("Owner")..";"..minetest.formspec_escape(owner).."]"..
            "container[0.5,0.3]"
    else
        formspec_data = formspec_data ..
            "container[0.5,0]"..
            "label[0,0.8;"..S("Owner")..": "..minetest.formspec_escape(owner).."]"
    end
    formspec_data = formspec_data ..
        "label[11,0.8;"..S("Flags in network")..": "..minetest.formspec_escape(total_flags).."]" ..
        "field[0,1.5;9.75,0.8;network;"..S("Network")..";"..minetest.formspec_escape(network).."]" ..
        "field[11,1.5;9.75,0.8;max_completions;"..S("Max completions per user (0 @=@= infinite)")..";"..minetest.formspec_escape(max_completions).."]"..
        "label[0,2.8;"..S("Reward for first completion")..":]"..
        "list["..pos_to_formspec_inv_loc(pos)..";first_compl_rewards;0,3.2;8,1]"..
        "label[0,4.5;"..S("Available: @1 (reopen the form to see the updated value)", minetest.formspec_escape(available_rewards.first_compl_rewards)).."]"..
        "label[11,2.8;"..S("Reward for further completions")..":]"..
        "list["..pos_to_formspec_inv_loc(pos)..";further_compl_rewards;11,3.2;8,1]"..
        "label[11,4.5;"..S("Available: @1 (reopen the form to see the updated value)", minetest.formspec_escape(available_rewards.further_compl_rewards)).."]"..
        "label[0,5;"..S("Reward box inventory")..":]"..
        "list["..pos_to_formspec_inv_loc(pos)..";reward_box_storage;0,5.4;8,4;]"..
        "label[11,5;"..S("Your inventory")..":]"..
        "list[current_player;main;11,5.4;8,4;]"..
        "button_exit[0,10.5;20.75,0.8;save;"..S("Save").."]"..
        "button[0,11.5;20.75,0.8;completions;"..S("Show completions").."]"..
        "button[0,12.5;20.75,0.8;user_mode;"..S("User mode").."]"..
        "container_end[]"

    return formspec_data
end

local function get_formspec_for_user(owner, network, pos, owner_mode_avail, max_completions, available_rewards, total_flags, completions, reached_flags)
    local completions_str = tostring(completions)
    if max_completions ~= 0 then
        completions_str = completions_str.." / "..max_completions
    end

    local formspec_data =
        "formspec_version[4]" ..
        "size[10.75,10.3]" ..
        "label[0.5,0.5;"..S("Owner")..": "..minetest.formspec_escape(owner).."]" ..
        "label[0.5,1;"..S("Network")..": "..minetest.formspec_escape(network).."]" ..
        "label[0.5,1.5;"..S("Your completions")..": "..minetest.formspec_escape(completions_str).."]"..
        "label[0.5,2;"..S("Flags reached")..": "..minetest.formspec_escape(reached_flags.." / "..total_flags).."]"..
        "label[0.5,2.8;"..S("Reward for first completion")..":]"..
        "list["..pos_to_formspec_inv_loc(pos)..";first_compl_rewards;0.5,3.1;8,1]"..
        "label[0.5,4.4;"..S("Available")..": "..minetest.formspec_escape(available_rewards.first_compl_rewards).."]"..
        "label[0.5,5.1;"..S("Reward for further completions")..":]"..
        "list["..pos_to_formspec_inv_loc(pos)..";further_compl_rewards;0.5,5.4;8,1]"..
        "label[0.5,6.7;"..S("Available")..": "..minetest.formspec_escape(available_rewards.further_compl_rewards).."]"..
        "button_exit[0.5,7;9.75,0.8;close;"..S("Close").."]"..
        "button[0.5,8;9.75,0.8;completions;"..S("Show completions").."]"
    if owner_mode_avail then
        formspec_data = formspec_data.."button[0.5,9;9.75,0.8;owner_mode;"..S("Owner mode").."]"
    end

    return formspec_data
end

local function get_completions_formspec(completions, is_owner)
    local formspec_data =
        "formspec_version[4]" ..
        "size[10,10]" ..
        "tablecolumns[text;text]" ..
        "table[0.5,0.5;9,7.3;compl_list;"
    for k, v in pairs(completions) do
        formspec_data = formspec_data .. minetest.formspec_escape(k) .. "," .. minetest.formspec_escape(v) .. ","
    end
    formspec_data = formspec_data .. ";1]"..
        "button_exit[0.5,8;9,0.8;close;"..S("Close").."]"
    if is_owner then
        formspec_data = formspec_data .. "button_exit[0.5,9;9,0.8;reset_completions;"..S("Reset Completions").."]"
    end

    return formspec_data
end

local function calc_avail_rewards_for_inventory(rewards_inv, reward_box_inventory)
    local available_rewards
    for k, v in pairs(rewards_inv) do
        if reward_box_inventory[k] ~= nil then
            local avail_cur_item_times = reward_box_inventory[k] / v
            if available_rewards == nil or avail_cur_item_times < available_rewards then
                available_rewards = avail_cur_item_times
            end
        else
            available_rewards = 0
            break
        end
    end

    if available_rewards == nil then
        available_rewards = 0
    end

    return math.floor(available_rewards)
end

local function calc_items_in_inv_list(inv_list)
    local items_count = {}
    for _, stack_v in pairs(inv_list) do
        if stack_v:get_count() ~= 0 then
            if items_count[stack_v:get_name()] == nil then
                items_count[stack_v:get_name()] = 0
            end
            items_count[stack_v:get_name()] = items_count[stack_v:get_name()] + stack_v:get_count()
        end
    end

    return items_count
end

local function get_number_of_avail_rewards(inv_lists)
    local items_count_per_inv = {}
    for list_k, list_v in pairs(inv_lists) do
        items_count_per_inv[list_k] = calc_items_in_inv_list(list_v)
    end

    return {
        first_compl_rewards = calc_avail_rewards_for_inventory(items_count_per_inv["first_compl_rewards"], items_count_per_inv["reward_box_storage"]),
        further_compl_rewards = calc_avail_rewards_for_inventory(items_count_per_inv["further_compl_rewards"], items_count_per_inv["reward_box_storage"]),
    }
end

local function show_formspec(owner, network, pos, is_user_mode, player_name, max_completions, inv_lists)
    local stats = superquest.data_processing.get_reward_box_stats_for_player(owner, network, pos, player_name)

    local available_rewards = get_number_of_avail_rewards(inv_lists)

    if (owner == player_name or superquest.privileges.can_edit_all(player_name)) and not is_user_mode then
        minetest.show_formspec(player_name, form_name_owner,
            get_formspec_for_owner(owner, network, pos, max_completions, available_rewards, stats.total_flags, superquest.privileges.can_edit_all(player_name)))
    else
        minetest.show_formspec(player_name, form_name_user,
            get_formspec_for_user(owner, network, pos, owner == player_name or superquest.privileges.can_edit_all(player_name), max_completions,
                available_rewards, stats.total_flags, stats.completions, stats.reached_flags))
    end
end

local function is_enough_place_for_reward_in_user_inv(reward_inv_list, user_inv_list)
    local items_count = calc_items_in_inv_list(reward_inv_list)

    local empty_cells_in_user_inv = 0
    local avail_for_items = {}

    for _, stack in pairs(user_inv_list) do
        if stack:get_count() == 0 then
            empty_cells_in_user_inv = empty_cells_in_user_inv + 1
        elseif items_count[stack:get_name()] ~= nil and stack:get_count() < stack:get_stack_max() then
            if avail_for_items[stack:get_name()] == nil then
                avail_for_items[stack:get_name()] = 0
            end
            avail_for_items[stack:get_name()] = avail_for_items[stack:get_name()] + stack:get_stack_max() - stack:get_count()
        end
    end

    for k, v in pairs(items_count) do
        local cur_items_remaining = v
        if avail_for_items[k] ~= nil then
            cur_items_remaining = cur_items_remaining - math.min(cur_items_remaining, avail_for_items[k])
        end

        local max_stack_size = ItemStack(k):get_stack_max()
        if empty_cells_in_user_inv < cur_items_remaining / max_stack_size then
            return false
        end

        empty_cells_in_user_inv = empty_cells_in_user_inv - math.ceil(cur_items_remaining / max_stack_size)

        -- We don't need to update avail_for_items for the current item here since we won't need it anymore
    end

    return true
end

local can_dig = function(pos, player)
        local meta = minetest.get_meta(pos)
        local inv = meta:get_inventory()

        local res = inv:is_empty("reward_box_storage")
        if (not res and player and player:is_player()) then
            minetest.chat_send_player(player:get_player_name(), S("The reward inventory is not empty!"))
        end

        return res

        -- We don't need to check if the player is owner because reward boxes should be placed in protected areas anyway
end

local after_place_node = function(pos, placer, itemstack)
    local meta = minetest.get_meta(pos)
    if placer and placer:is_player() then
        meta:set_string("owner", placer:get_player_name())
    end

    local inv = meta:get_inventory()

    inv:set_size("first_compl_rewards", 8)
    inv:set_size("further_compl_rewards", 8)
    inv:set_size("reward_box_storage", 32)
end

local on_rightclick = function(pos, node, clicker, itemstack, pointed_thing)
    if not (clicker and clicker:is_player()) then
        return
    end

    local meta = minetest.get_meta(pos)
    local owner = meta:get_string("owner")
    local network = meta:get_string("network")

    local clicker_name = clicker:get_player_name()
    node_positions[clicker_name] = pos

    show_formspec(owner, network, pos, superquest.data_processing.is_network_in_user_mode(owner, network),
    clicker_name, meta:get_int("max_completions"), meta:get_inventory():get_lists())
end

local on_destruct = function(pos)
    local meta = minetest.get_meta(pos)

    superquest.data_processing.remove_reward_box_from_network(meta:get_string("owner"), meta:get_string("network"), pos)
end


local facedir_to_front_dir_table = {
    [0] = 2, 4, 1, 3, 0, 4, 5, 3, 5, 4, 0, 3, 2, 0, 1, 5, 2, 5, 1, 0, 2, 3, 1, 4
}

local front_dir_to_particles_speed = {
    [0] = vector.new(0, 1, 0),
    [1] = vector.new(0, 0, 1),
    [2] = vector.new(0, 0, -1),
    [3] = vector.new(1, 0, 0),
    [4] = vector.new(-1, 0, 0),
    [5] = vector.new(0, -1, 0),
}

local on_punch = function(pos, node, puncher)
    if not (puncher and puncher:is_player()) then
        return
    end

    local meta = minetest.get_meta(pos)

    local player_name = puncher:get_player_name()

    local owner = meta:get_string("owner")
    local network = meta:get_string("network")

    if owner == "" or network == "" then
        minetest.chat_send_player(player_name, S("The Reward Box is not configured!"))
        return
    end

    if (owner == player_name or superquest.privileges.can_edit_all(player_name))
            and not superquest.data_processing.is_network_in_user_mode(owner, network) then
        return
    end

    local completions = superquest.data_processing.check_for_completion(owner, network, pos, player_name, meta:get_int("max_completions"))
    if not completions then
        return
    end

    local node_inv = meta:get_inventory()
    local player_inv = puncher:get_inventory()

    local node_inv_lists = node_inv:get_lists();

    local avail_rewards_total = get_number_of_avail_rewards(node_inv_lists)

    local rewards_list;
    local avail_rewards;

    if completions == 0 then
        rewards_list = node_inv:get_list("first_compl_rewards")
        avail_rewards = avail_rewards_total.first_compl_rewards
    else
        rewards_list = node_inv:get_list("further_compl_rewards")
        avail_rewards = avail_rewards_total.further_compl_rewards
    end

    if avail_rewards == 0 then
        minetest.chat_send_player(player_name, S("Not enought items for reward!"))
        return
    end

    local is_enough_place_in_inv = is_enough_place_for_reward_in_user_inv(rewards_list, player_inv:get_list("main"))
    if not is_enough_place_in_inv then
        minetest.chat_send_player(player_name, S("Not enough space in your inventory!"))
        return
    end

    superquest.data_processing.score_completion(owner, network, pos, player_name)

    for k, v in pairs(rewards_list) do
        player_inv:add_item("main", v)
        node_inv:remove_item("reward_box_storage", v)
    end

    local particles_speed = front_dir_to_particles_speed[facedir_to_front_dir_table[node.param2]]
    local pos_base = vector.subtract(vector.new(1, 1, 1), vector.apply(particles_speed, math.abs))
    local minpos = vector.multiply(pos_base, -0.5)
    local maxpos = vector.multiply(pos_base, 0.5)
    if node.name == "superquest:reward_box" then
        minpos = vector.add(minpos, vector.divide(particles_speed, 2))
        maxpos = vector.add(maxpos, vector.divide(particles_speed, 2))
    end

    minetest.add_particlespawner({
        amount = 50,
        time = 0.01,
        minpos = vector.add(pos, minpos),
        maxpos = vector.add(pos, maxpos),
        minvel = vector.multiply(particles_speed, 0.5),
        maxvel = vector.multiply(particles_speed, 2),
        minacc = vector.new(0, -9.8, 0),
        maxacc = vector.new(0, -9.8, 0),
        minexptime = 2,
        maxexptime = 2,
        minsize = 0.4,
        maxsize = 0.4,
        texture = "particle_red.png"
    })

    minetest.add_particlespawner({
        amount = 50,
        time = 0.01,
        minpos = vector.add(pos, minpos),
        maxpos = vector.add(pos, maxpos),
        minvel = vector.multiply(particles_speed, 0.5),
        maxvel = vector.multiply(particles_speed, 2),
        minacc = vector.new(0, -9.8, 0),
        maxacc = vector.new(0, -9.8, 0),
        minexptime = 2,
        maxexptime = 2,
        minsize = 0.4,
        maxsize = 0.4,
        texture = "particle_green.png"
    })

    minetest.add_particlespawner({
        amount = 50,
        time = 0.01,
        minpos = vector.add(pos, minpos),
        maxpos = vector.add(pos, maxpos),
        minvel = vector.multiply(particles_speed, 0.5),
        maxvel = vector.multiply(particles_speed, 2),
        minacc = vector.new(0, -9.8, 0),
        maxacc = vector.new(0, -9.8, 0),
        minexptime = 2,
        maxexptime = 2,
        minsize = 0.4,
        maxsize = 0.4,
        texture = "particle_blue.png"
    })

    superquest.hud.show_popup(puncher, S("Reward received!"))
end

local allow_metadata_inventory_put = function(pos, listname, item, stack, player)
    if not (player and player:is_player()) then
        return 0
    end

    local meta = minetest.get_meta(pos)
    local owner = meta:get_string("owner")
    local network = meta:get_string("network")

    local player_name = player:get_player_name()

    if (owner == player_name or superquest.privileges.can_edit_all(player_name))
            and not superquest.data_processing.is_network_in_user_mode(owner, network) then
        if listname == "reward_box_storage" then
            return stack:get_count()
        else
            local inv = meta:get_inventory()
            local cur_stack = inv:get_stack(listname, item)
            if cur_stack:get_name() == stack:get_name() then
                local final_count = cur_stack:get_count() + stack:get_count()
                inv:set_stack(listname, item, ItemStack({
                    name = stack:get_name(),
                    count = final_count
                }))
            else
                inv:set_stack(listname, item, stack)
            end
        end
    end

    return 0
end

local allow_metadata_inventory_take = function(pos, listname, item, stack, player)
    if not (player and player:is_player()) then
        return 0
    end

    local meta = minetest.get_meta(pos)
    local owner = meta:get_string("owner")
    local network = meta:get_string("network")

    local player_name = player:get_player_name()

    if (owner == player_name or superquest.privileges.can_edit_all(player_name))
            and not superquest.data_processing.is_network_in_user_mode(owner, network) then
        if listname == "reward_box_storage" then
            return stack:get_count()
        else
            local inv = meta:get_inventory()
            local cur_stack = inv:get_stack(listname, item)
            if cur_stack:get_count() > stack:get_count() then
                inv:set_stack(listname, item, ItemStack({
                    name = cur_stack:get_name(),
                    count = cur_stack:get_count() - stack:get_count()
                }))
            else
                inv:set_stack(listname, item, ItemStack())
            end
        end
    end

    return 0
end

local allow_metadata_inventory_move = function(pos, from_list, from_index, to_list, to_index, count, player)
    if not (player and player:is_player()) then
        return 0
    end

    local meta = minetest.get_meta(pos)
    local owner = meta:get_string("owner")
    local network = meta:get_string("network")

    local player_name = player:get_player_name()

    if (owner == player_name or superquest.privileges.can_edit_all(player_name))
            and not superquest.data_processing.is_network_in_user_mode(owner, network) then
        local inv = meta:get_inventory()
        local from_stack = inv:get_stack(from_list, from_index)
        local to_stack = inv:get_stack(to_list, to_index)

        if from_list == "reward_box_storage" and to_list ~= "reward_box_storage" then
            if from_stack:get_name() == to_stack:get_name() then
                local final_count = count + to_stack:get_count();
                if final_count > to_stack:get_stack_max() then
                    final_count = to_stack:get_stack_max()
                end
                inv:set_stack(to_list, to_index, ItemStack({
                    name = from_stack:get_name(),
                    count = final_count
                }))
            else
                inv:set_stack(to_list, to_index, ItemStack({
                    name = from_stack:get_name(),
                    count = count
                }))
            end
        elseif from_list ~= "reward_box_storage" and to_list == "reward_box_storage" then
            if from_stack:get_count() > count then
                inv:set_stack(from_list, from_index, ItemStack({
                    name = from_stack:get_name(),
                    count = from_stack:get_count() - count
                }))
            else
                inv:set_stack(from_list, from_index, ItemStack())
            end
        else
            return count
        end
    end

    return 0
end

minetest.register_node("superquest:reward_box", {
    description = S("Reward Box"),
    tiles = {
        "reward_box_sides.png",
        "reward_box_sides.png",
        "reward_box_sides.png",
        "reward_box_sides.png",
        "reward_box_sides.png",
        "reward_box_front.png"
    },
    groups = { cracky = 3 },
    paramtype2 = "facedir",

    can_dig = can_dig,
    after_place_node = after_place_node,
    on_rightclick = on_rightclick,
    on_destruct = on_destruct,
    on_punch = on_punch,
    allow_metadata_inventory_put = allow_metadata_inventory_put,
    allow_metadata_inventory_take = allow_metadata_inventory_take,
    allow_metadata_inventory_move = allow_metadata_inventory_move,
})

minetest.register_node("superquest:reward_box_flat", {
    description = S("Flat reward box"),
    tiles = {
        "reward_box_sides_flat_a.png",
        "reward_box_sides_flat_a.png",
        "reward_box_sides_flat_b.png",
        "reward_box_sides_flat_b.png",
        "reward_box_sides.png",
        "reward_box_front.png"
    },
    groups = { cracky = 3 },
    paramtype2 = "facedir",
    drawtype = "nodebox",
    node_box = { type = "fixed", fixed = { -0.5, -0.5, -0.0, 0.5, 0.5, 0.5 } },

    can_dig = can_dig,
    after_place_node = after_place_node,
    on_rightclick = on_rightclick,
    on_destruct = on_destruct,
    on_punch = on_punch,
    allow_metadata_inventory_put = allow_metadata_inventory_put,
    allow_metadata_inventory_take = allow_metadata_inventory_take,
    allow_metadata_inventory_move = allow_metadata_inventory_move,
})

local function show_completions_formspec(owner, network, pos, player_name, is_owner)
    local completions = superquest.data_processing.get_all_completions(owner, network, pos)

    minetest.show_formspec(player_name, form_name_completions, get_completions_formspec(completions, is_owner))
end

minetest.register_on_player_receive_fields(function(player, formname, fields)
    if formname ~= form_name_owner or not (player and player:is_player()) then
        return
    end

    local player_name = player:get_player_name()

    local pos = node_positions[player_name]
    if pos == nil then
        return
    end

    local meta = minetest.get_meta(pos)
    local owner = meta:get_string("owner")

    if owner ~= player_name and not superquest.privileges.can_edit_all(player_name) then
        return
    end

    if fields.user_mode then
        local network = meta:get_string("network")

        superquest.data_processing.set_mode_for_network(owner, network, true)

        show_formspec(owner, network, pos, true, player_name,
            meta:get_int("max_completions"), meta:get_inventory():get_lists())

    elseif fields.save then
        if not fields.network or not fields.max_completions then
            return
        end

        local old_network = meta:get_string("network")
        local new_network = fields.network:gsub('^%s*(.-)%s*$', '%1')
        local max_completions = tonumber(fields.max_completions)

        if max_completions == nil or max_completions < 0 or max_completions > 1000000 or max_completions % 1 ~= 0 then
            minetest.chat_send_player(player_name, S("Incorrect max completions value! The value should be an integer between 0 and 1000000!"))
            return
        end

        if new_network:len() > superquest.config.max_network_name_length then
            minetest.chat_send_player(player_name, S("Max network name length (@1 symbols) exceeded", superquest.config.max_network_name_length))
            return
        end
        if new_network:find("#") then
            minetest.chat_send_player(player_name, S("Using the '#' symbol is not allowed in the network name"))
            return
        end

        local new_owner = owner
        if superquest.privileges.can_edit_all(player_name) then
            if not fields.owner then
                return
            end

            new_owner = fields.owner:gsub('^%s*(.-)%s*$', '%1')
            
            if new_owner:len() > superquest.config.max_owner_name_length then
                minetest.chat_send_player(player_name, S("Max owner name length (@1 symbols) exceeded", superquest.config.max_owner_name_length))
                return
            end
            if new_owner:find("#") then
                minetest.chat_send_player(player_name, S("Using the '#' symbol is not allowed in the owner name"))
                return
            end
        end

        if old_network ~= new_network or owner ~= new_owner then
            if superquest.data_processing.has_a_reward_box(new_owner, new_network) then
                minetest.chat_send_player(player_name, S("The network already contains a reward box"))
                return
            end

            superquest.data_processing.remove_reward_box_from_network(owner, old_network, pos)
        end

        meta:set_string("owner", new_owner)
        meta:set_string("network", new_network)
        meta:set_int("max_completions", fields.max_completions)
        
        if old_network ~= new_network or owner ~= new_owner then
            superquest.data_processing.add_reward_box_to_network(new_owner, new_network, pos)
        end
    
    elseif fields.completions then
        local network = meta:get_string("network")
        if owner ~= "" and network ~= "" then
            show_completions_formspec(owner, network, pos, player_name, true)
        end
    end
end)

minetest.register_on_player_receive_fields(function(player, formname, fields)
    if formname ~= form_name_user or not (player and player:is_player()) then
        return
    end

    local player_name = player:get_player_name()
    
    local pos = node_positions[player_name]
    if pos == nil then
        return
    end

    local meta = minetest.get_meta(pos)
    local owner = meta:get_string("owner")
    local network = meta:get_string("network")

    if fields.owner_mode then
        if owner ~= player_name and not superquest.privileges.can_edit_all(player_name) then
            return
        end

        superquest.data_processing.set_mode_for_network(owner, network, false)

        show_formspec(owner, network, pos, false, player_name, meta:get_int("max_completions"), meta:get_inventory():get_lists())
    
    elseif fields.completions then
        if owner ~= "" and network ~= "" then
            show_completions_formspec(owner, network, pos, player_name, false)
        end
    end
end)

minetest.register_on_player_receive_fields(function(player, formname, fields)
    if formname ~= form_name_completions or not (player and player:is_player()) then
        return
    end

    local player_name = player:get_player_name()
    
    local pos = node_positions[player_name]
    if pos == nil then
        return
    end

    local meta = minetest.get_meta(pos)
    local owner = meta:get_string("owner")
    local network = meta:get_string("network")

    if owner == "" or network == "" then
        return
    end

    if owner ~= player_name and not superquest.privileges.can_edit_all(player_name) then
        return
    end

    if fields.reset_completions then
        superquest.data_processing.reset_all_completions(owner, network, pos)
    end
end)

minetest.register_craft({
    type = "shaped",
    output = "superquest:reward_box",
    recipe = {
        {"default:gold_ingot", "default:steel_ingot", "dye:red"},
        {"group:wood", "default:steel_ingot", "group:wood"},
        {"default:steel_ingot", "default:steelblock", "default:steel_ingot"}
    }
})

minetest.register_craft({
    type = "shaped",
    output = "superquest:reward_box_flat",
    recipe = {
        {"default:gold_ingot", "default:steel_ingot", "dye:red"},
        {"group:wood", "default:steel_ingot", "group:wood"},
        {"default:steel_ingot", "default:steel_ingot", "default:steel_ingot"}
    }
})