local S = superquest.S

local top_command_name = minetest.get_current_modname()

local commands_list = {}

local function generate_network_description(network, elements)
    local description = S("Network") .. " " .. network .. ":\n"

    if elements.flags then
        for _, v in ipairs(elements.flags) do
            description = description .. "\t" .. S("Flag at") .. " (" .. v .. ")\n"
        end
    end

    if elements.reward_boxes then
        for _, v in ipairs(elements.reward_boxes) do
            description = description .. "\t" .. S("Reward Box at") .. " (" .. v .. ")\n"
        end
    end

    return description
end

local function elements_to_waypoints(elements)
    local waypoints = {}

    for _, v in ipairs(elements.flags) do
        waypoints[superquest.storage.key_to_coords(v)] = { name = S("Flag"), color = 0xff0000 }
    end

    for _, v in ipairs(elements.reward_boxes) do
        waypoints[superquest.storage.key_to_coords(v)] = { name = S("Reward Box"), color = 0xffff00 }
    end

    return waypoints
end

commands_list["list_my_networks"] = {
    params = "",
    description = S("List networks of the current user"),
    func = function(name, param)
        local networks_list = superquest.storage.get_networks_list()

        if networks_list[name] ~= nil and next(networks_list[name], nil) ~= nil then
            local res = S("Networks") .. ":\n"

            for k, _ in pairs(networks_list[name]) do
                res = res .. "\t" .. k .. "\n"
            end

            minetest.chat_send_player(name, res)
        else
            minetest.chat_send_player(name, S("No networks"))
        end

        return true
    end,
}

commands_list["list_my_network_elements"] = {
    params = "<network_name>",
    description = S("List all the elements of a particular network of the current user"),
    func = function(name, param)
        if param == nil then
            minetest.chat_send_player(name, S("Network name is not specified"))
            return true
        end

        local network = param:gsub('^%s*(.-)%s*$', '%1')

        local elements = superquest.data_processing.get_network_elements(name, network)
        if elements == nil then
            minetest.chat_send_player(name, S("Failed to find network") .. ": " .. network)
            return true
        end

        minetest.chat_send_player(name, generate_network_description(network, elements))

        superquest.hud.show_waypoints(minetest.get_player_by_name(name), elements_to_waypoints(elements))

        return true
    end
}

local command_description = S("Commands") .. ":\n"
for k, v in pairs(commands_list) do
    command_description = command_description .. "/" .. top_command_name .. " " .. k .. " " .. v.params .. "\n" .. v.description .. "\n\n"
end
command_description = command_description:gsub('^%s*(.-)%s*$', '%1')

minetest.register_chatcommand(top_command_name, {
    params = "<command_name> [<parameters>]",
    description = command_description,
    func = function(name, param)
        local command_name
        local subparam
        local space_index = param:find(" ")
        if space_index ~= nil then
            command_name = param:sub(1, space_index - 1)
            subparam = param:sub(space_index + 1, -1)
        else
            command_name = param
            subparam = nil
        end

        local command = commands_list[command_name]
        if (command ~= nil) then
            return command.func(name, subparam)
        else
            minetest.chat_send_player(name, S("Unknown command") .. ": " .. command_name)
            return true
        end
    end
})

local top_admin_command_name = top_command_name .. "_admin"

local admin_commands_list = {}

admin_commands_list["list_owners"] = {
    params = "",
    description = S("List players which own networks"),
    func = function(name, param)
        local networks_list = superquest.storage.get_networks_list()
        local res = S("Players") .. ":\n";

        for k, v in pairs(networks_list) do
            res = res .. "\t" .. k .. "\n"
        end

        minetest.chat_send_player(name, res);

        return true;
    end
}

admin_commands_list["list_player_networks"] = {
    params = "<player_name>",
    description = S("List networks of a player"),
    func = function(name, param)
        local networks_list = superquest.storage.get_networks_list()
        
        if param == nil then
            minetest.chat_send_player(name, S("Player is not specified"))
            return true
        end

        local player_name = param:gsub('^%s*(.-)%s*$', '%1')

        if networks_list[player_name] ~= nil and next(networks_list[player_name], nil) ~= nil then
            local res = S("Networks") .. ":\n"

            for k, _ in pairs(networks_list[player_name]) do
                res = res .. "\t" .. k .. "\n"
            end

            minetest.chat_send_player(name, res)
        else
            minetest.chat_send_player(name, S("No networks"))
        end

        return true
    end
}

admin_commands_list["list_player_network_elements"] = {
    params = "<player_name>#<network_name>",
    description = S("List elements of a network of a player"),
    func = function(name, param)
        if param == nil then
            minetest.chat_send_player(name, S("Parameter is not specified"))
            return true
        end

        local delimiter_pos = param:find("#")
        if not delimiter_pos or delimiter_pos == 1 or delimiter_pos == #param then
            minetest.chat_send_player(name, S("Incorrect parameter format"))
            return true
        end

        local player = param:sub(1, delimiter_pos - 1):gsub('^%s*(.-)%s*$', '%1')
        local network = param:sub(delimiter_pos + 1, -1):gsub('^%s*(.-)%s*$', '%1')
        
        if #player == 0 or #network == 0 or network:find("#") then
            minetest.chat_send_player(name, S("Incorrect parameter value"))
            return true
        end

        local elements = superquest.data_processing.get_network_elements(player, network)
        if elements == nil then
            minetest.chat_send_player(name, S("Failed to find network") .. ": " .. network)
            return true
        end

        minetest.chat_send_player(name, generate_network_description(network, elements))

        superquest.hud.show_waypoints(minetest.get_player_by_name(name), elements_to_waypoints(elements))

        return true
    end
}

local admin_command_description = S("Commands") .. ":\n"
for k, v in pairs(admin_commands_list) do
    admin_command_description = admin_command_description .. "/" .. top_admin_command_name .. " " .. k .. " " .. v.params .. "\n" .. v.description .. "\n\n"
end
admin_command_description = admin_command_description:gsub('^%s*(.-)%s*$', '%1')

minetest.register_chatcommand(top_admin_command_name, {
    params = "<command_name> [<parameters>]",
    description = admin_command_description,
    privs = { [superquest.privileges.admin_priv_name] = true },
    func = function(name, param)
        local command_name
        local subparam
        local space_index = param:find(" ")
        if space_index ~= nil then
            command_name = param:sub(1, space_index - 1)
            subparam = param:sub(space_index + 1, -1)
        else
            command_name = param
            subparam = nil
        end

        local command = admin_commands_list[command_name]
        if (command ~= nil) then
            return command.func(name, subparam)
        else
            minetest.chat_send_player(name, S("Unknown command") .. ": " .. command_name)
            return true
        end
    end
})