local NetworkBase = {}

local function get_empty_flags_table()
    return {
        players_reached = {}
    }
end

local function get_empty_rb_table()
    return {
        completions = {}
    }
end

NetworkBase.save = function(self, ...)
    superquest.storage.set_network_data(self._owner_name, self._network_name, self._network_data)
    return ...
end

NetworkBase.add_flag = function(self, pos)
    self._network_data.flags = self._network_data.flags or {}
    self._network_data.flags[superquest.storage.coords_to_key(pos)] = get_empty_flags_table()
end

NetworkBase.remove_flag = function(self, pos)
    if not self._network_data.flags then
        return
    end

    self._network_data.flags[superquest.storage.coords_to_key(pos)] = nil
    if not next(self._network_data.flags, nil) then
        self._network_data.flags = nil
    end
end

NetworkBase.add_player_reached_flag = function(self, pos, player_name)
    local key = superquest.storage.coords_to_key(pos)
    self._network_data.flags = self._network_data.flags or {}
    self._network_data.flags[key] = self._network_data.flags[key] or get_empty_flags_table()
    if self._network_data.flags[key].players_reached[player_name] then
        -- The flag was already punched
        return false
    end
    self._network_data.flags[key].players_reached[player_name] = 1
    return true
end

NetworkBase.clear_reached_flags_for_player = function(self, player_name)
    if self._network_data.flags then
        for _, v in pairs(self._network_data.flags) do
            v.players_reached[player_name] = nil
        end
    end
end

NetworkBase.get_flags_stats_for_player = function(self, player_name)
    local reached = 0
    local total = 0

    if self._network_data.flags then
        for _, v in pairs(self._network_data.flags) do
            if v.players_reached[player_name] then
                reached = reached + 1
            end
            total = total + 1
        end
    end
    return reached, total
end

NetworkBase.add_reward_box = function(self, pos)
    self._network_data.reward_boxes = self._network_data.reward_boxes or {}
    self._network_data.reward_boxes[superquest.storage.coords_to_key(pos)] = get_empty_rb_table()
end

NetworkBase.remove_reward_box = function(self, pos)
    if not self._network_data.reward_boxes then
        return
    end

    self._network_data.reward_boxes[superquest.storage.coords_to_key(pos)] = nil
    if not next(self._network_data.reward_boxes, nil) then
        self._network_data.reward_boxes = nil
    end
end

NetworkBase.has_reward_box = function(self)
    return self._network_data.reward_boxes ~= nil
end

NetworkBase.get_completions_for_player = function(self, pos, player_name)
    local key = superquest.storage.coords_to_key(pos)
    self._network_data.reward_boxes = self._network_data.reward_boxes or {}
    self._network_data.reward_boxes[key] = self._network_data.reward_boxes[key] or get_empty_rb_table()

    return self._network_data.reward_boxes[key].completions[player_name] or 0
end

NetworkBase.set_completions_for_player = function(self, pos, player_name, completions)
    local key = superquest.storage.coords_to_key(pos)
    self._network_data.reward_boxes = self._network_data.reward_boxes or {}
    self._network_data.reward_boxes[key] = self._network_data.reward_boxes[key] or get_empty_rb_table()
    self._network_data.reward_boxes[key].completions[player_name] = completions
end

NetworkBase.get_all_completions = function(self, pos)
    local key = superquest.storage.coords_to_key(pos)
    self._network_data.reward_boxes = self._network_data.reward_boxes or {}
    self._network_data.reward_boxes[key] = self._network_data.reward_boxes[key] or get_empty_rb_table()

    local completions = {}
    for k, v in pairs(self._network_data.reward_boxes[key].completions) do
        completions[k] = v
    end
    return completions
end

NetworkBase.reset_all_completions = function(self, pos)
    local key = superquest.storage.coords_to_key(pos)
    self._network_data.reward_boxes = self._network_data.reward_boxes or {}
    self._network_data.reward_boxes[key] = self._network_data.reward_boxes[key] or get_empty_rb_table()

    self._network_data.reward_boxes[key].completions = {}
end

NetworkBase.is_in_user_mode = function(self)
    return self._network_data.user_mode ~= nil
end

NetworkBase.set_mode = function(self, is_user_mode)
    self._network_data.user_mode = is_user_mode and 1 or nil
end

NetworkBase.get_network_elements = function(self)
    local elements = {
        flags = {},
        reward_boxes = {}
    }

    if self._network_data.flags then
        for k, _ in pairs(self._network_data.flags) do
            table.insert(elements.flags, k)
        end
    end

    if self._network_data.reward_boxes then
        for k, _ in pairs(self._network_data.reward_boxes) do
            table.insert(elements.reward_boxes, k)
        end
    end

    return elements
end

local no_save_after = {
    [NetworkBase.save] = true,
    [NetworkBase.get_flags_stats_for_player] = true,
    [NetworkBase.has_reward_box] = true,
    [NetworkBase.get_completions_for_player] = true,
    [NetworkBase.get_all_completions] = true,
    [NetworkBase.is_in_user_mode] = true,
    [NetworkBase.get_network_elements] = true
}

NetworkBase.__index = function(self, key)
    return function(...)
        local func = NetworkBase[key]
        if self._auto_save and not no_save_after[func] then
            return NetworkBase.save(self, func(...))
        else
            return func(...)
        end
    end
end

superquest.Network = function(owner_name, network_name, auto_save)
    if owner_name == "" or network_name == "" then
        return nil
    end

    local data = {
        _owner_name = owner_name,
        _network_name = network_name,
        _network_data = superquest.storage.get_network_data(owner_name, network_name) or {},
        _auto_save = not (auto_save == false),
    }
    setmetatable(data, NetworkBase)
    return data
end
