--[[
    Timer object internal structure:
    {
        -- Name of the player to which the timer is shown
        _player_name = <string>,
        -- Number of seconds from start (not from now!) to expiration
        _seconds = <number>,
        -- State of the running timer
        _r_state = {
            -- Time, at which the timer was started
            start_time = <number>
        }
    }
]]

superquest.timer = {}

local huds = {}

-- To do: consider allowing having separate callbacks for different timers
local expiration_callback = function(_) end

local function update_timer(timer, r_state)
    if r_state ~= timer._r_state then
        -- The current running state object is outdated because the timer was stopped
        return
    end

    local cur_time = minetest.get_gametime()
    local seconds_remained = timer._seconds - (cur_time - r_state.start_time)

    if seconds_remained <= 0 then
        superquest.timer.stop(timer)
        expiration_callback(timer._player_name)
        return
    end

    huds[r_state]:set_time(seconds_remained)

    minetest.after(1, update_timer, timer, r_state)
end

superquest.timer.start = function(timer)
    if timer._r_state then
        return
    end

    timer._r_state = {
        start_time = minetest.get_gametime(),
    }
    huds[timer._r_state] = superquest.hud.TimerHud(timer._player_name)

    -- Perform deferred call to prevent calling expiration_callback inside timer.start()
    minetest.after(0.1, update_timer, timer, timer._r_state)
end

superquest.timer.stop = function(timer)
    if not timer._r_state then
        -- The timer is not started
        return
    end

    local cur_time = minetest.get_gametime()
    timer._seconds = timer._seconds - (cur_time - timer._r_state.start_time)
    if timer._seconds < 0 then
        timer._seconds = 0
    end

    if huds[timer._r_state] then
        huds[timer._r_state]:remove()
        huds[timer._r_state] = nil
    end
    timer._r_state = nil
end

superquest.timer.adjust = function(timer, diff)
    local is_running = timer._r_state and true or false

    if is_running then
        superquest.timer.stop(timer)
    end

    timer._seconds = timer._seconds + diff

    if timer._seconds < 0 then
        timer._seconds = 0
    end

    if is_running or timer._seconds == 0 then
        superquest.timer.start(timer)
    end
end

superquest.timer.expire = function(timer)
    superquest.timer.stop(timer)
    timer._seconds = 0
    superquest.timer.start(timer)
end

superquest.Timer = function(player_name, seconds)
    local data = {
        _player_name = player_name,
        -- Add +1 second to guarantee that the timeout will be not less than specified
        _seconds = seconds + 1
    }
    return data
end

superquest.timer.set_expiration_callback = function(callback)
    expiration_callback = callback
end
