local S = superquest.S

local form_name = minetest.get_current_modname()..":flag_settings"
local node_positions = {}

local coords_field_hint = S("Should contain positive integer values of distance of x,y,z, relative to the position of the flag, separated by comma") .. ".\n" ..
                          S("The values in the first field are subtracted from the flag's position while the values in the second field are added to it") .. ".\n" ..
                          S("The flag will be automatically triggered for players detected within the cube described by the coordinates") .. "."

local function get_formspec(owner, network, from_dev, to_dev, can_edit_owner)
    local height_add = 0
    if can_edit_owner then
        height_add = height_add + 0.5
    end
    if superquest.config.flags_autotrigger then
        height_add = height_add + 1.2
    end

    local formspec_data =
        "formspec_version[4]" ..
        "size[6,"..tostring(3.4 + height_add).."]"
    if can_edit_owner then
        formspec_data = formspec_data .. "field[0.5,0.5;5.0,0.8;owner;" .. S("Owner") .. ";"..minetest.formspec_escape(owner).."]" ..
        "container[0.5,1.8]"
    else
        formspec_data = formspec_data .. "label[0.5,0.5;"..minetest.formspec_escape(S("Owner") .. ": "..owner).."]" ..
        "container[0.5,1.3]"
    end
    formspec_data = formspec_data ..
        "field[0,0;5.0,0.8;network;" .. S("Network") .. ";"..minetest.formspec_escape(network).."]"
    if superquest.config.flags_autotrigger then
        formspec_data = formspec_data .. "field[0,1.3;2.2,0.8;from_coords;" .. S("From coords") .. ";"..minetest.formspec_escape(from_dev).."]" ..
        "field[2.8,1.3;2.2,0.8;to_coords;" .. S("To coords") .. ";"..minetest.formspec_escape(to_dev).."]" ..
        "tooltip[0,1.3;5,0.8;"..coords_field_hint.."]" ..
        "container[0,2.3]"
    else
        formspec_data = formspec_data .. "container[0,1.1]"
    end
    formspec_data = formspec_data ..
        "button_exit[0,0;5.0,0.8;save;" .. S("Save") .. "]" ..
        "container_end[]" ..
        "container_end[]"

    return formspec_data
end

local function try_to_count_flag(pos, player, print_to_chat)
    local meta = minetest.get_meta(pos)

    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    local player_name = player:get_player_name()

    if owner == "" or network_name == "" then
        if print_to_chat then
            minetest.chat_send_player(player_name, S("The flag is not configured") .. "!")
        end

        return
    end

    local network = superquest.Network(owner, network_name)
    assert(network)

    if (owner == player_name or superquest.privileges.can_edit_all(player_name))
            and not network:is_in_user_mode() then
        if print_to_chat then
            minetest.chat_send_player(player_name, "You need to switch to the User mode to participate")
        end
        return
    end

    local timed_duration = network:get_timed_duration()

    if timed_duration ~= 0 then
        local active_network = superquest.active_timed_quests:get_active_network_for_player(player_name)
        if not active_network or active_network.owner ~= owner or active_network.network ~= network_name then
            if print_to_chat then
                minetest.chat_send_player(player_name, S("The flag belongs to a timed quest. You need to start it first."))
            end
            return
        end
    end

    local res = network:add_player_reached_flag(pos, player_name)
    if not res then
        if print_to_chat then
            minetest.chat_send_player(player_name, S("The flag is already counted") .. "!")
        end
        return
    end

    local reached, total = network:get_flags_stats_for_player(player_name)
    if reached >= total then
        if timed_duration then
            superquest.active_timed_quests:stop_and_remove(player_name)
        end

        if superquest.config.teleportation then
            if network:get_teleport_after_compl() then
                superquest.teleportation.set_tp_dest_to_quest_machine(player_name, owner, network_name, false)
                minetest.chat_send_player(player_name, S("Quest completed").."! "..S("You can type '/superquest teleport' to teleport to the Quest Machine."))
            end
        end
    end

    local maxpos_deviation_table = {
        [0] = vector.new(0, 0.5, 0),
              vector.new(0, 0, 0.5),
              vector.new(0, 0, -0.5),
              vector.new(0.5, 0, 0),
              vector.new(-0.5, 0, 0),
              vector.new(0, -0.5, 0),
    }
    local deviation = maxpos_deviation_table[math.floor(minetest.get_node(pos).param2 / 4)]

    minetest.add_particlespawner({
        amount = 50,
        time = 0.01,
        minpos = vector.add(pos, deviation * 2),
        maxpos = vector.add(pos, deviation * 3),
        minvel = vector.new(-2, -2, -2),
        maxvel = vector.new(2, 2, 2),
        minexptime = 0.3,
        maxexptime = 0.3,
        minsize = 0.4,
        maxsize = 0.4,
        texture = "superquest_particle_red.png"
    })

    superquest.hud.show_popup(player, S("Flags counted") .. ": "..reached.." / "..total)
end

minetest.register_node("superquest:flag", {
    description = S("Flag"),
    drawtype = "mesh",
    mesh = "superquest_flag.obj",
    tiles = { "superquest_flag.png" },
    inventory_image = "superquest_flag.png",
    selection_box = {
        type = "fixed",
        fixed = { -0.1, -0.5, 0.1, 0.1, 1.5, -0.5 }
    },
    paramtype = "light",
    paramtype2 = "facedir",
    sunlight_propagates = true,
	use_texture_alpha = "clip",
    walkable = false,
    groups = { cracky=1, oddly_breakable_by_hand = 2 },

    after_place_node = function(pos, placer, itemstack)
        local meta = minetest.get_meta(pos)
        if placer and placer:is_player() then
            meta:set_string("owner", placer:get_player_name())
        end
    end,

    on_rightclick = function(pos, node, clicker, itemstack, pointed_thing)
        if not (clicker and clicker:is_player()) then
            return
        end

        local meta = minetest.get_meta(pos)

        local player_name = clicker:get_player_name()

        if not (meta:get_string("owner") == player_name or superquest.privileges.can_edit_all(player_name)) then
            return
        end

        node_positions[player_name] = pos

        local from_v = meta:get_int("scan_from_x")..","..meta:get_int("scan_from_y")..","..meta:get_int("scan_from_z")
        local to_v = meta:get_int("scan_to_x")..","..meta:get_int("scan_to_y")..","..meta:get_int("scan_to_z")

        minetest.show_formspec(player_name, form_name,
            get_formspec(meta:get_string("owner"), meta:get_string("network"), from_v, to_v, superquest.privileges.can_edit_all(player_name)))
    end,

    on_destruct = function(pos)
        local meta = minetest.get_meta(pos)

        local network = superquest.Network(meta:get_string("owner"), meta:get_string("network"))
        if not network then
            return
        end
        network:remove_flag(pos)

        -- We don't need to check the player because flags should be placed in protected areas anyway
    end,

    on_punch = function(pos, node, puncher)
        if not (puncher and puncher:is_player()) then
            return
        end

        try_to_count_flag(pos, puncher, true)
    end,

    on_place = function(itemstack, placer, pointed_thing)
        local dir = minetest.dir_to_wallmounted(
            vector.subtract(pointed_thing.under, pointed_thing.above)
        )
        local facedir_table = {[0] = 20, 0, 17, 15, 8, 6}
        if dir == 0 then
            facedir_table[dir] = facedir_table[dir] + (4 - minetest.dir_to_facedir(placer:get_look_dir())) % 4
        elseif dir == 1 then
            facedir_table[dir] = facedir_table[dir] + minetest.dir_to_facedir(placer:get_look_dir())
        end
        minetest.item_place(itemstack, placer, pointed_thing, facedir_table[dir])
        return itemstack
    end,
})

minetest.register_on_player_receive_fields(function(player, formname, fields)
    if formname ~= form_name or not (player and player:is_player()) then
        return
    end

    local player_name = player:get_player_name()

    local pos = node_positions[player_name]
    if pos == nil then
        return
    end

    local meta = minetest.get_meta(pos)
    local owner = meta:get_string("owner")

    if owner ~= player_name and not superquest.privileges.can_edit_all(player_name) then
        return
    end

    if fields.save then
        if not fields.network then
            return
        end

        local old_network = meta:get_string("network")
        local new_network = fields.network:gsub('^%s*(.-)%s*$', '%1')

        if new_network:len() > superquest.config.max_network_name_length then
            minetest.chat_send_player(player_name, S("Max network name length (@1 symbols) exceeded", superquest.config.max_network_name_length))
            return
        end
        if new_network:find("#") then
            minetest.chat_send_player(player_name, S("Using the '#' symbol is not allowed in the network name"))
            return
        end

        local new_owner = owner
        if superquest.privileges.can_edit_all(player_name) then
            if not fields.owner then
                return
            end

            new_owner = fields.owner:gsub('^%s*(.-)%s*$', '%1')

            if new_owner:len() > superquest.config.max_owner_name_length then
                minetest.chat_send_player(player_name, S("Max owner name length (@1 symbols) exceeded", superquest.config.max_owner_name_length))
                return
            end
            if new_owner:find("#") then
                minetest.chat_send_player(player_name, S("Using the '#' symbol is not allowed in the owner name"))
                return
            end
        end

        if old_network ~= new_network or owner ~= new_owner then
            local old_network = superquest.Network(owner, old_network)
            if old_network then
                old_network:remove_flag(pos)
            end

            meta:set_string("owner", new_owner)
            meta:set_string("network", new_network)

            local network = superquest.Network(new_owner, new_network)
            if network then
                network:add_flag(pos)
            end
        end

        if superquest.config.flags_autotrigger then
            if not fields.from_coords or not fields.to_coords then
                return
            end

            local from_coords = fields.from_coords:split(",")
            local to_coords = fields.to_coords:split(",")

            if #from_coords ~= 3 or #to_coords ~= 3 then
                minetest.chat_send_player(player_name, S("Incorrect format of distance fields"))
                return
            end

            local from_x = tonumber(from_coords[1])
            local from_y = tonumber(from_coords[2])
            local from_z = tonumber(from_coords[3])
            local to_x = tonumber(to_coords[1])
            local to_y = tonumber(to_coords[2])
            local to_z = tonumber(to_coords[3])

            if not from_x or from_x < 0 or from_x > superquest.config.flags_autotrigger_max_distance or
               not from_y or from_y < 0 or from_y > superquest.config.flags_autotrigger_max_distance or
               not from_z or from_z < 0 or from_z > superquest.config.flags_autotrigger_max_distance or
               not to_x or to_x < 0 or to_x > superquest.config.flags_autotrigger_max_distance or
               not to_y or to_y < 0 or to_y > superquest.config.flags_autotrigger_max_distance or
               not to_z or to_z < 0 or to_z > superquest.config.flags_autotrigger_max_distance then

                minetest.chat_send_player(player_name, S("Incorrect distance values. Distance values should be integers between 0 and @1", superquest.config.flags_autotrigger_max_distance))
                return
            end

            meta:set_int("scan_from_x", from_x)
            meta:set_int("scan_from_y", from_y)
            meta:set_int("scan_from_z", from_z)
            meta:set_int("scan_to_x", to_x)
            meta:set_int("scan_to_y", to_y)
            meta:set_int("scan_to_z", to_z)
        end
    end
end)

minetest.register_craft({
    type = "shaped",
    output = "superquest:flag 2",
    recipe = {
        {"farming:cotton", "dye:red", "farming:cotton"},
        {"farming:cotton", "default:stick", "farming:cotton"},
        {"farming:cotton", "default:stick", "farming:cotton"}
    }
})

if superquest.config.flags_autotrigger then
    minetest.register_abm({
        nodenames = { "superquest:flag" },
        interval = superquest.config.flags_autotrigger_interval,
        chance = 1,
        action = function(pos, node)
            local meta = minetest.get_meta(pos)
            local from_x_dev = meta:get_int("scan_from_x")
            local from_y_dev = meta:get_int("scan_from_y")
            local from_z_dev = meta:get_int("scan_from_z")
            local to_x_dev = meta:get_int("scan_to_x")
            local to_y_dev = meta:get_int("scan_to_y")
            local to_z_dev = meta:get_int("scan_to_z")

            local from_v = vector.subtract(pos, vector.new(from_x_dev, from_y_dev, from_z_dev))
            local to_v = vector.add(pos, vector.new(to_x_dev, to_y_dev, to_z_dev))

            local objs = minetest.get_objects_in_area(from_v, to_v)

            for _, obj in pairs(objs) do
                if obj:is_player() then
                    try_to_count_flag(pos, obj, false)
                end
            end
        end
    })
end
