local serpent = require("serpent")

local privileges = {}
local node_defs = {}
local on_player_receive_fields = {}
local crafts = {}
local abms = {}
local aliases = {}
local chat = {}
local chatcommands = {}
local on_dieplayer_handlers = {}
local on_joinplayer_handlers = {}
local on_leaveplayer_handlers = {}

local players = {}
local nodes = {}

local worldpath = nil

local mod_storage = {
    _data = {
        fields = {}
    },

    from_table = function(self, t)
        assert(t.fields ~= nil)
        self._data = mut.copy_object(t)
    end,

    get_int = function(self, name)
        local value = self._data.fields[name]
        if value == nil then
            return 0
        end
        return tonumber(value)
    end,

    get_string = function(self, name)
        return self._data.fields[name] or ""
    end,

    set_int = function(self, name, value)
        assert(type(value) == "number")
        self._data.fields[name] = tostring(value)
    end,

    set_string = function(self, name, value)
        assert(type(value) == "string")
        if value == "" then
            self._data.fields[name] = nil
        else
            self._data.fields[name] = value
        end
    end,

    to_table = function(self)
        return mut.copy_object(self._data)
    end,
}


_G.vector = {
    add = function(vec1, vec2)
        return { x = vec1.x + vec2.x, y = vec1.y + vec2.y, z = vec1.z + vec2.z }
    end,

    new = function(x, y, z)
        return { x = x, y = y, z = z }
    end,

    subtract = function(vec1, vec2)
        return { x = vec1.x - vec2.x, y = vec1.y - vec2.y, z = vec1.z - vec2.z }
    end,

    to_string = function(vec)
        return "(" .. vec.x .. ", " .. vec.y .. ", " .. vec.z .. ")"
    end,
}


local ItemStackBase = {
    get_count = function(self)
        return self.count
    end,
}
ItemStackBase.__index = ItemStackBase

function _G.ItemStack(par)
    local stack_obj = { name = "", count = 0 }
    setmetatable(stack_obj, ItemStackBase)
    if type(par) == "table" then
        stack_obj.name = par.name or stack_obj.name
        stack_obj.count = par.count or stack_obj.count
    elseif type(par) == "string" then
        local _, e, name = par:find("^(%S+)")
        stack_obj.name = name
        if #par >= e + 2 then
            local _, _, count = par:find(par:find("(%S+)$", e + 2))
            stack_obj.count = tonumber(count)
        else
            stack_obj.count = 1
        end
    end

    return stack_obj
end


local InventoryBase = {
    get_lists = function(self)
        local res_list = {}

        for k_inv, v_inv in pairs(self._data) do
            res_list[k_inv] = {}

            for i = 1,v_inv.size do
                table.insert(res_list[k_inv], v_inv.slots[i])
            end
        end

        return res_list
    end,

    is_empty = function(self, name)
        local size = self._data[name].size
        for i = 1,size do
            local slot = self._data[name].slots[i]
            if slot and slot.name ~= "" and slot.count > 0 then
                return false
            end
        end

        return true
    end,

    set_size = function(self, name, size)
        if not size then
            self._data[name] = nil
        else
            local old_size;
            if not self._data[name] then
                self._data[name] = {
                    size = size,
                    slots = {}
                }
                old_size = 0
            else
                old_size = self._data[name].size
            end

            if old_size < size then
                for i = old_size+1,size do
                    self._data[name].slots[i] = ItemStack()
                end
            elseif size < old_size then
                for i = size+1,old_size do
                    self._data[name].slots[i] = nil
                end
            end
        end
    end,
}
InventoryBase.__index = InventoryBase


local MetadataBase = {
    contains = function(self, name)
        return self._data.fields[name] ~= nil
    end,

    get_int = function(self, name)
        return tonumber(self._data.fields[name] or "0")
    end,

    get_inventory = function(self)
        return self._data.inventory
    end,

    get_string = function(self, name)
        return self._data.fields[name] or ""
    end,

    set_int = function(self, name, value)
        assert(type(value) == "number")
        self._data.fields[name] = tostring(value)
    end,

    set_string = function(self, name, value)
        assert(type(value) == "string")
        self._data.fields[name] = value
    end,
}
MetadataBase.__index = MetadataBase


local PlayerBase = {
    get_look_dir = function(self)
        return 0
    end,

    get_player_name = function(self)
        return self._data.name
    end,

    is_player = function(self)
        return true
    end,
}
PlayerBase.__index = PlayerBase


_G.mut = {}

-- Check if two objects have equivalent data structure. The function doesn't take references into account.
function mut.compare_objects(obj1, obj2)
    if type(obj1) ~= type(obj2) then
        print("aaa type(obj1) = " .. type(obj1) .. " (" .. obj1 .."), type(obj2) = " .. type(obj2))
        return false
    elseif type(obj1) ~= "table" then
        if obj1 ~= obj2 then
            print("ddd type(obj1) = " ..type(obj1) .. ", obj1 = \n" .. obj1 .. "\n obj2 = \n" .. obj2)
        end
        return obj1 == obj2
    end

    if mut.count_table_entries(obj1) ~= mut.count_table_entries(obj2) then
        print("bbb")
        return false
    end

    for k, v in pairs(obj1) do
        assert(type(k) ~= table)

        if not mut.compare_objects(v, obj2[k]) then
            print("ccc")
            return false
        end
    end

    return true
end

local function _copy_object(obj, old_tables, new_tables)
    if type(obj) ~= "table" then
        return obj
    end

    -- If the object was referenced before in another place, use its corresponding copy
    for k, v in ipairs(old_tables) do
        if obj == v then
            return new_tables[k]
        end
    end

    local obj_copy = {}
    table.insert(old_tables, obj)
    table.insert(new_tables, obj_copy)

    for k, v in pairs(obj) do
        obj_copy[_copy_object(k, old_tables, new_tables)] = _copy_object(v, old_tables, new_tables)
    end

    return obj_copy
end

function mut.copy_object(obj)
    return _copy_object(obj, {}, {})
end

function mut.count_table_entries(t)
    local count = 0

    for _ in pairs(t) do
        count = count + 1
    end

    return count
end

function mut.create_player(name, data, no_join)
    local privs = {}
    for _, v in ipairs(data.privs or {}) do
        privs[v] = true
    end

    players[name] = {
        _data = {
            name = name,
            pos = data.pos or vector.new(0, 0, 0),
            privs = privs,
            active_formspec = nil
        }
    }
    setmetatable(players[name], PlayerBase)

    if not no_join then
        for _, func in ipairs(on_joinplayer_handlers) do
            func(players[name], true)
        end
    end

    return { type = "player", name = name }
end

function mut.dig_node(pos, digger)
    if digger then
        assert(digger.type == "player")
    end

    local node_data = nodes[vector.to_string(pos)]._data
    local node_def = node_defs[node_data.name]

    if node_def.can_dig and not node_def.can_dig(pos, players[digger.name]) then
        return false
    end

    if node_def.on_destruct then
        node_def.on_destruct(pos)
    end

    local player_obj = nil
    if digger then
        player_obj = players[digger.name]
    end

    minetest.node_dig(pos, nil, player_obj)

    return true
end

function mut.get_mod_storage_data()
    return mod_storage:to_table()
end

function mut.is_formspec_active(player_name, formspec_name)
    return players[player_name]._data.active_formspec == formspec_name
end

function mut.place_node(pos, data)
    if data.placer then
        -- To do: maybe support mobs in the future
        assert(data.placer.type == "player")
    end

    local pointed_thing = {
        above = pos,
        under = vector.add(pos, data.bottom_dir or vector.new(0, -1, 0))
    }

    local node_def = node_defs[data.stack.name]

    local ret_stack
    if node_def.on_place then
        ret_stack = node_def.on_place(data.stack, players[data.placer.name], pointed_thing)
    else
        ret_stack = minetest.item_place(data.stack, players[data.placer.name], pointed_thing, nil)
    end

    if node_def.after_place_node then
        node_def.after_place_node(pos, players[data.placer.name], ret_stack, pointed_thing)
    end

    return { type = "node", pos = pos }, ret_stack
end

function mut.punch_node(pos, puncher)
    assert(puncher.type == "player")

    local node_data = nodes[vector.to_string(pos)]._data
    local node_def = node_defs[node_data.name]

    if node_def.on_punch then
        node_def.on_punch(pos, nil, players[puncher.name])
    end
end

function mut.rightclick_node(pos, data)
    if data.clicker then
        assert(data.clicker.type == "player")
    end
    local node_data = nodes[vector.to_string(pos)]._data
    local node_def = node_defs[node_data.name]

    if not node_def.on_rightclick then
        return
    end

    local player_obj = nil
    if data.clicker then
        player_obj = players[data.clicker.name]
    end

    node_def.on_rightclick(pos, nil, player_obj, nil, nil)
end

function mut.set_mod_storage_data(storage)
    mod_storage:from_table(storage)
end

function mut.set_settings(settings)
    minetest.settings._data = mut.copy_object(settings)
end

function mut.submit_active_formspec(player_name, data)
    local active_formspec = players[player_name]._data.active_formspec
    assert(active_formspec)

    for _, v in ipairs(on_player_receive_fields) do
        v(players[player_name], active_formspec, data)
    end
end

_G.minetest = {}

minetest.settings = {
    _data = {}
}

function minetest.chat_send_player(name, message)
    table.insert(chat, { name = name, msg = message })
end

function minetest.check_player_privs(name, privilege)
    return players[name]._data.privs[privilege]
end

function minetest.deserialize(string)
    local ok, res = serpent.load(string)
    assert(ok)
    return res
end

function minetest.dir_to_facedir(dir)
    -- To do
    return 0
end

function minetest.dir_to_wallmounted(dir)
    -- To do
    return 0
end

function minetest.formspec_escape(text)
    -- To do
    return text
end

function minetest.get_current_modname()
    local f = assert(io.open('mod.conf'), "r")

    local modname = ""
    for line in f:lines() do
        modname = line:match('^name = (%S*)')
        if modname then
            break
        end
    end
    f:close()

    return modname
end

function minetest.get_gametime()
    return os.time()
end

function minetest.get_meta(pos)
    return nodes[vector.to_string(pos)]._data.metadata
end

function minetest.get_mod_storage()
    return mod_storage
end

function minetest.get_modpath(modname)
    -- To do
    return "."
end

local function translator(str, ...)
    local res_str = ""
    local cursor = 1
    local string_len = str:len()

    while cursor <= string_len do
        local s = str:find("@", cursor)
        if not s then
            res_str = res_str .. str:sub(cursor)
            break
        end

        if cursor < s then
            res_str = res_str .. str:sub(cursor, s - 1)
        end

        assert(s ~= string_len)

        local symbol = str:sub(s + 1, s + 1)
        if symbol == '\n' or symbol == '=' or symbol == '@' then
            res_str = res_str .. symbol
        elseif symbol == 'n' then
            res_str = res_str .. '\n'
        elseif symbol:match('%d') then
            local num = tonumber(symbol)
            assert(num and arg[num])
            res_str = res_str .. tostring(arg[num])
        else
            error("Unexpected symbol after @")
        end

        cursor = s + 2
    end

    return res_str
end

function minetest.get_translator(_)
    return translator
end

function minetest.get_worldpath()
    if not worldpath then
        local f = assert(io.popen("mktemp -d"))
        worldpath = f:read("*l")
        f:close()
    end

    return worldpath
end

function minetest.item_place(itemstack, placer, pointed_thing, param2)
    assert(itemstack.count > 0)
    assert(not nodes[vector.to_string(pointed_thing.above)])

    local node = {
        _data = {
            name = itemstack.name,
            param2 = param2,
            metadata = {
                _data = {
                    fields = {},
                    inventory = {
                        _data = {}
                    }
                }
            }
        }
    }
    setmetatable(node._data.metadata, MetadataBase)
    setmetatable(node._data.metadata._data.inventory, InventoryBase)

    nodes[vector.to_string(pointed_thing.above)] = node

    itemstack.count = itemstack.count - 1

    return pointed_thing.above, itemstack
end

function minetest.mkdir(dir)
    os.execute("mkdir " .. dir)
end

function minetest.node_dig(pos, node, digger)
    nodes[vector.to_string(pos)] = nil
end

function minetest.register_abm(data)
    table.insert(abms, mut.copy_object(data))
end

function minetest.register_alias(alias, original_name)
    aliases[alias] = original_name
end

function minetest.register_chatcommand(name, data)
    chatcommands[name] = mut.copy_object(data)
end

function minetest.register_craft(data)
    table.insert(crafts, mut.copy_object(data))
end

function minetest.register_node(name, data)
    node_defs[name] = mut.copy_object(data)
end

function minetest.register_on_dieplayer(func)
    table.insert(on_dieplayer_handlers, func)
end

function minetest.register_on_joinplayer(func)
    table.insert(on_joinplayer_handlers, func)
end

function minetest.register_on_leaveplayer(func)
    table.insert(on_leaveplayer_handlers, func)
end

function minetest.register_on_player_receive_fields(func)
    table.insert(on_player_receive_fields, func)
end

function minetest.register_privilege(name, data)
    privileges[name] = mut.copy_object(data)
end

function minetest.serialize(table)
    return serpent.dump(table)
end

function minetest.settings:get_bool(name, value)
    local ret = minetest.settings._data[name] or value
    assert(type(ret) == "boolean")
    return ret
end

function minetest.settings:get(name)
    return minetest.settings._data[name]
end

function minetest.show_formspec(player_name, form_name, formspec)
    players[player_name]._data.active_formspec = form_name
end

local string_mt = getmetatable("")
string_mt.__index["split"] =  function(self, separator)
    local res = {}
    for str in self:gmatch("([^" .. separator .. "]+)") do
        table.insert(res, str)
    end
    return res
end
