local S = superquest.S

local top_command_name = minetest.get_current_modname()

local commands_list = {}

local function generate_network_description(network, elements)
    local description = S("Network") .. " " .. network .. ":\n"

    for _, v in ipairs(elements.flags) do
        description = description .. "\t" .. S("Flag at") .. " " .. v .. "\n"
    end

    for _, v in ipairs(elements.quest_machines) do
        description = description .. "\t" .. S("Quest Machine at") .. " " .. v .. "\n"
    end

    return description
end

local function elements_to_waypoints(elements)
    local waypoints = {}

    for _, v in ipairs(elements.flags) do
        waypoints[superquest.storage.key_to_coords(v)] = { name = S("Flag"), color = 0xff0000 }
    end

    for _, v in ipairs(elements.quest_machines) do
        waypoints[superquest.storage.key_to_coords(v)] = { name = S("Quest Machine"), color = 0xffff00 }
    end

    return waypoints
end

commands_list["list_my_networks"] = {
    params = "",
    description = S("List networks of the current user"),
    func = function(name, param)
        local networks_list = superquest.storage.get_networks_list()

        if networks_list[name] ~= nil and next(networks_list[name], nil) ~= nil then
            local res = S("Networks") .. ":\n"

            for k, _ in pairs(networks_list[name]) do
                res = res .. "\t" .. k .. "\n"
            end

            minetest.chat_send_player(name, res)
        else
            minetest.chat_send_player(name, S("No networks"))
        end

        return true
    end,
}

commands_list["list_my_network_elements"] = {
    params = "<network_name>",
    description = S("List all the elements of a particular network of the current user"),
    func = function(player_name, param)
        if param == nil then
            minetest.chat_send_player(player_name, S("Network name is not specified"))
            return true
        end

        local network_name = param:gsub('^%s*(.-)%s*$', '%1')

        local network_obj = assert(superquest.Network(player_name, network_name))
        local elements = network_obj:get_network_elements()

        minetest.chat_send_player(player_name, generate_network_description(network_name, elements))

        superquest.hud.show_waypoints(minetest.get_player_by_name(player_name), elements_to_waypoints(elements))

        return true
    end
}

commands_list["cancel"] = {
    params = "",
    description = S("Cancel the current active quest"),
    func = function(player_name, _)
        superquest.active_timed_quests:expire_current_for_player(player_name)
    end
}

if superquest.config.teleportation then
    commands_list["teleport"] = {
        params = "",
        description = S("Teleport to the Quest Machine of the last completed or expired quest."),
        func = function(player_name, _)
            superquest.teleportation.teleport_to_destination(player_name)
        end
    }
end

local command_description = S("Commands") .. ":\n"
for k, v in pairs(commands_list) do
    command_description = command_description .. "/" .. top_command_name .. " " .. k .. " " .. v.params .. "\n" .. v.description .. "\n\n"
end
command_description = command_description:gsub('^%s*(.-)%s*$', '%1')

minetest.register_chatcommand(top_command_name, {
    params = "<command_name> [<parameters>]",
    description = command_description,
    func = function(name, param)
        local command_name
        local subparam
        local space_index = param:find(" ")
        if space_index ~= nil then
            command_name = param:sub(1, space_index - 1)
            subparam = param:sub(space_index + 1, -1)
        else
            command_name = param
            subparam = nil
        end

        local command = commands_list[command_name]
        if (command ~= nil) then
            return command.func(name, subparam)
        else
            minetest.chat_send_player(name, S("Unknown command") .. ": " .. command_name)
            return true
        end
    end
})

local function parse_owner_network_name(caller_name, param)
    if param == nil then
        minetest.chat_send_player(caller_name, S("Parameter is not specified"))
        return nil
    end

    local delimiter_pos = param:find("#")
    if not delimiter_pos or delimiter_pos == 1 or delimiter_pos == #param then
        minetest.chat_send_player(caller_name, S("Incorrect parameter format"))
        return nil
    end

    local owner = param:sub(1, delimiter_pos - 1):gsub('^%s*(.-)%s*$', '%1')
    local network_name = param:sub(delimiter_pos + 1, -1):gsub('^%s*(.-)%s*$', '%1')

    if #owner == 0 or #network_name == 0 or network_name:find("#") then
        minetest.chat_send_player(caller_name, S("Incorrect parameter value"))
        return nil
    end

    return owner, network_name
end

local top_admin_command_name = top_command_name .. "_admin"

local admin_commands_list = {}

admin_commands_list["list_owners"] = {
    params = "",
    description = S("List players which own networks"),
    func = function(name, param)
        local networks_list = superquest.storage.get_networks_list()
        local res = S("Players") .. ":\n";

        for k, v in pairs(networks_list) do
            res = res .. "\t" .. k .. "\n"
        end

        minetest.chat_send_player(name, res);

        return true;
    end
}

admin_commands_list["list_player_networks"] = {
    params = "<player_name>",
    description = S("List networks of a player"),
    func = function(name, param)
        local networks_list = superquest.storage.get_networks_list()

        if param == nil then
            minetest.chat_send_player(name, S("Player is not specified"))
            return true
        end

        local player_name = param:gsub('^%s*(.-)%s*$', '%1')

        if networks_list[player_name] ~= nil and next(networks_list[player_name], nil) ~= nil then
            local res = S("Networks") .. ":\n"

            for k, _ in pairs(networks_list[player_name]) do
                res = res .. "\t" .. k .. "\n"
            end

            minetest.chat_send_player(name, res)
        else
            minetest.chat_send_player(name, S("No networks"))
        end

        return true
    end
}

admin_commands_list["list_player_network_elements"] = {
    params = "<player_name>#<network_name>",
    description = S("List elements of a network of a player"),
    func = function(name, param)
        local owner, network_name = parse_owner_network_name(name, param)
        if not owner or not network_name then
            return true
        end

        local network_obj = assert(superquest.Network(owner, network_name))
        local elements = network_obj:get_network_elements()

        minetest.chat_send_player(name, generate_network_description(network_name, elements))

        superquest.hud.show_waypoints(minetest.get_player_by_name(name), elements_to_waypoints(elements))

        return true
    end
}

admin_commands_list["fix_network"] = {
    params = "<player_name>#<network_name>",
    description = S("Scan the network and remove all the elements which don't match"),
    func = function(name, param)
        local owner, network_name = parse_owner_network_name(name, param)
        if not owner or not network_name then
            return true
        end

        local network_obj = assert(superquest.Network(owner, network_name))
        local elements = network_obj:get_network_elements()

        for _, pos_str in ipairs(elements.flags) do
            local pos = superquest.storage.key_to_coords(pos_str)
            local node = superquest.utils.get_node(pos)
            local meta = minetest.get_meta(pos)
            if not superquest.flag.is_flag(node.name) or meta:get_string("owner") ~= owner
                    or meta:get_string("network") ~= network_name then
                network_obj:remove_flag(pos)
                minetest.chat_send_player(name, S("Flag at") .. " " .. pos_str .. " " .. S("was removed from the network"))
            end
        end

        for _, pos_str in ipairs(elements.quest_machines) do
            local pos = superquest.storage.key_to_coords(pos_str)
            local node = superquest.utils.get_node(pos)
            local meta = minetest.get_meta(pos)
            if not superquest.quest_machine.is_quest_machine(node.name)
                    or meta:get_string("owner") ~= owner
                    or meta:get_string("network") ~= network_name then
                network_obj:remove_quest_machine(pos)
                minetest.chat_send_player(name, S("Quest machine at") .. " " .. pos_str .. " " .. S("was removed from the network"))
            end
        end
    end
}

local admin_command_description = S("Commands") .. ":\n"
for k, v in pairs(admin_commands_list) do
    admin_command_description = admin_command_description .. "/" .. top_admin_command_name .. " " .. k .. " " .. v.params .. "\n" .. v.description .. "\n\n"
end
admin_command_description = admin_command_description:gsub('^%s*(.-)%s*$', '%1')

minetest.register_chatcommand(top_admin_command_name, {
    params = "<command_name> [<parameters>]",
    description = admin_command_description,
    privs = { [superquest.privileges.admin_priv_name] = true },
    func = function(name, param)
        local command_name
        local subparam
        local space_index = param:find(" ")
        if space_index ~= nil then
            command_name = param:sub(1, space_index - 1)
            subparam = param:sub(space_index + 1, -1)
        else
            command_name = param
            subparam = nil
        end

        local command = admin_commands_list[command_name]
        if (command ~= nil) then
            return command.func(name, subparam)
        else
            minetest.chat_send_player(name, S("Unknown command") .. ": " .. command_name)
            return true
        end
    end
})
