local storage = minetest.get_mod_storage()

local latest_format_version = 3
local installed_format_version = storage:get_int("format_version")

if installed_format_version == 0 then
    -- The mod is newly installed, just set the latest format version and exit
    storage:set_int("format_version", latest_format_version)
    return
end

-- Make a backup of the previous version before migrating (just in case)
if installed_format_version < latest_format_version then
    local backups_dir = minetest.get_worldpath().."/superquest_mig_backups/"
    local data = storage:to_table()

    minetest.mkdir(backups_dir)

    local f = assert(io.open(backups_dir..tostring(installed_format_version), "w"))
    f:write(minetest.serialize(data))
    f:close()
end

-- Might happen in case of downgrading the mod's version
assert (installed_format_version <= latest_format_version,
        "The latest format version is older than the current format version."..
        "This might have happened in case of downgrading the mod's version."..
        "Upgrade the mod to the latest version to resolve the issue.")

if installed_format_version == 1 then
    local data = storage:to_table()

    for network_k, network_v in pairs(data.fields) do
        if network_k:match("^network#") then
            local network_data = minetest.deserialize(network_v)

            if network_data.flags then
                local flags_updated = {}
                for k_flags, v_flags in pairs(network_data.flags) do
                    local s, e = k_flags:find("[%d%-]+")
                    local x = tonumber(k_flags:sub(s, e))
                    s, e = k_flags:find("[%d%-]+", e + 2)
                    local y = tonumber(k_flags:sub(s, e))
                    s, e = k_flags:find("[%d%-]+", e + 2)
                    local z = tonumber(k_flags:sub(s, e))

                    flags_updated[vector.to_string(vector.new(x, y, z))] = v_flags
                end
                network_data.flags = flags_updated
            end

            if network_data.reward_boxes then
                local boxes_updated = {}
                for k_boxes, v_boxes in pairs(network_data.reward_boxes) do
                    local s, e = k_boxes:find("[%d%-]+")
                    local x = tonumber(k_boxes:sub(s, e))
                    s, e = k_boxes:find("[%d%-]+", e + 2)
                    local y = tonumber(k_boxes:sub(s, e))
                    s, e = k_boxes:find("[%d%-]+", e + 2)
                    local z = tonumber(k_boxes:sub(s, e))

                    boxes_updated[vector.to_string(vector.new(x, y, z))] = v_boxes
                end
                network_data.reward_boxes = boxes_updated
            end

            data.fields[network_k] = minetest.serialize(network_data)
        end
    end

    data.fields["format_version"] = 2

    storage:from_table(data)

    installed_format_version = 2
end

--[[
    Version 2 format:
    {
        [flags = {
            ["<x>,<y>,<z>"] = {
                players_reached = {
                    <player_name> = 1,
                    ...
                }
            },
            ...
        }],
        [reward_boxes = {
            ["<x>,<y>,<z>"] = {
                completions = {
                    [<player_name> = <count>],
                    ...
                }
            },
            ...
        }],
        [params = {
            [timed_duration = <seconds>],
            [teleport_after_compl = true]
        }],
        [user_mode = 1],
        [reward_boxes_count = <unused>],
    }
]]

if installed_format_version == 2 then
    local data = storage:to_table()

    for network_k, network_v in pairs(data.fields) do
        if network_k:match("^network#") then
            local network_data = minetest.deserialize(network_v)

            if network_data.flags then
                for flag_k, flag_v in pairs(network_data.flags) do
                    for player_k, player_v in pairs(flag_v.players_reached) do
                        flag_v.players_reached[player_k] = true
                    end
                end
            end

            if network_data.reward_boxes and next(network_data.reward_boxes) then
                local qm_key, qm_val = next(network_data.reward_boxes)

                network_data.quest_machine = qm_val
                network_data.quest_machine.coords = qm_key
                network_data.quest_machine.params = network_data.params
                network_data.quest_machine.user_mode = network_data.user_mode and true or nil
            end

            network_data.reward_boxes = nil
            network_data.params = nil
            network_data.user_mode = nil
            network_data.reward_boxes_count = nil

            if not network_data.flags and not network_data.quest_machine then
                data.fields[network_k] = nil
            else
                data.fields[network_k] = minetest.serialize(network_data)
            end
        end
    end

    data.fields["format_version"] = 3

    storage:from_table(data)

    installed_format_version = 3
end

--[[
    Version 3 format:
    {
        [flags = {
            ["<x>,<y>,<z>"] = {
                players_reached = {
                    <player_name> = true,
                    ...
                }
            },
            ...
        }],
        [quest_machine = {
            completions = {
                [<player_name> = <count>],
                ...
            },
            coords = "<x>,<y>,<z>",
            [params = {
                [timed_duration = <seconds>],
                [teleport_after_compl = true]
            }],
            [user_mode = 1],
        }],
    }
]]

--[[
    Place here the code responsible for migration to next data formats.
    Example:


    if installed_format_version == 1 then
        -- Upgrade to 2
        storage:set_int("format_version", 2)
    end

    if installed_format_version == 2 then
        -- Upgrade to 3
        storage:set_int("format_version", 3)
    end

    if installed_format_version == 3 then
        -- Upgrade to 5 because version 4 is broken
        storage:set_int("format_version", 5)
    end

    if installed_format_version == 4 then
        -- Upgrade to 5 and try to recover
        storage:set_int("format_version", 5)
    end


    Never use `latest_format_version` in these `if` conditions otherwise it might cause bugs at next upgrades
--]]
