local S = superquest.S

superquest.quest_machine.logic = {}

local function calc_avail_rewards_for_inventory(rewards_inv, quest_machine_inventory)
    local available_rewards
    for k, v in pairs(rewards_inv) do
        if quest_machine_inventory[k] ~= nil then
            local avail_cur_item_times = quest_machine_inventory[k] / v
            if available_rewards == nil or avail_cur_item_times < available_rewards then
                available_rewards = avail_cur_item_times
            end
        else
            available_rewards = 0
            break
        end
    end

    if available_rewards == nil then
        available_rewards = 0
    end

    return math.floor(available_rewards)
end

local function calc_items_in_inv_list(inv_list)
    local items_count = {}
    for _, stack_v in pairs(inv_list) do
        if stack_v:get_count() ~= 0 then
            if items_count[stack_v:get_name()] == nil then
                items_count[stack_v:get_name()] = 0
            end
            items_count[stack_v:get_name()] = items_count[stack_v:get_name()] + stack_v:get_count()
        end
    end

    return items_count
end

local function get_number_of_avail_rewards(inv_lists)
    local items_count_per_inv = {}
    for list_k, list_v in pairs(inv_lists) do
        items_count_per_inv[list_k] = calc_items_in_inv_list(list_v)
    end

    return {
        first_compl_rewards = calc_avail_rewards_for_inventory(items_count_per_inv["first_compl_rewards"], items_count_per_inv["reward_box_storage"]),
        further_compl_rewards = calc_avail_rewards_for_inventory(items_count_per_inv["further_compl_rewards"], items_count_per_inv["reward_box_storage"]),
    }
end

local function is_enough_place_for_reward(reward_inv_list, user_inv_list)
    local items_count = calc_items_in_inv_list(reward_inv_list)

    local empty_cells_in_user_inv = 0
    local avail_for_items = {}

    for _, stack in pairs(user_inv_list) do
        if stack:get_count() == 0 then
            empty_cells_in_user_inv = empty_cells_in_user_inv + 1
        elseif items_count[stack:get_name()] ~= nil and stack:get_count() < stack:get_stack_max() then
            if avail_for_items[stack:get_name()] == nil then
                avail_for_items[stack:get_name()] = 0
            end
            avail_for_items[stack:get_name()] = avail_for_items[stack:get_name()] + stack:get_stack_max() - stack:get_count()
        end
    end

    for k, v in pairs(items_count) do
        local cur_items_remaining = v
        if avail_for_items[k] ~= nil then
            cur_items_remaining = cur_items_remaining - math.min(cur_items_remaining, avail_for_items[k])
        end

        local max_stack_size = ItemStack(k):get_stack_max()
        if empty_cells_in_user_inv < cur_items_remaining / max_stack_size then
            return false
        end

        empty_cells_in_user_inv = empty_cells_in_user_inv - math.ceil(cur_items_remaining / max_stack_size)

        -- We don't need to update avail_for_items for the current item here since we won't need it anymore
    end

    return true
end


superquest.quest_machine.logic.show_formspec_user_or_owner = function(owner, network_name, pos, player_name, max_completions, inv_lists, params)
    local reached_flags = 0
    local total_flags = 0
    local network_obj = superquest.Network(owner, network_name)
    if network_obj then
        reached_flags, total_flags = network_obj:get_flags_stats_for_player(player_name)
    end

    local available_rewards = get_number_of_avail_rewards(inv_lists)

    if (owner == player_name or superquest.privileges.can_edit_all(player_name)) and (not network_obj or not network_obj:is_in_user_mode()) then
        superquest.quest_machine.formspec.owner.show(player_name, pos, {
            network = network_name,
            owner = owner,
            max_completions = max_completions,
            available_rewards = available_rewards,
            total_flags = total_flags,
            timed_duration = params.timed_duration,
            teleport_after_compl = params.teleport_after_compl,
            can_edit_owner = superquest.privileges.can_edit_all(player_name),
        })
    else
        local completions = network_obj and network_obj:get_completions_for_player(player_name) or 0
        superquest.quest_machine.formspec.user.show(player_name, pos, {
            owner = owner,
            network = network_name,
            owner_mode_avail = owner == player_name or superquest.privileges.can_edit_all(player_name),
            max_completions = max_completions,
            available_rewards = available_rewards,
            timed_duration = params.timed_duration,
            total_flags = total_flags,
            completions = completions,
            reached_flags = reached_flags
        })
    end
end

superquest.quest_machine.logic.set_mode = function(player, pos, mode)
    local player_name = player:get_player_name()
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    if owner ~= player_name and not superquest.privileges.can_edit_all(player_name) then
        return
    end

    local network_obj = superquest.Network(owner, network_name)
    if not network_obj then
        return
    end

    if network_obj:get_quest_machine_coords() ~= pos then
        -- The quest machine doesn't correspond to the quest machine in storage for this network
        core.chat_send_player(player_name, S("The Quest Machine wasn't assigned to the specified network, resetting..."))
        meta:set_string("owner", "")
        meta:set_string("network", "")
        return
    end

    network_obj:set_mode(mode)

    superquest.quest_machine.logic.show_formspec_user_or_owner(owner, network_name, pos, player_name, meta:get_int("max_completions"), meta:get_inventory():get_lists(), {
        timed_duration = meta:get_int("timed_duration"),
        teleport_after_compl = meta:contains("teleport_after_compl")
    })
end

superquest.quest_machine.logic.show_completions = function(player, pos)
    local player_name = player:get_player_name()
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    local network_obj = superquest.Network(owner, network_name)
    if not network_obj then
        return
    end
    local completions = network_obj:get_all_completions(pos)

    superquest.quest_machine.formspec.completions.show(player_name, pos, completions, player_name == owner)
end

superquest.quest_machine.logic.reset_completions = function(player, pos)
    local player_name = player:get_player_name()
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    if owner == "" or network_name == "" then
        return
    end

    if owner ~= player_name and not superquest.privileges.can_edit_all(player_name) then
        return
    end

    local network_obj = assert(superquest.Network(owner, network_name))
    network_obj:reset_all_completions(pos)
end

superquest.quest_machine.logic.start_quest = function(player, pos)
    local player_name = player:get_player_name()
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    if superquest.active_timed_quests:get_active_network_for_player(player_name) then
        core.chat_send_player(player_name, S("You already have an active quest. Type '/superquest cancel' to cancel."))
        return
    end

    local network_obj = superquest.Network(owner, network_name)
    if not network_obj then
        return
    end

    local reached_flags, total_flags = network_obj:get_flags_stats_for_player(player_name)
    if reached_flags >= total_flags then
        core.chat_send_player(player_name, S("The quest is already completed. Take your reward first."))
        return
    end

    superquest.active_timed_quests:add(player_name, owner, network_name, meta:get_int("timed_duration"))
end

superquest.quest_machine.logic.save_data = function(player, pos, data)
    local player_name = player:get_player_name()
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local old_network_name = meta:get_string("network")

    if data.new_owner then
        if not superquest.privileges.can_edit_all(player_name) then
            return
        end
    else
        data.new_owner = owner
    end

    local network_obj = superquest.Network(data.new_owner, data.new_network_name)

    if old_network_name == data.new_network_name and owner == data.new_owner
            and network_obj and network_obj:get_quest_machine_coords() ~= pos then
        -- The quest machine doesn't correspond to the quest machine in storage for this network
        core.chat_send_player(player_name, S("The Quest Machine wasn't assigned to the specified network, resetting..."))
        meta:set_string("owner", "")
        meta:set_string("network", "")
        return
    end

    local old_timed_duration = nil
    if meta:contains("timed_duration") then
        old_timed_duration = meta:get_int("timed_duration")
    end

    if old_network_name ~= data.new_network_name or owner ~= data.new_owner then
        if network_obj then
            if network_obj:get_quest_machine_coords() then
                core.chat_send_player(player_name, S("The network already contains a quest machine"))
                return
            end

            if data.timed_duration then
                network_obj:clear_reached_flags_for_all()
            end
        end

        local old_network_obj = superquest.Network(owner, old_network_name)
        if old_network_obj then
            old_network_obj:remove_quest_machine(pos)

            if old_timed_duration then
                superquest.active_timed_quests:stop_and_remove_all_timers_for_quest(owner, old_network_name)
            end
        end
    elseif network_obj and owner == data.new_owner and old_network_name == data.new_network_name then
        if old_timed_duration and not data.timed_duration then
            superquest.active_timed_quests:stop_and_remove_all_timers_for_quest(owner, old_network_name)
        elseif not old_timed_duration and data.timed_duration then
            network_obj:clear_reached_flags_for_all()
        elseif old_timed_duration and data.timed_duration and old_timed_duration ~= data.timed_duration then
            superquest.active_timed_quests:adjust_timers_for_quest(owner, old_network_name, data.timed_duration - old_timed_duration)
        end
    end

    meta:set_string("owner", data.new_owner)
    meta:set_string("network", data.new_network_name)
    meta:set_int("max_completions", data.max_completions)

    if data.timed_duration then
        meta:set_int("timed_duration", data.timed_duration)
    else
        meta:set_string("timed_duration", "")
    end

    if superquest.config.teleportation then
        if data.teleport_after_compl then
            meta:set_int("teleport_after_compl", 1)
        else
            meta:set_string("teleport_after_compl", "")
        end
    end

    if network_obj then
        if old_network_name ~= data.new_network_name or owner ~= data.new_owner then
            network_obj:add_quest_machine(pos, {
                timed_duration = data.timed_duration,
                teleport_after_compl = data.teleport_after_compl,
            })
        else
            network_obj:set_timed_duration(data.timed_duration)
            if superquest.config.teleportation then
                network_obj:set_teleport_after_compl(data.teleport_after_compl)
            end
        end
    end
end

superquest.quest_machine.logic.remove_quest_machine_data = function(pos)
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    local network_obj = superquest.Network(owner, network_name)
    if not network_obj then
        return
    end

    network_obj:remove_quest_machine(pos)

    if meta:get_string("timed_duration") then
        superquest.active_timed_quests:stop_and_remove_all_timers_for_quest(owner, network_name)
    end
end

superquest.quest_machine.logic.try_to_receive_reward = function(player, pos)
    local meta = core.get_meta(pos)

    local player_name = player:get_player_name()

    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    if owner == "" or network_name == "" then
        core.chat_send_player(player_name, S("The Quest Machine is not configured!"))
        return false
    end

    local network_obj = assert(superquest.Network(owner, network_name, false))

    if network_obj:get_quest_machine_coords() ~= pos then
        -- The quest machine doesn't correspond to the quest machine in storage for this network
        core.chat_send_player(player_name, S("The Quest Machine wasn't assigned to the specified network, resetting..."))
        meta:set_string("owner", "")
        meta:set_string("network", "")
        return false
    end

    if (owner == player_name or superquest.privileges.can_edit_all(player_name))
            and not network_obj:is_in_user_mode() then
        return false
    end

    local max_completions = meta:get_int("max_completions")

    local completions = network_obj:get_completions_for_player(player_name)
    if max_completions ~= 0 and completions >= max_completions then
        return false
    end

    local punched_flags, total_flags = network_obj:get_flags_stats_for_player(player_name)
    if punched_flags ~= total_flags then
        return false
    end

    local node_inv = meta:get_inventory()
    local player_inv = player:get_inventory()

    local node_inv_lists = node_inv:get_lists()

    local avail_rewards_total = get_number_of_avail_rewards(node_inv_lists)

    local rewards_list;
    local avail_rewards;

    if completions == 0 then
        rewards_list = node_inv:get_list("first_compl_rewards")
        avail_rewards = avail_rewards_total.first_compl_rewards
    else
        rewards_list = node_inv:get_list("further_compl_rewards")
        avail_rewards = avail_rewards_total.further_compl_rewards
    end

    if avail_rewards == 0 then
        core.chat_send_player(player_name, S("Not enought items for reward!"))
        return false
    end

    local is_enough_place_in_inv = is_enough_place_for_reward(rewards_list, player_inv:get_list("main"))
    if not is_enough_place_in_inv then
        core.chat_send_player(player_name, S("Not enough space in your inventory!"))
        return false
    end

    network_obj:clear_reached_flags_for_player(player_name)
    network_obj:set_completions_for_player(player_name, completions + 1)
    network_obj:save()

    for _, v in pairs(rewards_list) do
        player_inv:add_item("main", v)
        node_inv:remove_item("reward_box_storage", v)
    end

    if superquest.config.teleportation then
        superquest.teleportation.reset_tp_destination(player_name)
    end

    return true
end
