-- Description of network data's format can be found in src/migrations.lua

local NetworkBase = {}

local function get_empty_flags_table()
    return {
        players_reached = {}
    }
end

NetworkBase.save = function(self, ...)
    superquest.storage.set_network_data(self._owner_name, self._network_name, self._network_data)
    return ...
end

NetworkBase.add_flag = function(self, pos)
    self._network_data.flags = self._network_data.flags or {}
    self._network_data.flags[superquest.storage.coords_to_key(pos)] = get_empty_flags_table()
end

NetworkBase.remove_flag = function(self, pos)
    if not self._network_data.flags then
        return
    end

    self._network_data.flags[superquest.storage.coords_to_key(pos)] = nil
    if not next(self._network_data.flags, nil) then
        self._network_data.flags = nil
    end
end

NetworkBase.add_player_reached_flag = function(self, pos, player_name)
    local key = superquest.storage.coords_to_key(pos)
    self._network_data.flags = self._network_data.flags or {}
    self._network_data.flags[key] = self._network_data.flags[key] or get_empty_flags_table()
    if self._network_data.flags[key].players_reached[player_name] then
        -- The flag was already punched
        return false
    end
    self._network_data.flags[key].players_reached[player_name] = true
    return true
end

NetworkBase.clear_reached_flags_for_player = function(self, player_name)
    if self._network_data.flags then
        for _, v in pairs(self._network_data.flags) do
            v.players_reached[player_name] = nil
        end
    end
end

NetworkBase.clear_reached_flags_for_all = function(self)
    if self._network_data.flags then
        for _, v in pairs(self._network_data.flags) do
            v.players_reached = {}
        end
    end
end

NetworkBase.get_flags_stats_for_player = function(self, player_name)
    local reached = 0
    local total = 0

    if self._network_data.flags then
        for _, v in pairs(self._network_data.flags) do
            if v.players_reached[player_name] then
                reached = reached + 1
            end
            total = total + 1
        end
    end
    return reached, total
end

NetworkBase.add_quest_machine = function(self, pos, params)
    assert(not self._network_data.quest_machine)

    self._network_data.quest_machine = {
        completions = {},
        coords = superquest.storage.coords_to_key(pos)
    }

    self._network_data.quest_machine.params = {}
    if params.timed_duration ~= 0 then
        self._network_data.quest_machine.params.timed_duration = params.timed_duration
    end
    self._network_data.quest_machine.params.teleport_after_compl = params.teleport_after_compl or nil

    if not next(self._network_data.quest_machine.params) then
        self._network_data.quest_machine.params = nil
    end
end

NetworkBase.remove_quest_machine = function(self, pos)
    if not self._network_data.quest_machine
            and self._network_data.quest_machine.coords ~= superquest.storage.coords_to_key(pos) then
        return
    end

    self._network_data.quest_machine = nil
end

NetworkBase.get_quest_machine_coords = function(self)
    if self._network_data.quest_machine then
        return superquest.storage.key_to_coords(self._network_data.quest_machine.coords)
    else
        return nil
    end
end

NetworkBase.get_timed_duration = function(self)
    if self._network_data.quest_machine and self._network_data.quest_machine.params then
        return self._network_data.quest_machine.params.timed_duration or 0
    else
        return 0
    end
end

NetworkBase.set_timed_duration = function(self, timed_duration)
    if not self._network_data.quest_machine then
        return
    end

    self._network_data.quest_machine.params = self._network_data.quest_machine.params or {}
    if timed_duration == 0 then
        timed_duration = nil
    end

    self._network_data.quest_machine.params.timed_duration = timed_duration

    if not next(self._network_data.quest_machine.params) then
        self._network_data.quest_machine.params = nil
    end
end

NetworkBase.get_teleport_after_compl = function(self)
    if self._network_data.quest_machine and self._network_data.quest_machine.params then
        return self._network_data.quest_machine.params.teleport_after_compl
    else
        return false
    end
end

NetworkBase.set_teleport_after_compl = function(self, val)
    if not self._network_data.quest_machine then
        return
    end

    self._network_data.quest_machine.params = self._network_data.quest_machine.params or {}

    self._network_data.quest_machine.params.teleport_after_compl = val and true or nil

    if not next(self._network_data.quest_machine.params) then
        self._network_data.quest_machine.params = nil
    end
end

NetworkBase.get_completions_for_player = function(self, player_name)
    if not self._network_data.quest_machine then
        return 0
    end

    return self._network_data.quest_machine.completions[player_name] or 0
end

NetworkBase.set_completions_for_player = function(self, player_name, completions)
    if not self._network_data.quest_machine then
        return
    end

    self._network_data.quest_machine.completions[player_name] = completions
end

NetworkBase.get_all_completions = function(self)
    local completions = {}
    if self._network_data.quest_machine then
        for k, v in pairs(self._network_data.quest_machine.completions) do
            completions[k] = v
        end
    end
    return completions
end

NetworkBase.reset_all_completions = function(self)
    if self._network_data.quest_machine then
        self._network_data.quest_machine.completions = {}
    end
end

NetworkBase.is_in_user_mode = function(self)
    return self._network_data.quest_machine ~= nil and self._network_data.quest_machine.user_mode ~= nil
end

NetworkBase.set_mode = function(self, is_user_mode)
    if self._network_data.quest_machine then
        self._network_data.quest_machine.user_mode = is_user_mode and true or nil
    end
end

NetworkBase.get_network_elements = function(self)
    local elements = {
        flags = {},
        quest_machines = {}
    }

    if self._network_data.flags then
        for k, _ in pairs(self._network_data.flags) do
            table.insert(elements.flags, k)
        end
    end

    if self._network_data.quest_machine then
        table.insert(elements.quest_machines, self._network_data.quest_machine.coords)
    end

    return elements
end

local no_save_after = {
    [NetworkBase.save] = true,
    [NetworkBase.get_flags_stats_for_player] = true,
    [NetworkBase.get_quest_machine_coords] = true,
    [NetworkBase.get_timed_duration] = true,
    [NetworkBase.get_teleport_after_compl] = true,
    [NetworkBase.get_completions_for_player] = true,
    [NetworkBase.get_all_completions] = true,
    [NetworkBase.is_in_user_mode] = true,
    [NetworkBase.get_network_elements] = true
}

NetworkBase.__index = function(self, key)
    return function(...)
        local func = NetworkBase[key]
        if self._auto_save and not no_save_after[func] then
            return NetworkBase.save(self, func(...))
        else
            return func(...)
        end
    end
end

superquest.Network = function(owner_name, network_name, auto_save)
    if owner_name == "" or network_name == "" then
        return nil
    end

    local data = {
        _owner_name = owner_name,
        _network_name = network_name,
        _network_data = superquest.storage.get_network_data(owner_name, network_name) or {},
        _auto_save = not (auto_save == false),
    }
    setmetatable(data, NetworkBase)
    return data
end
