--[[
    Active timed quest object internal structure:
    {
        [<player_name>] = {
            -- Owner of the quest
            owner = <string>
            -- Network name of the quest
            network = <string>
            -- Timer of the current active quest
            timer = <superquest.Timer>
            -- Time of last logout of the player since the start of the quest
            last_logout = <number>
        }
    }
]]

local expiration_period = superquest.config.timed_quest_expir_after_logout * 3600
-- Might not contain the relevant value but it's not crucial
local closest_expire = nil

local Base = {}

Base.add = function(self, player_name, owner_name, network_name, seconds)
    local timer = superquest.Timer(player_name, seconds)
    superquest.timer.start(timer)

    self._obj[player_name] = {
        owner = owner_name,
        network = network_name,
        timer = timer
    }

    superquest.storage.set_active_timed_quests(self._obj)
end

Base.get_active_network_for_player = function(self, player_name)
    local active_quest = self._obj[player_name]
    if not active_quest then
        return nil
    end
    return {
        owner = active_quest.owner,
        network = active_quest.network
    }
end

Base.stop_and_remove = function(self, player_name)
    local active_quest = self._obj[player_name]
    if not active_quest then
        return
    end

    superquest.timer.stop(active_quest.timer)
    self._obj[player_name] = nil

    superquest.storage.set_active_timed_quests(self._obj)
end

Base.stop_and_remove_all_timers_for_quest = function(self, owner_name, network_name)
    for player_name, quest in pairs(self._obj) do
        if quest.owner == owner_name and quest.network == network_name then
            superquest.timer.stop(quest.timer)
            self._obj[player_name] = nil
        end
    end

    superquest.storage.set_active_timed_quests(self._obj)
end

Base.adjust_timers_for_quest = function(self, owner_name, network_name, diff)
    for _, quest in pairs(self._obj) do
        if quest.owner == owner_name and quest.network == network_name then
            superquest.timer.adjust(quest.timer, diff)
        end
    end

    superquest.storage.set_active_timed_quests(self._obj)
end

Base.expire_current_for_player = function(self, player_name)
    local active_quest = self._obj[player_name]
    if not active_quest then
        return
    end

    superquest.timer.expire(active_quest.timer)

    superquest.storage.set_active_timed_quests(self._obj)
end

Base.__index = Base

local _active_timed_quests = superquest.storage.get_active_timed_quests()

superquest.active_timed_quests = {
    _obj = _active_timed_quests
}
setmetatable(superquest.active_timed_quests, Base)

local cleanup_max_interval = 3600

local function get_time_to_call_cleanup(gametime)
    local expires_in = closest_expire - gametime
    if expires_in <= 0 then
        expires_in = 0.1
    elseif expires_in > cleanup_max_interval then
        expires_in = cleanup_max_interval
    end

    return expires_in
end

local function check_and_cleanup()
    local cur_gametime = core.get_gametime()

    if closest_expire and cur_gametime >= closest_expire then
        closest_expire = nil

        for _, v in pairs(_active_timed_quests) do
            if v.last_logout then
                if v.last_logout + expiration_period <= cur_gametime then
                    superquest.timer.expire(v.timer)
                else
                    if not closest_expire or v.last_logout + expiration_period < closest_expire then
                        closest_expire = v.last_logout + expiration_period
                    end
                end
            end
        end
    end

    if closest_expire then
        local expires_in = get_time_to_call_cleanup(cur_gametime)
        core.after(expires_in, check_and_cleanup)
    end
end

local updated = false

local function try_to_update_timers_after_load(must_succeed)
    if updated then
        return
    end

    local gametime = core.get_gametime()
    if not gametime then
        assert(not must_succeed)
        -- Retry after 1s if gametime is not available yet
        core.after(1, try_to_update_timers_after_load)

        return
    end

    if next(_active_timed_quests) then
        closest_expire = gametime + expiration_period

        for _, quest in pairs(_active_timed_quests) do
            superquest.timer.stop(quest.timer)
            quest.last_logout = quest.last_logout or gametime
            local expires_in = quest.last_logout + expiration_period
            if expires_in < closest_expire then
                closest_expire = expires_in
            end
        end

        superquest.storage.set_active_timed_quests(_active_timed_quests)

        if closest_expire then
            local expires_in = get_time_to_call_cleanup(gametime)
            core.after(expires_in, check_and_cleanup)
        end
    else
        closest_expire = nil
    end

    updated = true
end

try_to_update_timers_after_load()

core.register_on_joinplayer(function(player)
    if not updated then
        try_to_update_timers_after_load(true)
    end

    local quest = _active_timed_quests[player:get_player_name()]
    if not quest then
        return
    end

    quest.last_logout = nil
    superquest.timer.start(quest.timer)
    superquest.storage.set_active_timed_quests(_active_timed_quests)
end)

core.register_on_leaveplayer(function(player)
    local player_name = player:get_player_name()
    if not _active_timed_quests[player_name] then
        return
    end

    local timer = _active_timed_quests[player_name].timer
    superquest.timer.stop(timer)
    local cur_gametime = core.get_gametime()
    _active_timed_quests[player_name].last_logout = cur_gametime
    if not closest_expire then
        closest_expire = expiration_period
        local expires_in = get_time_to_call_cleanup(cur_gametime)
        core.after(expires_in, check_and_cleanup)
    end
    superquest.storage.set_active_timed_quests(_active_timed_quests)
end)
