local S = superquest.S

local form_name = core.get_current_modname()..":flag_settings"
local node_positions = {}

local autotr_coords_field_hint = S("Should contain positive integer values of distance of x,y,z, relative to the position of the flag, separated by commas") .. ".\n" ..
                                 S("The values in the first field are subtracted from the flag's position while the values in the second field are added to it") .. ".\n" ..
                                 S("This flag will be automatically triggered for players detected within the cube described by the coordinates") .. "."

local teleport_coords_field_hint = S("Should contain integer values of distance of x,y,z, relative to the position of the flag, separated by commas") .. ".\n" ..
                                   S("The value can be empty that means that teleportation is disabled for the flag") .. "."

local function get_formspec(owner, network, from_dev, to_dev, teleport_coords, digiline_channel, can_edit_owner)
    local height_cur = 0.5
    local formspec_data = ""

    if can_edit_owner then
        formspec_data = formspec_data .. "field[0.5,"..tostring(height_cur)..";5.0,0.8;owner;" .. S("Owner") .. ";"..core.formspec_escape(owner).."]"
        height_cur = height_cur + 1.3
    else
        formspec_data = formspec_data .. "label[0.5,"..tostring(height_cur)..";"..core.formspec_escape(S("Owner") .. ": "..owner).."]"
        height_cur = height_cur + 0.8
    end

    formspec_data = formspec_data ..
        "field[0.5,"..tostring(height_cur)..";5.0,0.8;network;" .. S("Network") .. ";"..core.formspec_escape(network).."]"
    height_cur = height_cur + 1.3

    if superquest.config.flags_autotrigger then
        formspec_data = formspec_data .. "field[0.5,"..tostring(height_cur)..";2.2,0.8;from_coords;" .. S("From coords") .. ";"..core.formspec_escape(from_dev).."]" ..
        "field[3.3,"..tostring(height_cur)..";2.2,0.8;to_coords;" .. S("To coords") .. ";"..core.formspec_escape(to_dev).."]" ..
        "tooltip[0.5,"..tostring(height_cur)..";5,0.8;"..autotr_coords_field_hint.."]"
        height_cur = height_cur + 1.2
    end

    if superquest.config.teleportation then
        formspec_data = formspec_data ..
            "field[0.5,"..tostring(height_cur)..";5,0.8;teleport_to;" .. S("Teleportation coords (relative)") .. ";"..core.formspec_escape(teleport_coords).."]" ..
            "tooltip[0.5,"..tostring(height_cur)..";5,0.8;"..teleport_coords_field_hint.."]"
        height_cur = height_cur + 1.2
    end

    if digilines then
        formspec_data = formspec_data ..
            "field[0.5,"..tostring(height_cur)..";4.2,0.8;digiline_channel;"..S("Digiline channel")..";"..core.formspec_escape(digiline_channel).."]" ..
            "button[4.7,"..tostring(height_cur)..";0.8,0.8;digiline_guide;?]"
        height_cur = height_cur + 1.2
    end

    formspec_data = formspec_data ..
        "button_exit[0.5,"..tostring(height_cur)..";5.0,0.8;save;" .. S("Save") .. "]"
    height_cur = height_cur + 1

    formspec_data = "formspec_version[4]" ..
        "size[6,"..tostring(height_cur).."]" ..
        formspec_data

    return formspec_data
end

local function get_network_index(owner, network)
    return owner .. "#" .. network
end

local get_flag_index = superquest.storage.coords_to_key

-- For digiline-requiring flags
local allowed_players = {}

local function allowed_players_add(owner, network, pos, player_name)
    local network_index = get_network_index(owner, network)
    local flag_index = get_flag_index(pos)

    if not allowed_players[network_index] then
        allowed_players[network_index] = {}
    end

    if not allowed_players[network_index][flag_index] then
        allowed_players[network_index][flag_index] = {}
    end

    local allowed_players_flag = allowed_players[network_index][flag_index]

    for _, v in ipairs(allowed_players_flag) do
        if v == player_name then
            return
        end
    end

    table.insert(allowed_players_flag, player_name)
    if #allowed_players_flag > 10 then
        table.remove(allowed_players_flag, 1)
    end
end

local function _allowed_players_remove_by_indices(network_index, flag_index, player_index)
    table.remove(allowed_players[network_index][flag_index], player_index)
    if #allowed_players[network_index][flag_index] == 0 then
        allowed_players[network_index][flag_index] = nil
        if not next(allowed_players[network_index]) then
            allowed_players[network_index] = nil
        end
    end
end

local function allowed_players_remove_for_flag(owner, network, pos, player_name)
    local index = 0
    local network_index = get_network_index(owner, network)
    local flag_index = get_flag_index(pos)
    if allowed_players[network_index] and allowed_players[network_index][flag_index] then
        for k, v in ipairs(allowed_players[network_index][flag_index]) do
            if v == player_name then
                index = k
                break
            end
        end
    end

    if index ~= 0 then
        _allowed_players_remove_by_indices(network_index, flag_index, index)

        return true
    else
        return false
    end
end

local function allowed_players_remove_for_all(owner, network, player_name)
    local network_index = get_network_index(owner, network)

    if allowed_players[network_index] then
        for flag_index, flag_players in pairs(allowed_players[network_index]) do
            for i_player, v_player in ipairs(flag_players) do
                if player_name == v_player then
                    _allowed_players_remove_by_indices(network_index, flag_index, i_player)
                    break
                end
            end
        end
    end
end

local function try_to_count_flag(pos, player, print_to_chat, is_digiline_requiring)
    local meta = core.get_meta(pos)

    local owner = meta:get_string("owner")
    local network = meta:get_string("network")

    local player_name = player:get_player_name()

    if owner == "" or network == "" then
        if print_to_chat then
            core.chat_send_player(player_name, S("This flag is not configured") .. "!")
        end

        return
    end

    local network_obj = assert(superquest.Network(owner, network))

    if (owner == player_name or superquest.privileges.can_edit_all(player_name))
            and not network_obj:is_in_user_mode() then
        if print_to_chat then
            core.chat_send_player(player_name, "You need to switch to the User mode to participate")
        end
        return
    end

    local timed_duration = network_obj:get_timed_duration()

    if timed_duration ~= 0 then
        local active_network = superquest.active_timed_quests:get_active_network_for_player(player_name)
        if not active_network or active_network.owner ~= owner or active_network.network ~= network then
            if print_to_chat then
                core.chat_send_player(player_name, S("This flag belongs to a timed quest. You need to start it first in the corresponding quest machine."))
            end
            return
        end
    end

    local max_completions = network_obj:get_max_completions()

    if max_completions ~= 0 and network_obj:get_completions_for_player(player_name) >= max_completions then
        if print_to_chat then
            core.chat_send_player(player_name, S("You already reached the maximum number of completions"))
        end
        return
    end

    local is_counted = network_obj:is_flag_counted(pos, player_name)
    if is_counted then
        if print_to_chat then
            core.chat_send_player(player_name, S("This flag is already counted") .. "!")
        end
        return
    end

    if is_digiline_requiring then
        local removed = allowed_players_remove_for_flag(owner, network, pos, player_name)
        if not removed then
            if print_to_chat then
                core.chat_send_player(player_name, S("You cannot trigger this flag without the corresponding digiline signal.").." "..
                        S("Receiving this signal depends on the quest itself. Ask the quest's owner if it's not obvious how to get it."))
            end
            return
        end
    end

    network_obj:add_player_reached_flag(pos, player_name)

    local reached, total = network_obj:get_flags_stats_for_player(player_name)
    if reached >= total then
        if timed_duration then
            superquest.active_timed_quests:stop_and_remove(player_name)
        end

        if superquest.config.teleportation then
            if network_obj:get_teleport_after_compl() then
                superquest.teleportation.set_tp_dest_to_quest_machine(player_name, owner, network, false)
                core.chat_send_player(player_name, S("Quest completed").."! "..S("You can type '/superquest teleport' to teleport to the Quest Machine.")..
                        " "..S("Punch the Quest Machine to receive the reward."))
            end
        end
    elseif superquest.config.teleportation then
        superquest.teleportation.set_tp_dest_to_flag_if_possible(player_name, owner, network, pos)
    end

    local channel = meta:get_string("digiline_channel")
    if digilines and channel ~= "" then
        superquest.digilines.send.flag_triggered(pos, channel, player_name, reached, total)
    end

    local maxpos_deviation_table = {
        [0] = vector.new(0, 0.5, 0),
              vector.new(0, 0, 0.5),
              vector.new(0, 0, -0.5),
              vector.new(0.5, 0, 0),
              vector.new(-0.5, 0, 0),
              vector.new(0, -0.5, 0),
    }
    local deviation = maxpos_deviation_table[math.floor(core.get_node(pos).param2 / 4)]

    core.add_particlespawner({
        amount = 50,
        time = 0.01,
        minpos = vector.add(pos, deviation * 2),
        maxpos = vector.add(pos, deviation * 3),
        minvel = vector.new(-2, -2, -2),
        maxvel = vector.new(2, 2, 2),
        minexptime = 0.3,
        maxexptime = 0.3,
        minsize = 0.4,
        maxsize = 0.4,
        texture = is_digiline_requiring and "superquest_particle_blue.png" or "superquest_particle_red.png"
    })

    superquest.hud.show_popup(player, S("Flags counted") .. ": "..reached.." / "..total)
end

local function flag_after_place(pos, placer, itemstack)
    local meta = core.get_meta(pos)
    if placer and placer:is_player() then
        meta:set_string("owner", placer:get_player_name())
    end
end

local function flag_on_destruct(pos)
    local meta = core.get_meta(pos)

        local network_obj = superquest.Network(meta:get_string("owner"), meta:get_string("network"))
        if not network_obj then
            return
        end
        network_obj:remove_flag(pos)

        -- We don't need to check the player because flags should be placed in protected areas anyway
end

local function flag_on_place(itemstack, placer, pointed_thing)
    local dir = core.dir_to_wallmounted(
        vector.subtract(pointed_thing.under, pointed_thing.above)
    )
    local facedir_table = {[0] = 20, 0, 17, 15, 8, 6}
    if dir == 0 then
        facedir_table[dir] = facedir_table[dir] + (4 - core.dir_to_facedir(placer:get_look_dir())) % 4
    elseif dir == 1 then
        facedir_table[dir] = facedir_table[dir] + core.dir_to_facedir(placer:get_look_dir())
    end
    core.item_place(itemstack, placer, pointed_thing, facedir_table[dir])
    return itemstack
end

local function show_menu(pos, player)
    if not (player and player:is_player()) then
        return
    end

    local meta = core.get_meta(pos)

    local player_name = player:get_player_name()

    if not (meta:get_string("owner") == player_name or superquest.privileges.can_edit_all(player_name)) then
        return
    end

    node_positions[player_name] = pos

    local from_v = meta:get_int("scan_from_x")..","..meta:get_int("scan_from_y")..","..meta:get_int("scan_from_z")
    local to_v = meta:get_int("scan_to_x")..","..meta:get_int("scan_to_y")..","..meta:get_int("scan_to_z")

    core.show_formspec(player_name, form_name,
        get_formspec(meta:get_string("owner"), meta:get_string("network"), from_v, to_v, meta:get_string("teleport_to"),
        meta:get_string("digiline_channel"), superquest.privileges.can_edit_all(player_name)))
end

core.register_node("superquest:flag", {
    description = S("Flag"),
    drawtype = "mesh",
    mesh = "superquest_flag.obj",
    tiles = { "superquest_flag.png" },
    inventory_image = "superquest_flag.png",
    selection_box = {
        type = "fixed",
        fixed = { -0.1, -0.5, 0.1, 0.1, 1.5, -0.5 }
    },
    paramtype = "light",
    paramtype2 = "facedir",
    sunlight_propagates = true,
    use_texture_alpha = "clip",
    walkable = false,
    groups = { cracky=1, oddly_breakable_by_hand = 2 },

    digilines = {
        receptor = {}
    },

    after_place_node = flag_after_place,

    on_rightclick = function(pos, node, clicker, itemstack, pointed_thing)
        show_menu(pos, clicker)
    end,

    on_destruct = flag_on_destruct,

    on_punch = function(pos, node, puncher)
        if not (puncher and puncher:is_player()) then
            return
        end

        try_to_count_flag(pos, puncher, true, false)
    end,

    on_place = flag_on_place,
})

core.register_craft({
    type = "shaped",
    output = "superquest:flag 2",
    recipe = {
        {"farming:cotton", "dye:red", "farming:cotton"},
        {"farming:cotton", "default:stick", "farming:cotton"},
        {"farming:cotton", "default:stick", "farming:cotton"}
    }
})

if digilines then
    local digiline_handlers = {
        allow_player = function(pos, player_name)
            local meta = core.get_meta(pos)
            local owner = meta:get_string("owner")
            local network = meta:get_string("network")

            if owner == "" or network == "" then
                return
            end

            local network_obj = assert(superquest.Network(owner, network))

            if network_obj:get_timed_duration() ~= 0 then
                local active_network = superquest.active_timed_quests:get_active_network_for_player(player_name)
                if not active_network or active_network.owner ~= owner or active_network.network ~= network then
                    return
                end
            end

            local max_completions = network_obj:get_max_completions()
            if max_completions ~= 0 and network_obj:get_completions_for_player(player_name) >= max_completions then
                return
            end

            local is_counted = network_obj:is_flag_counted(pos, player_name)
            if is_counted then
                return
            end

            allowed_players_add(owner, network, pos, player_name)
        end,

        revoke_allow_player = function(pos, player_name)
            local meta = core.get_meta(pos)
            local owner = meta:get_string("owner")
            local network = meta:get_string("network")

            if owner == "" or network == "" then
                return
            end

            allowed_players_remove_for_flag(owner, network, pos, player_name)
        end,
    }
    core.register_node("superquest:flag_digiline_requiring", {
        description = S("Digiline-Requiring Flag"),
        drawtype = "mesh",
        mesh = "superquest_flag.obj",
        tiles = { "superquest_flag_blue.png" },
        inventory_image = "superquest_flag_blue.png",
        selection_box = {
            type = "fixed",
            fixed = { -0.1, -0.5, 0.1, 0.1, 1.5, -0.5 }
        },
        paramtype = "light",
        paramtype2 = "facedir",
        sunlight_propagates = true,
        use_texture_alpha = "clip",
        walkable = false,
        groups = { cracky=1, oddly_breakable_by_hand = 2 },

        digilines = {
            receptor = {},
            effector = {
                action = function(pos, _, channel, msg)
                    local meta = core.get_meta(pos)
                    if (meta:get_string("digiline_channel") ~= channel) then
                        return
                    end

                    superquest.digilines.parse.flag_digiline_requiring(digiline_handlers, pos, msg)
                end
            }
        },

        after_place_node = flag_after_place,

        on_rightclick = function(pos, node, clicker, itemstack, pointed_thing)
            show_menu(pos, clicker)
        end,

        on_destruct = flag_on_destruct,

        on_punch = function(pos, node, puncher)
            if not (puncher and puncher:is_player()) then
                return
            end

            try_to_count_flag(pos, puncher, true, true)
        end,

        on_place = flag_on_place,
    })

    core.register_craft({
        type = "shapeless",
        output = "superquest:flag_digiline_requiring",
        recipe = {
            "superquest:flag",
            "dye:blue",
            "digilines:wire_std_00000000"
        }
    })
end

core.register_on_player_receive_fields(function(player, formname, fields)
    if formname ~= form_name or not (player and player:is_player()) then
        return
    end

    local player_name = player:get_player_name()

    local pos = node_positions[player_name]
    if pos == nil then
        return
    end

    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")

    if owner ~= player_name and not superquest.privileges.can_edit_all(player_name) then
        return
    end

    if fields.save then
        if not fields.network then
            return
        end

        local old_network = meta:get_string("network")
        local new_network = fields.network:gsub('^%s*(.-)%s*$', '%1')

        if new_network:len() > superquest.config.max_network_name_length then
            core.chat_send_player(player_name, S("Max network name length (@1 symbols) exceeded", superquest.config.max_network_name_length))
            return
        end
        if new_network:find("#") then
            core.chat_send_player(player_name, S("Using the '#' symbol is not allowed in the network name"))
            return
        end

        local new_owner = owner
        if superquest.privileges.can_edit_all(player_name) then
            if not fields.owner then
                return
            end

            new_owner = fields.owner:gsub('^%s*(.-)%s*$', '%1')

            if new_owner:len() > superquest.config.max_owner_name_length then
                core.chat_send_player(player_name, S("Max owner name length (@1 symbols) exceeded", superquest.config.max_owner_name_length))
                return
            end
            if new_owner:find("#") then
                core.chat_send_player(player_name, S("Using the '#' symbol is not allowed in the owner name"))
                return
            end
        end

        if old_network ~= new_network or owner ~= new_owner then
            local old_network_obj = superquest.Network(owner, old_network)
            if old_network_obj then
                old_network_obj:remove_flag(pos)
            end

            meta:set_string("owner", new_owner)
            meta:set_string("network", new_network)

            local network_obj = superquest.Network(new_owner, new_network)
            if network_obj then
                network_obj:add_flag(pos)
            end
        end

        local max_channel_len = superquest.digilines.max_channel_length

        if fields.digiline_channel then
            if fields.digiline_channel:len() > max_channel_len then
                core.chat_send_player(player_name, S("Max Digiline channel name length (@1 symbols) exceeded", max_channel_len))
                return
            end
            meta:set_string("digiline_channel", fields.digiline_channel)
        end

        if superquest.config.flags_autotrigger then
            if not fields.from_coords or not fields.to_coords then
                return
            end

            local from_coords = fields.from_coords:split(",")
            local to_coords = fields.to_coords:split(",")

            if #from_coords ~= 3 or #to_coords ~= 3 then
                core.chat_send_player(player_name, S("Incorrect format of distance fields"))
                return
            end

            local from_x = tonumber(from_coords[1])
            local from_y = tonumber(from_coords[2])
            local from_z = tonumber(from_coords[3])
            local to_x = tonumber(to_coords[1])
            local to_y = tonumber(to_coords[2])
            local to_z = tonumber(to_coords[3])

            if not from_x or from_x < 0 or from_x > superquest.config.flags_autotrigger_max_distance or
               not from_y or from_y < 0 or from_y > superquest.config.flags_autotrigger_max_distance or
               not from_z or from_z < 0 or from_z > superquest.config.flags_autotrigger_max_distance or
               not to_x or to_x < 0 or to_x > superquest.config.flags_autotrigger_max_distance or
               not to_y or to_y < 0 or to_y > superquest.config.flags_autotrigger_max_distance or
               not to_z or to_z < 0 or to_z > superquest.config.flags_autotrigger_max_distance then

                core.chat_send_player(player_name, S("Incorrect distance values. Distance values should be integers between @1 and @2",
                    0, superquest.config.flags_autotrigger_max_distance))
                return
            end

            meta:set_int("scan_from_x", from_x)
            meta:set_int("scan_from_y", from_y)
            meta:set_int("scan_from_z", from_z)
            meta:set_int("scan_to_x", to_x)
            meta:set_int("scan_to_y", to_y)
            meta:set_int("scan_to_z", to_z)
        end

        if superquest.config.teleportation then
            if not fields.teleport_to then
                return
            end

            if fields.teleport_to ~= "" then
                local teleport_to = fields.teleport_to:split(",")
                if #teleport_to ~= 3 then
                    core.chat_send_player(player_name, S("Incorrect format of distance fields"))
                    return
                end

                local tp_to_x = tonumber(teleport_to[1])
                local tp_to_y = tonumber(teleport_to[2])
                local tp_to_z = tonumber(teleport_to[3])

                local max_dist = superquest.config.flags_teleport_max_distance

                if not tp_to_x or tp_to_x < -max_dist or tp_to_x > max_dist or
                   not tp_to_y or tp_to_y < -max_dist or tp_to_y > max_dist or
                   not tp_to_z or tp_to_z < -max_dist or tp_to_z > max_dist then
                    core.chat_send_player(player_name, S("Incorrect distance values. Distance values should be integers between @1 and @2", -max_dist, max_dist))
                    return
                end

                meta:set_string("teleport_to", tostring(tp_to_x)..","..tostring(tp_to_y)..","..tostring(tp_to_z))
            else
                meta:set_string("teleport_to", "")
            end
        end
    elseif fields.digiline_guide then
        superquest.digilines.open_guide_formspec(player_name)
    end
end)

if superquest.config.flags_autotrigger then
    local nodes_to_scan = { "superquest:flag" }
    if digilines then
        table.insert(nodes_to_scan, "superquest:flag_digiline_requiring")
    end
    core.register_abm({
        nodenames = nodes_to_scan,
        interval = superquest.config.flags_autotrigger_interval,
        chance = 1,
        action = function(pos, node)
            local meta = core.get_meta(pos)
            local from_x_dev = meta:get_int("scan_from_x")
            local from_y_dev = meta:get_int("scan_from_y")
            local from_z_dev = meta:get_int("scan_from_z")
            local to_x_dev = meta:get_int("scan_to_x")
            local to_y_dev = meta:get_int("scan_to_y")
            local to_z_dev = meta:get_int("scan_to_z")

            local from_v = vector.subtract(pos, vector.new(from_x_dev, from_y_dev, from_z_dev))
            local to_v = vector.add(pos, vector.new(to_x_dev, to_y_dev, to_z_dev))

            local objs = core.get_objects_in_area(from_v, to_v)

            for _, obj in pairs(objs) do
                if obj:is_player() then
                    try_to_count_flag(pos, obj, false, node.name == "superquest:flag_digiline_requiring")
                end
            end
        end
    })
end

superquest.flag = {}

superquest.flag.is_flag = function(node_name)
    return node_name == "superquest:flag" or node_name == "superquest:flag_digiline_requiring"
end

if superquest.config.teleportation then
    superquest.flag.get_tp_destination = function(owner, network_name, pos)
        local node = superquest.utils.get_node(pos)
        if not superquest.flag.is_flag(node.name) then
            return nil
        end

        local meta = core.get_meta(pos)
        if meta:get_string("owner") ~= owner or meta:get_string("network") ~= network_name then
            return nil
        end

        local teleport_to = meta:get_string("teleport_to")
        if teleport_to == "" then
            return nil
        end

        local teleport_to = teleport_to:split(",")

        return pos + vector.new(teleport_to[1], teleport_to[2], teleport_to[3])
    end

    core.register_on_dieplayer(function(player)
        local player_name = player:get_player_name()
        local allowed = superquest.teleportation.allow_tp_for_flag(player_name)
        if allowed then
            core.chat_send_player(player_name, S("You've just died!").." "..S("You can type '/superquest teleport' to teleport to the last triggered flag."))
        end
    end)
end

if digilines then
    superquest.flag.remove_allowed_players = allowed_players_remove_for_all
end
