local serpent = require("serpent")

local privileges = {}
local node_defs = {}
local on_player_receive_fields = {}
local crafts = {}
local abms = {}
local aliases = {}
local chat = {}
local chatcommands = {}
local on_dieplayer_handlers = {}
local on_joinplayer_handlers = {}
local on_leaveplayer_handlers = {}
local deferred_after_functions = {}

local players = {}
local nodes = {}

local worldpath = nil

local time_skipped = 0

local mod_storage = {
    _data = {
        fields = {}
    },

    from_table = function(self, t)
        assert(t.fields ~= nil)
        self._data = mut.copy_object(t)
    end,

    get_int = function(self, name)
        local value = self._data.fields[name]
        if value == nil then
            return 0
        end
        return tonumber(value)
    end,

    get_string = function(self, name)
        return self._data.fields[name] or ""
    end,

    set_int = function(self, name, value)
        assert(type(value) == "number")
        self._data.fields[name] = tostring(value)
    end,

    set_string = function(self, name, value)
        assert(type(value) == "string")
        if value == "" then
            self._data.fields[name] = nil
        else
            self._data.fields[name] = value
        end
    end,

    to_table = function(self)
        return mut.copy_object(self._data)
    end,
}

local vector_meta = {}

_G.vector = {
    add = function(vec1, vec2)
        local res = { x = vec1.x + vec2.x, y = vec1.y + vec2.y, z = vec1.z + vec2.z }
        return setmetatable(res, vector_meta)
    end,

    new = function(x, y, z)
        return setmetatable({ x = x, y = y, z = z }, vector_meta)
    end,

    subtract = function(vec1, vec2)
        local res = { x = vec1.x - vec2.x, y = vec1.y - vec2.y, z = vec1.z - vec2.z }
        return setmetatable(res, vector_meta)
    end,

    equals = function(a, b)
        return a.x == b.x and a.y == b.y and a.z == b.z
    end,

    from_string = function(s, init)
        local x, y, z, np = string.match(s, "^%s*%(%s*([^%s,]+)%s*[,%s]%s*([^%s,]+)%s*[,%s]" ..
                "%s*([^%s,]+)%s*[,%s]?%s*%)()", init)
        x = tonumber(x)
        y = tonumber(y)
        z = tonumber(z)
        if not (x and y and z) then
            return nil
        end
        return setmetatable({ x = x, y = y, z = z }, vector_meta)
    end,

    to_string = function(v)
        return string.format("(%g, %g, %g)", v.x, v.y, v.z)
    end,

    __mul = function(a, b)
        if type(a) == "table" then
            return setmetatable({ x = a.x * b, y = a.y * b, z = a.z * b }, vector_meta)
        else
            return setmetatable({ x = a * b.x, y = a * b.y, z = a * b.z }, vector_meta)
        end
    end
}
vector_meta.__mul = vector.__mul
vector_meta.__eq = vector.equals


local ItemStackBase = {
    get_count = function(self)
        return self.count
    end,
}
ItemStackBase.__index = ItemStackBase

function _G.ItemStack(par)
    local stack_obj = { name = "", count = 0 }
    setmetatable(stack_obj, ItemStackBase)
    if type(par) == "table" then
        stack_obj.name = par.name or stack_obj.name
        stack_obj.count = par.count or stack_obj.count
    elseif type(par) == "string" then
        local _, e, name = par:find("^(%S+)")
        stack_obj.name = name
        if #par >= e + 2 then
            local _, _, count = par:find(par:find("(%S+)$", e + 2))
            stack_obj.count = tonumber(count)
        else
            stack_obj.count = 1
        end
    end

    return stack_obj
end


local InventoryBase = {
    get_lists = function(self)
        local res_list = {}

        for k_inv, v_inv in pairs(self._data) do
            res_list[k_inv] = {}

            for i = 1,v_inv.size do
                table.insert(res_list[k_inv], v_inv.slots[i])
            end
        end

        return res_list
    end,

    is_empty = function(self, name)
        local size = self._data[name].size
        for i = 1,size do
            local slot = self._data[name].slots[i]
            if slot and slot.name ~= "" and slot.count > 0 then
                return false
            end
        end

        return true
    end,

    set_size = function(self, name, size)
        if not size then
            self._data[name] = nil
        else
            local old_size;
            if not self._data[name] then
                self._data[name] = {
                    size = size,
                    slots = {}
                }
                old_size = 0
            else
                old_size = self._data[name].size
            end

            if old_size < size then
                for i = old_size+1,size do
                    self._data[name].slots[i] = ItemStack()
                end
            elseif size < old_size then
                for i = size+1,old_size do
                    self._data[name].slots[i] = nil
                end
            end
        end
    end,
}
InventoryBase.__index = InventoryBase


local MetadataBase = {
    contains = function(self, name)
        return self._data.fields[name] ~= nil
    end,

    get_int = function(self, name)
        return tonumber(self._data.fields[name] or "0")
    end,

    get_inventory = function(self)
        return self._data.inventory
    end,

    get_string = function(self, name)
        return self._data.fields[name] or ""
    end,

    set_int = function(self, name, value)
        assert(type(value) == "number")
        self._data.fields[name] = tostring(value)
    end,

    set_string = function(self, name, value)
        assert(type(value) == "string")
        if value ~= "" then
            self._data.fields[name] = value
        else
            self._data.fields[name] = nil
        end
    end,
}
MetadataBase.__index = MetadataBase


local PlayerBase = {
    get_look_dir = function(self)
        return 0
    end,

    get_player_name = function(self)
        return self._data.name
    end,

    is_player = function(self)
        return true
    end,

    hud_add = function(self, hud_def)
        table.insert(self._data.huds, hud_def)
        return #self._data.huds
    end,

    hud_remove = function(self, id)
        table.remove(self._data.huds, id)
    end,
}
PlayerBase.__index = PlayerBase


_G.mut = {}

-- Check if two objects have equivalent data structure. The function doesn't take references into account.
function mut.compare_objects(obj1, obj2)
    if type(obj1) ~= type(obj2) then
        return false
    elseif type(obj1) ~= "table" then
        return obj1 == obj2
    end

    if mut.count_table_entries(obj1) ~= mut.count_table_entries(obj2) then
        return false
    end

    for k, v in pairs(obj1) do
        assert(type(k) ~= table)

        if not mut.compare_objects(v, obj2[k]) then
            return false
        end
    end

    return true
end

local function _copy_object(obj, old_tables, new_tables)
    if type(obj) ~= "table" then
        return obj
    end

    -- If the object was referenced before in another place, use its corresponding copy
    for k, v in ipairs(old_tables) do
        if obj == v then
            return new_tables[k]
        end
    end

    local obj_copy = {}
    table.insert(old_tables, obj)
    table.insert(new_tables, obj_copy)

    for k, v in pairs(obj) do
        obj_copy[_copy_object(k, old_tables, new_tables)] = _copy_object(v, old_tables, new_tables)
    end

    return obj_copy
end

function mut.copy_object(obj)
    return _copy_object(obj, {}, {})
end

function mut.count_table_entries(t)
    local count = 0

    for _ in pairs(t) do
        count = count + 1
    end

    return count
end

function mut.create_player(name, data, no_join)
    local privs = {}
    for _, v in ipairs(data.privs or {}) do
        privs[v] = true
    end

    players[name] = {
        _data = {
            name = name,
            pos = data.pos or vector.new(0, 0, 0),
            privs = privs,
            active_formspec = nil,
            huds = {}
        }
    }
    setmetatable(players[name], PlayerBase)

    if not no_join then
        for _, func in ipairs(on_joinplayer_handlers) do
            func(players[name], true)
        end
    end

    return { type = "player", name = name }
end

function mut.dig_node(pos, digger)
    if digger then
        assert(digger.type == "player")
    end

    local node_data = nodes[vector.to_string(pos)]._data.node
    local node_def = node_defs[node_data.name]

    local player_obj = nil
    if digger then
        player_obj = players[digger.name]
    end

    if node_def.on_dig then
        return node_def.on_dig(pos, node_data, digger)
    else
        return core.node_dig(pos, node_data, player_obj)
    end
end

function mut.get_mod_storage_data()
    return mod_storage:to_table()
end

function mut.is_formspec_active(player_name, formspec_name)
    return players[player_name]._data.active_formspec == formspec_name
end

function mut.place_node(pos, data)
    if data.placer then
        -- To do: maybe support mobs in the future
        assert(data.placer.type == "player")
    end

    local pointed_thing = {
        above = pos,
        under = vector.add(pos, data.bottom_dir or vector.new(0, -1, 0))
    }

    local node_def = node_defs[data.stack.name]

    local ret_stack
    if node_def.on_place then
        ret_stack = node_def.on_place(data.stack, players[data.placer.name], pointed_thing)
    else
        ret_stack = core.item_place(data.stack, players[data.placer.name], pointed_thing, nil)
    end

    if node_def.after_place_node then
        node_def.after_place_node(pos, players[data.placer.name], ret_stack, pointed_thing)
    end

    return { type = "node", pos = pos }, ret_stack
end

function mut.punch_node(pos, puncher)
    assert(puncher.type == "player")

    local node_data = nodes[vector.to_string(pos)]._data.node
    local node_def = node_defs[node_data.name]

    if node_def.on_punch then
        node_def.on_punch(pos, nil, players[puncher.name])
    end
end

function mut.rightclick_node(pos, data)
    if data.clicker then
        assert(data.clicker.type == "player")
    end
    local node_data = nodes[vector.to_string(pos)]._data.node
    local node_def = node_defs[node_data.name]

    if not node_def.on_rightclick then
        return
    end

    local player_obj = nil
    if data.clicker then
        player_obj = players[data.clicker.name]
    end

    node_def.on_rightclick(pos, nil, player_obj, nil, nil)
end

function mut.set_mod_storage_data(storage)
    mod_storage:from_table(storage)
end

function mut.set_settings(settings)
    core.settings._data = mut.copy_object(settings)
end

function mut.skip_time(time)
    local run_to_element = 0
    for k, v in ipairs(deferred_after_functions) do
        if v.time <= time then
            v.func(table.unpack(v.args))
            run_to_element = k
        end
    end
    if run_to_element ~= 0 then
        local new_table = {}
        for i = run_to_element+1,#deferred_after_functions do
            table.insert(new_table, deferred_after_functions[i])
        end
        deferred_after_functions = new_table
    end
    for k, v in ipairs(deferred_after_functions) do
        v.time = v.time - time
    end
    time_skipped = time_skipped + time
end

function mut.submit_active_formspec(player_name, data)
    local active_formspec = players[player_name]._data.active_formspec
    assert(active_formspec)

    for _, v in ipairs(on_player_receive_fields) do
        v(players[player_name], active_formspec, data)
    end
end

_G.core = {}

core.settings = {
    _data = {}
}

function core.add_particlespawner(definition)
end

function core.after(time, func, ...)
    local to_index = nil
    for k, v in ipairs(deferred_after_functions) do
        if v.time > time then
            to_index = k
            break
        end
    end
    if not to_index then
        to_index = #deferred_after_functions + 1
    end

    deferred_after_functions[to_index] = {
        time = time,
        func = func,
        args = table.pack(...)
    }
end

function core.chat_send_player(name, message)
    table.insert(chat, { name = name, msg = message })
end

function core.check_player_privs(name, privilege)
    return players[name]._data.privs[privilege]
end

function core.deserialize(string)
    local ok, res = serpent.load(string)
    assert(ok)
    return res
end

function core.dir_to_facedir(dir)
    -- To do
    return 0
end

function core.dir_to_wallmounted(dir)
    -- To do
    return 0
end

function core.formspec_escape(text)
    -- To do
    return text
end

function core.get_current_modname()
    local f = assert(io.open('mod.conf'), "r")

    local modname = ""
    for line in f:lines() do
        modname = line:match('^name = (%S*)')
        if modname then
            break
        end
    end
    f:close()

    return modname
end

function core.get_gametime()
    return os.time() + math.floor(time_skipped)
end

function core.get_meta(pos)
    return nodes[vector.to_string(pos)]._data.metadata
end

function core.get_mod_storage()
    return mod_storage
end

function core.get_modpath(modname)
    -- To do
    return "."
end

function core.get_node(pos)
    return nodes[vector.to_string(pos)]._data.node
end

function core.get_player_by_name(name)
    return players[name]
end

local function translator(str, ...)
    local res_str = ""
    local cursor = 1
    local string_len = str:len()

    while cursor <= string_len do
        local s = str:find("@", cursor)
        if not s then
            res_str = res_str .. str:sub(cursor)
            break
        end

        if cursor < s then
            res_str = res_str .. str:sub(cursor, s - 1)
        end

        assert(s ~= string_len)

        local symbol = str:sub(s + 1, s + 1)
        if symbol == '\n' or symbol == '=' or symbol == '@' then
            res_str = res_str .. symbol
        elseif symbol == 'n' then
            res_str = res_str .. '\n'
        elseif symbol:match('%d') then
            local num = tonumber(symbol)
            assert(num and arg[num])
            res_str = res_str .. tostring(arg[num])
        else
            error("Unexpected symbol after @")
        end

        cursor = s + 2
    end

    return res_str
end

function core.get_translator(_)
    return translator
end

function core.get_worldpath()
    if not worldpath then
        local f = assert(io.popen("mktemp -d"))
        worldpath = f:read("*l")
        f:close()
    end

    return worldpath
end

function core.item_place(itemstack, placer, pointed_thing, param2)
    assert(itemstack.count > 0)

    local node_def = node_defs[itemstack.name]

    -- Check for unsupported features
    assert(node_def)
    assert(not nodes[vector.to_string(pointed_thing.above)])
    assert(not nodes[vector.to_string(pointed_thing.under)])

    core.set_node(pointed_thing.above, {name = itemstack.name, param2 = param2})

    local remove_from_stack = 1

    if node_def.after_place_node then
        if node_def.after_place_node(pointed_thing.above, placer, itemstack, pointed_thing) then
            remove_from_stack = 0
        end
    end

    itemstack.count = itemstack.count - remove_from_stack

    return pointed_thing.above, itemstack
end

function core.load_area(pos)
end

function core.mkdir(dir)
    os.execute("mkdir " .. dir)
end

function core.node_dig(pos, node, digger)
    local node_def = node_defs[node.name]

    if node_def.can_dig and not node_def.can_dig(pos, digger) then
        return false
    end

    core.set_node(pos, {name = "air"})

    assert(not node_def.after_dig_node)

    return true
end

function core.register_abm(data)
    table.insert(abms, mut.copy_object(data))
end

function core.register_alias(alias, original_name)
    aliases[alias] = original_name
end

function core.register_chatcommand(name, data)
    chatcommands[name] = mut.copy_object(data)
end

function core.register_craft(data)
    table.insert(crafts, mut.copy_object(data))
end

function core.register_node(name, data)
    node_defs[name] = mut.copy_object(data)
end

function core.register_on_dieplayer(func)
    table.insert(on_dieplayer_handlers, func)
end

function core.register_on_joinplayer(func)
    table.insert(on_joinplayer_handlers, func)
end

function core.register_on_leaveplayer(func)
    table.insert(on_leaveplayer_handlers, func)
end

function core.register_on_player_receive_fields(func)
    table.insert(on_player_receive_fields, func)
end

function core.register_privilege(name, data)
    privileges[name] = mut.copy_object(data)
end

function core.serialize(table)
    return serpent.dump(table)
end

function core.set_node(pos, node)
    local pos_str = vector.to_string(pos)
    if nodes[pos_str] then
        local node_data = nodes[pos_str]._data.node
        local node_def = node_defs[node_data.name]

        if node_def.on_destruct then
            node_def.on_destruct(pos)
        end
    end

    if node.name == "air" then
        nodes[pos_str] = nil
    else
        local node_obj = {
            _data = {
                node = node,
                metadata = {
                    _data = {
                        fields = {},
                        inventory = {
                            _data = {}
                        }
                    }
                }
            }
        }
        setmetatable(node_obj._data.metadata, MetadataBase)
        setmetatable(node_obj._data.metadata._data.inventory, InventoryBase)

        nodes[pos_str] = node_obj

        local node_def = node_defs[node.name]

        if node_def.on_construct then
            node_def.on_construct(pos)
        end
    end
end

function core.settings:get_bool(name, value)
    local ret = core.settings._data[name] or value
    assert(type(ret) == "boolean")
    return ret
end

function core.settings:get(name)
    return core.settings._data[name]
end

function core.show_formspec(player_name, form_name, formspec)
    players[player_name]._data.active_formspec = form_name
end

local string_mt = getmetatable("")
string_mt.__index["split"] =  function(self, separator)
    local res = {}
    for str in self:gmatch("([^" .. separator .. "]+)") do
        table.insert(res, str)
    end
    return res
end
