local S = superquest.S

local path = superquest.path .. "/src/quest_machine"

local function synchronize_params(network_obj, meta)
    local updated = false
    if network_obj:get_max_completions() ~= meta:get_int("max_completions") then
        network_obj:set_max_completions_and_remove_flags_if_exceeds(meta:get_int("max_completions"))
        updated = true
    end

    if network_obj:get_timed_duration() ~= meta:get_int("timed_duration") then
        if network_obj:get_timed_duration() == 0 then
            network_obj:clear_reached_flags_for_all()
        end
        network_obj:set_timed_duration(meta:get_int("timed_duration"))
        updated = true
    end

    if network_obj:get_teleport_after_compl() ~= meta:contains("teleport_after_compl") then
        network_obj:set_teleport_after_compl(meta:contains("teleport_after_compl"))
        updated = true
    end

    if updated then
        network_obj:save()
    end
end

local function calc_avail_rewards_for_inventory(rewards_inv, quest_machine_inventory)
    local available_rewards
    for k_item, v_item in pairs(rewards_inv) do
        if k_item == "superquest:certificate" then
            k_item = "superquest:certificate_blank"
        end

        if not quest_machine_inventory[k_item] then
            return 0
        end

        local avail_cur_item_times
        if v_item.no_meta ~= 0 then
            avail_cur_item_times = quest_machine_inventory[k_item].no_meta / v_item.no_meta

            if available_rewards == nil or avail_cur_item_times < available_rewards then
                available_rewards = avail_cur_item_times
            end
            if available_rewards == 0 then
                return 0
            end
        end

        if #v_item.with_meta > 0 then
            for _, v_item_meta1 in ipairs(v_item.with_meta) do
                local is_found = false
                for _, v_item_meta2 in ipairs(quest_machine_inventory[k_item].with_meta) do
                    if v_item_meta1.meta_itemstack:get_meta():equals(v_item_meta2.meta_itemstack:get_meta()) then
                        avail_cur_item_times = v_item_meta2.count / v_item_meta1.count
                        if available_rewards == nil or avail_cur_item_times < available_rewards then
                            available_rewards = avail_cur_item_times
                        end
                        if available_rewards == 0 then
                            return 0
                        end
                        is_found = true

                        break
                    end
                end
                if not is_found then
                    return 0
                end
            end
        end
    end

    if available_rewards == nil then
        available_rewards = 0
    end

    return math.floor(available_rewards)
end

local function calc_items_in_inv_list(inv_list)
    local items_count = {}
    for _, stack_v in pairs(inv_list) do
        if not stack_v:is_empty() then
            items_count[stack_v:get_name()] = items_count[stack_v:get_name()] or {
                no_meta = 0,
                with_meta = {}
            }
            local item_total_data = items_count[stack_v:get_name()]
            if #stack_v:get_meta():get_keys() == 0 then
                item_total_data.no_meta = item_total_data.no_meta + stack_v:get_count()
            else
                local found_meta_total_data = nil
                for _, item_meta_total_data in ipairs(item_total_data.with_meta) do
                    if item_meta_total_data.meta_itemstack:get_meta():equals(stack_v:get_meta()) then
                        found_meta_total_data = item_meta_total_data
                        break
                    end
                end
                if not found_meta_total_data then
                    local stack_to_insert = {
                        count = stack_v:get_count(),
                        meta_itemstack = ItemStack({
                            name = stack_v:get_name(),
                            count = 1
                        })
                    }
                    stack_to_insert.meta_itemstack:get_meta():from_table(stack_v:get_meta():to_table())
                    table.insert(item_total_data.with_meta, stack_to_insert)
                else
                    found_meta_total_data.count = found_meta_total_data.count + stack_v:get_count()
                end
            end
        end
    end

    return items_count
end

local function is_enough_place_for_reward(reward_inv_list, user_inv_list)
    local items_count = calc_items_in_inv_list(reward_inv_list)

    local empty_cells_in_user_inv = 0
    local avail_for_items = {}

    for _, stack in pairs(user_inv_list) do
        if stack:is_empty() then
            empty_cells_in_user_inv = empty_cells_in_user_inv + 1
        elseif items_count[stack:get_name()] ~= nil and stack:get_count() < stack:get_stack_max() then
            avail_for_items[stack:get_name()] = avail_for_items[stack:get_name()] or {
                no_meta = 0,
                with_meta = {}
            }
            local item_total_data = avail_for_items[stack:get_name()]
            if #stack:get_meta():get_keys() == 0 then
                item_total_data.no_meta = item_total_data.no_meta + stack:get_stack_max() - stack:get_count()
            else
                local found_meta_total_data = nil
                for _, item_meta_total_data in ipairs(item_total_data.with_meta) do
                    if item_meta_total_data.meta_itemstack:get_meta():equals(stack:get_meta()) then
                        found_meta_total_data = item_meta_total_data
                        break
                    end
                end
                if not found_meta_total_data then
                    local stack_to_insert = {
                        count = stack:get_stack_max() - stack:get_count(),
                        meta_itemstack = ItemStack({
                            name = stack:get_name(),
                            count = 1
                        })
                    }
                    stack_to_insert.meta_itemstack:get_meta():from_table(stack:get_meta():to_table())
                    table.insert(item_total_data.with_meta, stack_to_insert)
                else
                    found_meta_total_data.count = found_meta_total_data.count + stack:get_stack_max() - stack:get_count()
                end
            end
        end
    end

    for k, v in pairs(items_count) do
        local cur_items_remaining = v
        if avail_for_items[k] ~= nil then
            cur_items_remaining.no_meta = cur_items_remaining.no_meta - math.min(cur_items_remaining.no_meta, avail_for_items[k].no_meta)
            for _, v_meta_src in ipairs(cur_items_remaining.with_meta) do
                for _, v_meta_dst in ipairs(avail_for_items[k].with_meta) do
                    if v_meta_src.meta_itemstack:get_meta():equals(v_meta_dst.meta_itemstack:get_meta()) then
                        v_meta_src.count = v_meta_src.count - math.min(v_meta_src.count, v_meta_dst.count)
                        break
                    end
                end
            end
        end

        local max_stack_size = ItemStack(k):get_stack_max()

        if empty_cells_in_user_inv < cur_items_remaining.no_meta / max_stack_size then
            return false
        end
        empty_cells_in_user_inv = empty_cells_in_user_inv - math.ceil(cur_items_remaining.no_meta / max_stack_size)

        for _, v_meta in ipairs(cur_items_remaining.with_meta) do
            if empty_cells_in_user_inv < v_meta.count / max_stack_size then
                return false
            end
            empty_cells_in_user_inv = empty_cells_in_user_inv - math.ceil(v_meta.count / max_stack_size)
        end

        -- We don't need to update avail_for_items for the current item here since we won't need it anymore
    end

    return true
end

local function remove_item_metadata_aware(inv, listname, stack)
    local stack_remaining = ItemStack(stack)

    if stack_remaining:is_empty() then
        return
    end

    for i = 1, inv:get_size(listname) do
        local cur_stack = inv:get_stack(listname, i)
        if not cur_stack:is_empty() and cur_stack:get_name() == stack_remaining:get_name() and cur_stack:get_meta():equals(stack_remaining:get_meta()) then
            local smallest_count = math.min(cur_stack:get_count(), stack_remaining:get_count())
            cur_stack:set_count(cur_stack:get_count() - smallest_count)
            stack_remaining:set_count(stack_remaining:get_count() - smallest_count)
            inv:set_stack(listname, i, cur_stack)

            if stack_remaining:is_empty() then
                break
            end
        end
    end
end

-- Initialized later because of circular dependency
local show_formspec_user_or_owner

local function set_mode_handler(player, pos, mode)
    local player_name = player:get_player_name()
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    if owner ~= player_name and not superquest.privileges.can_edit_all(player_name) then
        return
    end

    local network_obj = superquest.Network(owner, network_name)
    if not network_obj then
        return
    end

    if network_obj:get_quest_machine_coords() ~= pos then
        -- The quest machine doesn't correspond to the quest machine in storage for this network
        core.chat_send_player(player_name, S("The Quest Machine wasn't assigned to the specified network, resetting..."))
        meta:set_string("owner", "")
        meta:set_string("network", "")
        return
    end

    network_obj:set_mode(mode)

    show_formspec_user_or_owner(pos, player_name)
end

local function reset_completions(player, pos)
    local player_name = player:get_player_name()
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    if owner == "" or network_name == "" then
        return
    end

    if owner ~= player_name and not superquest.privileges.can_edit_all(player_name) then
        return
    end

    local network_obj = assert(superquest.Network(owner, network_name))
    network_obj:reset_all_completions(pos)
end

local formspec_reset_compl_confirmation = assert(loadfile(superquest.path.."/src/formspec_confirmation.lua"))({
    form_name = core.get_current_modname()..":reset_compl_confirmation",
    label = S("Are you sure you want to reset completions?"),
    yes_handler = reset_completions
})

local function reset_completions_handler(player, pos)
    formspec_reset_compl_confirmation.show(player:get_player_name(), pos)
end

local formspec_completions = assert(loadfile(path.."/formspec/formspec_completions.lua"))({
    reset_completions_handler = reset_completions_handler
})

local function show_completions_handler(player, pos)
    local player_name = player:get_player_name()
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    local network_obj = superquest.Network(owner, network_name)
    if not network_obj then
        return
    end
    local completions = network_obj:get_all_completions(pos)

    formspec_completions.show(player_name, pos, completions, player_name == owner)
end

local function start_quest_handler(player, pos)
    local player_name = player:get_player_name()
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    if superquest.active_timed_quests:get_active_network_for_player(player_name) then
        core.chat_send_player(player_name, S("You already have an active quest. Type '/superquest cancel' to cancel."))
        return
    end

    local network_obj = superquest.Network(owner, network_name)
    if not network_obj then
        return
    end

    local max_completions = meta:get_int("max_completions")

    if max_completions ~= 0 and network_obj:get_completions_for_player(player_name) >= max_completions then
        core.chat_send_player(player_name, S("You already reached the maximum number of completions"))
        return
    end

    local reached_flags, total_flags = network_obj:get_flags_stats_for_player(player_name)
    if reached_flags >= total_flags then
        core.chat_send_player(player_name, S("This quest has already been completed. Take your reward first."))
        return
    elseif reached_flags ~= 0 then
        network_obj:clear_reached_flags_for_player(player_name)
    end

    superquest.active_timed_quests:add(player_name, owner, network_name, meta:get_int("timed_duration"))

    local channel = meta:get_string("digiline_channel")
    if digilines and channel ~= "" then
        superquest.digilines.send.quest_started(pos, channel, player_name)
    end
end

local function reset_flags_handler(player, pos)
    local player_name = player:get_player_name()
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    local network_obj = superquest.Network(owner, network_name)
    if not network_obj then
        return
    end

    network_obj:clear_reached_flags_for_player(player_name)

    show_formspec_user_or_owner(pos, player_name)
end

local formspec_user = assert(loadfile(path.."/formspec/formspec_user.lua"))({
    set_mode_handler = set_mode_handler,
    show_completions_handler = show_completions_handler,
    start_quest_handler = start_quest_handler,
    reset_flags_handler = reset_flags_handler,
})

local function save_data_handler(player, pos, data)
    local player_name = player:get_player_name()
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local old_network_name = meta:get_string("network")
    local old_color = meta:get_string("color")
    local is_same_network = old_network_name == data.new_network_name and owner == data.new_owner

    if data.new_owner then
        if not superquest.privileges.can_edit_all(player_name) then
            return
        end
    else
        data.new_owner = owner
    end

    local network_obj = superquest.Network(data.new_owner, data.new_network_name, false)

    if is_same_network and network_obj and network_obj:get_quest_machine_coords() ~= pos then
        -- The quest machine doesn't correspond to the quest machine in storage for this network
        core.chat_send_player(player_name, S("The Quest Machine wasn't assigned to the specified network, resetting..."))
        meta:set_string("owner", "")
        meta:set_string("network", "")
        return
    end

    local old_timed_duration = nil
    if meta:contains("timed_duration") then
        old_timed_duration = meta:get_int("timed_duration")
    end

    if not is_same_network then
        if network_obj then
            if network_obj:get_quest_machine_coords() then
                core.chat_send_player(player_name, S("This network already contains a quest machine"))
                return
            end

            if data.timed_duration then
                network_obj:clear_reached_flags_for_all()
            end
        end

        local old_network_obj = superquest.Network(owner, old_network_name)
        if old_network_obj then
            old_network_obj:remove_quest_machine(pos)

            if old_timed_duration then
                superquest.active_timed_quests:stop_and_remove_all_timers_for_quest(owner, old_network_name)
            end

            if old_color ~= "" then
                local old_flags = old_network_obj:get_network_elements().flags
                for _, flag in ipairs(old_flags) do
                    local flag_pos = superquest.storage.key_to_coords(flag.coords)
                    local flag_node = core.get_node_or_nil(flag_pos)
                    if flag_node then
                        superquest.flag.remove_color_mark(flag_pos)
                    end
                end
            end
        end
    elseif network_obj and is_same_network then
        if old_timed_duration and not data.timed_duration then
            superquest.active_timed_quests:stop_and_remove_all_timers_for_quest(owner, old_network_name)
        elseif not old_timed_duration and data.timed_duration then
            network_obj:clear_reached_flags_for_all()
        elseif old_timed_duration and data.timed_duration and old_timed_duration ~= data.timed_duration then
            superquest.active_timed_quests:adjust_timers_for_quest(owner, old_network_name, data.timed_duration - old_timed_duration)
        end
    end

    meta:set_string("owner", data.new_owner)
    meta:set_string("network", data.new_network_name)
    meta:set_int("max_completions", data.max_completions)
    if data.color then
        meta:set_string("color", data.color)
    else
        meta:set_string("color", "")
    end

    if data.digiline_channel then
        meta:set_string("digiline_channel", data.digiline_channel)
    end

    if data.timed_duration then
        meta:set_int("timed_duration", data.timed_duration)
    else
        meta:set_string("timed_duration", "")
    end

    if superquest.config.teleportation then
        if data.teleport_after_compl then
            meta:set_int("teleport_after_compl", 1)
        else
            meta:set_string("teleport_after_compl", "")
        end
    end

    if network_obj then
        if not is_same_network then
            network_obj:add_quest_machine(pos, {
                max_completions = data.max_completions,
                timed_duration = data.timed_duration,
                teleport_after_compl = data.teleport_after_compl,
                color = data.color
            })
        else
            network_obj:set_timed_duration(data.timed_duration)
            network_obj:set_color(data.color)
            if superquest.config.teleportation then
                network_obj:set_teleport_after_compl(data.teleport_after_compl)
            end
        end

        if not is_same_network or old_color ~= meta:get_string("color") then
            local flags = network_obj:get_network_elements().flags
            for _, flag in ipairs(flags) do
                local flag_pos = superquest.storage.key_to_coords(flag.coords)
                local flag_node = core.get_node_or_nil(flag_pos)
                if flag_node then
                    if is_same_network and old_color ~= "" then
                        superquest.flag.remove_color_mark(flag_pos)
                    end
                    if meta:get_string("color") ~= "" then
                        superquest.flag.add_color_mark(flag_pos, flag_node.param2, meta:get_string("color"))
                    end
                end
            end
        end

        network_obj:save()
    end
end

local function open_digiline_guide_handler(player)
    local player_name = player:get_player_name()

    superquest.digilines.open_guide_formspec(player_name)
end

local formspec_owner = assert(loadfile(path.."/formspec/formspec_owner.lua"))({
    set_mode_handler = set_mode_handler,
    save_data_handler = save_data_handler,
    show_completions_handler = show_completions_handler,
    open_digiline_guide_handler = open_digiline_guide_handler
})

show_formspec_user_or_owner = function(pos, player_name)
    local meta = core.get_meta(pos)

    local owner = meta:get_string("owner")
    local network = meta:get_string("network")
    local max_completions = meta:get_int("max_completions")
    local node_inv = meta:get_inventory()
    local timed_duration = meta:get_int("timed_duration")
    local color = meta:get_string("color")
    if color == "" then
        color = nil
    end

    local reached_flags = 0
    local total_flags = 0
    local network_obj = superquest.Network(owner, network)
    if network_obj then
        synchronize_params(network_obj, meta)
        reached_flags, total_flags = network_obj:get_flags_stats_for_player(player_name)
    end

    local qm_inv_items_count = calc_items_in_inv_list(node_inv:get_list("reward_box_storage"))

    local available_rewards = {
        first_compl_rewards = calc_avail_rewards_for_inventory(calc_items_in_inv_list(node_inv:get_list("first_compl_rewards")), qm_inv_items_count),
        further_compl_rewards = calc_avail_rewards_for_inventory(calc_items_in_inv_list(node_inv:get_list("further_compl_rewards")), qm_inv_items_count),
    }

    if (owner == player_name or superquest.privileges.can_edit_all(player_name)) and (not network_obj or not network_obj:is_in_user_mode()) then
        local teleport_after_compl = meta:contains("teleport_after_compl")
        local digiline_channel = meta:get_string("digiline_channel")

        formspec_owner.show(player_name, pos, {
            network = network,
            owner = owner,
            max_completions = max_completions,
            available_rewards = available_rewards,
            total_flags = total_flags,
            timed_duration = timed_duration,
            digiline_channel = digiline_channel,
            teleport_after_compl = teleport_after_compl,
            can_edit_owner = superquest.privileges.can_edit_all(player_name),
            color = color,
        })
    else
        local completions = network_obj and network_obj:get_completions_for_player(player_name) or 0

        formspec_user.show(player_name, pos, {
            owner = owner,
            network = network,
            owner_mode_avail = owner == player_name or superquest.privileges.can_edit_all(player_name),
            max_completions = max_completions,
            available_rewards = available_rewards,
            timed_duration = timed_duration,
            total_flags = total_flags,
            completions = completions,
            reached_flags = reached_flags,
            color = color,
        })
    end
end

local function remove_quest_machine_data(pos)
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    local network_obj = superquest.Network(owner, network_name)
    if not network_obj then
        return
    end

    network_obj:remove_quest_machine(pos)

    if meta:get_string("timed_duration") then
        superquest.active_timed_quests:stop_and_remove_all_timers_for_quest(owner, network_name)
    end
end

local function try_to_receive_reward(player, pos)
    local meta = core.get_meta(pos)

    local player_name = player:get_player_name()

    local owner = meta:get_string("owner")
    local network_name = meta:get_string("network")

    if owner == "" or network_name == "" then
        core.chat_send_player(player_name, S("The Quest Machine is not configured!"))
        return false
    end

    local network_obj = assert(superquest.Network(owner, network_name, false))

    if network_obj:get_quest_machine_coords() ~= pos then
        -- The quest machine doesn't correspond to the quest machine in storage for this network
        core.chat_send_player(player_name, S("The Quest Machine wasn't assigned to the specified network, resetting..."))
        meta:set_string("owner", "")
        meta:set_string("network", "")
        return false
    end

    synchronize_params(network_obj, meta)

    if (owner == player_name or superquest.privileges.can_edit_all(player_name))
            and not network_obj:is_in_user_mode() then
        return false
    end

    local max_completions = meta:get_int("max_completions")

    local completions = network_obj:get_completions_for_player(player_name)
    if max_completions ~= 0 and completions >= max_completions then
        core.chat_send_player(player_name, S("You already reached the maximum number of completions"))
        return false
    end

    local punched_flags, total_flags = network_obj:get_flags_stats_for_player(player_name)
    if punched_flags ~= total_flags then
        return false
    end

    local node_inv = meta:get_inventory()
    local player_inv = player:get_inventory()

    local rewards_list;

    local qm_inv_items_count = calc_items_in_inv_list(node_inv:get_list("reward_box_storage"))

    if completions == 0 then
        rewards_list = node_inv:get_list("first_compl_rewards")
    else
        rewards_list = node_inv:get_list("further_compl_rewards")
    end

    local avail_rewards = calc_avail_rewards_for_inventory(calc_items_in_inv_list(rewards_list), qm_inv_items_count)

    if avail_rewards == 0 then
        core.chat_send_player(player_name, S("There are not enough items to pay out your reward!"))
        return false
    end

    local certificate_inv_index = {}
    local cert_meta_table = {
        fields = {
            receiver = player_name,
            quest_name = network_name,
            quest_author = owner,
        }
    }
    for key, stack in ipairs(rewards_list) do
        if stack:get_name() == "superquest:certificate" then
            stack:get_meta():from_table(cert_meta_table)
            certificate_inv_index[key] = true
        end
    end

    local is_enough_place_in_inv = is_enough_place_for_reward(rewards_list, player_inv:get_list("main"))
    if not is_enough_place_in_inv then
        core.chat_send_player(player_name, S("There is not enough space in your inventory!"))
        return false
    end

    completions = completions + 1
    network_obj:clear_reached_flags_for_player(player_name)
    network_obj:set_completions_for_player(player_name, completions)
    network_obj:save()

    for key, v in pairs(rewards_list) do
        local item_remove
        if certificate_inv_index[key] then
            item_remove = ItemStack({
                name = "superquest:certificate_blank",
                count = v:get_count(),
            })
        else
            item_remove = v
        end
        if not node_inv:contains_item("reward_box_storage", item_remove, true) then
            -- Should never happen but just in case
            core.log("error", "BUG: Quest Machine's storage doesn't contain enough items while it should have, pos = " .. vector.to_string(pos))
        end
        player_inv:add_item("main", v)
        remove_item_metadata_aware(node_inv, "reward_box_storage", item_remove)
    end

    local channel = meta:get_string("digiline_channel")
    if digilines and channel ~= "" then
        superquest.digilines.send.reward_received(pos, channel, player_name, completions)
    end

    if superquest.config.teleportation then
        superquest.teleportation.reset_tp_destination(player_name)
    end

    return true
end

superquest.quest_machine.node = assert(loadfile(path.."/nodes.lua"))({
    get_quest_machine_coords = function(owner, network)
        local network_obj = assert(superquest.Network(owner, network))
        return network_obj:get_quest_machine_coords()
    end,
    is_owner_mode_for_player = function(owner, network, player_name)
        if owner == player_name or superquest.privileges.can_edit_all(player_name) then
            local network_obj = superquest.Network(owner, network)
            return not network_obj or not network_obj:is_in_user_mode()
        else
            return false
        end
    end,
    remove_quest_machine_data_handler = remove_quest_machine_data,
    show_formspec_handler = show_formspec_user_or_owner,
    try_to_receive_reward_handler = try_to_receive_reward
})
