local S = superquest.S

local receiver_field = "receiver"
local quest_name_field = "quest_name"
local quest_author_field = "quest_author"

display_api.register_display_entity("superquest:certificate_text")

local metadata_to_set = nil

local function to_multiline(text, font, max_len)
    local text_rem = text
    local codepoint
    local index = 0
    local ws_cp = font:get_next_char(" ")
    local max_lines = 3

    local lines_indices = {}
    local cur_line_start = 1
    local cur_line_end = 0
    local cur_line_width = 0
    local cur_word_width = 0

    max_len = max_len - font:get_char_width(font:get_next_char("\""))

    while text_rem ~= "" and #lines_indices < max_lines do
        index = index + 1

        codepoint, text_rem = font:get_next_char(text_rem)
        if codepoint == nil then
            return ""
        end
        if codepoint == ws_cp then
            cur_line_end = index - 1
            cur_line_width = cur_line_width + cur_word_width
            cur_word_width = 0
        end

        local cur_cp_width = font:get_char_width(codepoint)
        cur_word_width = cur_word_width + cur_cp_width
        if cur_line_width + cur_word_width > max_len then
            if cur_line_width == 0 then
                table.insert(lines_indices, { s = cur_line_start, e = index - 1})
                cur_line_start = index
                cur_line_end = index - 1
                cur_word_width = cur_cp_width
            else
                table.insert(lines_indices, { s = cur_line_start, e = cur_line_end })
                cur_line_start = cur_line_end + 2
                cur_line_end = cur_line_start - 1
                cur_word_width = cur_word_width - font:get_char_width(ws_cp) -- Whitespace was transformed into \n
                cur_line_width = 0
            end
            if #lines_indices == max_lines - 1 then
                -- To do: check if the text would fit instead of ...
                max_len = max_len - font:get_char_width(font:get_next_char(".")) * 3
            end
        end
    end

    local is_not_full = #lines_indices < max_lines
    if is_not_full then
        table.insert(lines_indices, { s = cur_line_start, e = index })
    end

    local final_text = ""
    for k, v in ipairs(lines_indices) do
        if k ~= 1 then
            final_text = final_text .. "\n"
        end
        final_text = final_text .. text:sub(v.s, v.e)
    end

    if not is_not_full then
        final_text = final_text .. "..."
    end

    return final_text
end

local function build_label_text(pos, font, tex_width)
    local meta = core.get_meta(pos)

    if metadata_to_set then
        meta:set_string(receiver_field, metadata_to_set.fields[receiver_field])
        meta:set_string(quest_name_field, metadata_to_set.fields[quest_name_field])
        meta:set_string(quest_author_field, metadata_to_set.fields[quest_author_field])
        metadata_to_set = nil
    end

    if not meta:contains(receiver_field) or not meta:contains(quest_name_field) or not meta:contains(quest_author_field) then
        return ""
    end

    local final_text = "\nTo " .. meta:get_string(receiver_field) ..
                        "\nfor completing quest" ..
                        "\n\"" .. to_multiline(meta:get_string(quest_name_field), font, tex_width) .. "\"" ..
                        "\n(by " .. meta:get_string(quest_author_field) .. ")"

    return final_text
end

core.register_node("superquest:certificate", {
    description = S("SuperQuest Certificate"),
    inventory_image = "superquest_certificate_front_filled.png",
    wield_image = "superquest_certificate_front_filled.png",
    drawtype = "nodebox",
    sunlight_propagates = true,
    paramtype = "light",
    paramtype2 = "facedir",
    stack_max = 1,
    groups = { display_api = 1, choppy = 2, dig_immediate = 2, not_in_creative_inventory = 1 },
    tiles = {
        "superquest_certificate_top.png",
        "superquest_certificate_bottom.png",
        "superquest_certificate_right.png",
        "superquest_certificate_left.png",
        "superquest_certificate_back.png",
        "superquest_certificate_front_blank.png", -- Should be "blank" since we print real text over it
    },
    use_texture_alpha = "clip",
    node_box = {
        type = "fixed",
        fixed = { -0.40625, -0.3203125, 0.49, 0.40625, 0.3203125, 0.5 }
    },

    display_entities = {
        ["superquest:certificate_text"] = {
            depth = 0.5 - display_api.entity_spacing - 0.01,
            on_display_update = function(pos, objref)
                local size = { x = 0.8, y = 0.6 }
                local lines = 10

                local font = font_api.get_font()
                local tex_height = font:get_height(lines)
                local tex_width = tex_height / size.y * size.x
                local text = build_label_text(pos, font, tex_width)

                objref:set_properties({
                    textures = {
                        font:render(text, tex_width, tex_height, {
                            lines = lines,
                        })
                    },
                    visual_size = size
                })
            end,
        }
    },

    preserve_metadata = function(pos, oldnode, oldmeta, drops)
        if not oldmeta[receiver_field] or not oldmeta[quest_name_field] or not oldmeta[quest_author_field] then
            return
        end

        local s_meta = drops[1]:get_meta()
        s_meta:set_string(receiver_field, oldmeta[receiver_field])
        s_meta:set_string(quest_name_field, oldmeta[quest_name_field])
        s_meta:set_string(quest_author_field, oldmeta[quest_author_field])
    end,

    on_place = function(itemstack, placer, pointed_thing)
        local s_meta = itemstack:get_meta()
        if s_meta:contains(receiver_field) and s_meta:contains(quest_name_field) and s_meta:contains(quest_author_field) then
            metadata_to_set = s_meta:to_table()
        end

        local leftover = display_api.on_place(ItemStack(itemstack), placer, pointed_thing)
        metadata_to_set = nil

        return leftover
    end,
    on_construct = display_api.on_construct,
    on_destruct = display_api.on_destruct,
    on_rotate = display_api.on_rotate,
})

core.register_craftitem("superquest:certificate_blank", {
    description = S("SuperQuest Certificate Blank"),
    inventory_image = "superquest_certificate_front_blank.png",
    wield_image = "superquest_certificate_front_blank.png",
})
