local S = superquest.S

local top_command_name = core.get_current_modname()

local commands_list = {}

local function generate_network_data(player_name, owner_name, network_name)
    local network_obj = assert(superquest.Network(owner_name, network_name))
    local qm_coords = network_obj:get_quest_machine_coords()

    local description = S("Network") .. " " .. network_name .. ":\n"
    local waypoints = {}

    for pos_str in network_obj:get_flags_static_iterator() do
        description = description .. "\t" .. S("Flag at") .. " " .. pos_str .. "\n"
        waypoints[superquest.storage.key_to_coords(pos_str)] = { name = S("Flag"), color = 0xff0000 }
    end

    if qm_coords then
        description = description .. "\t" .. S("Quest Machine at") .. " " .. superquest.storage.coords_to_key(qm_coords) .. "\n"
        waypoints[qm_coords] = { name = S("Quest Machine"), color = 0xffff00 }
    end

    core.chat_send_player(player_name, description)
    superquest.hud.show_waypoints(core.get_player_by_name(player_name), waypoints)
end

commands_list["list_my_networks"] = {
    params = "",
    description = S("Lists all the networks of the current user"),
    func = function(name, param)
        local networks_list = superquest.storage.get_networks_list()

        if networks_list[name] ~= nil and next(networks_list[name], nil) ~= nil then
            local res = S("Networks") .. ":\n"

            for k, _ in pairs(networks_list[name]) do
                res = res .. "\t" .. k .. "\n"
            end

            core.chat_send_player(name, res)
        else
            core.chat_send_player(name, S("No networks"))
        end

        return true
    end,
}

commands_list["list_my_network_elements"] = {
    params = "<network_name>",
    description = S("Lists all the elements of a particular network of the current user"),
    func = function(player_name, param)
        if param == nil then
            core.chat_send_player(player_name, S("Network name is not specified"))
            return true
        end

        local network_name = param:gsub('^%s*(.-)%s*$', '%1')

        generate_network_data(player_name, player_name, network_name)

        return true
    end
}

commands_list["cancel"] = {
    params = "",
    description = S("Cancel the current active quest"),
    func = function(player_name, _)
        superquest.active_timed_quests:expire_current_for_player(player_name)
    end
}

if superquest.config.teleportation then
    commands_list["teleport"] = {
        params = "",
        description = S("Teleport to the Quest Machine of the last completed or expired quest."),
        func = function(player_name, _)
            superquest.teleportation.teleport_to_destination(player_name)
        end
    }
end

local command_description = S("Commands") .. ":\n"
for k, v in pairs(commands_list) do
    command_description = command_description .. "/" .. top_command_name .. " " .. k .. " " .. v.params .. "\n" .. v.description .. "\n\n"
end
command_description = command_description:gsub('^%s*(.-)%s*$', '%1')

core.register_chatcommand(top_command_name, {
    params = "<command_name> [<parameters>]",
    description = command_description,
    func = function(name, param)
        local command_name
        local subparam
        local space_index = param:find(" ")
        if space_index ~= nil then
            command_name = param:sub(1, space_index - 1)
            subparam = param:sub(space_index + 1, -1)
        else
            command_name = param
            subparam = nil
        end

        local command = commands_list[command_name]
        if (command ~= nil) then
            return command.func(name, subparam)
        else
            core.chat_send_player(name, S("Unknown command") .. ": " .. command_name)
            return true
        end
    end
})

local function parse_owner_network_name(caller_name, param)
    if param == nil then
        core.chat_send_player(caller_name, S("Parameter is not specified"))
        return nil
    end

    local delimiter_pos = param:find("#")
    if not delimiter_pos or delimiter_pos == 1 or delimiter_pos == #param then
        core.chat_send_player(caller_name, S("Incorrect parameter format"))
        return nil
    end

    local owner = param:sub(1, delimiter_pos - 1):gsub('^%s*(.-)%s*$', '%1')
    local network_name = param:sub(delimiter_pos + 1, -1):gsub('^%s*(.-)%s*$', '%1')

    if #owner == 0 or #network_name == 0 or network_name:find("#") then
        core.chat_send_player(caller_name, S("Incorrect parameter value"))
        return nil
    end

    return owner, network_name
end

local top_admin_command_name = top_command_name .. "_admin"

local admin_commands_list = {}

admin_commands_list["list_owners"] = {
    params = "",
    description = S("Lists all the players which own networks"),
    func = function(name, param)
        local networks_list = superquest.storage.get_networks_list()
        local res = S("Players") .. ":\n";

        for k, v in pairs(networks_list) do
            res = res .. "\t" .. k .. "\n"
        end

        core.chat_send_player(name, res);

        return true;
    end
}

admin_commands_list["list_player_networks"] = {
    params = "<player_name>",
    description = S("Lists all the networks of a player"),
    func = function(name, param)
        local networks_list = superquest.storage.get_networks_list()

        if param == nil then
            core.chat_send_player(name, S("Player is not specified"))
            return true
        end

        local player_name = param:gsub('^%s*(.-)%s*$', '%1')

        if networks_list[player_name] ~= nil and next(networks_list[player_name], nil) ~= nil then
            local res = S("Networks") .. ":\n"

            for k, _ in pairs(networks_list[player_name]) do
                res = res .. "\t" .. k .. "\n"
            end

            core.chat_send_player(name, res)
        else
            core.chat_send_player(name, S("No networks"))
        end

        return true
    end
}

admin_commands_list["list_player_network_elements"] = {
    params = "<player_name>#<network_name>",
    description = S("Lists all the elements of a network of a player"),
    func = function(player_name, param)
        local owner_name, network_name = parse_owner_network_name(player_name, param)
        if not owner_name or not network_name then
            return true
        end

        generate_network_data(player_name, owner_name, network_name)

        return true
    end
}

admin_commands_list["fix_network"] = {
    params = "<player_name>#<network_name>",
    description = S("Scan the network and remove all the elements which don't match"),
    func = function(player_name, param)
        local owner_name, network_name = parse_owner_network_name(player_name, param)
        if not owner_name or not network_name then
            return true
        end

        local network_obj = assert(superquest.Network(owner_name, network_name))

        for pos_str, flag_name in network_obj:get_flags_static_iterator() do
            local pos = superquest.storage.key_to_coords(pos_str)
            local node = superquest.utils.get_node(pos)
            local meta = core.get_meta(pos)
            if not node or not superquest.flag.is_flag(node.name) or meta:get_string("owner") ~= owner_name
                    or meta:get_string("network") ~= network_name
                    or (flag_name or "") ~= meta:get_string("name") then
                network_obj:remove_flag(pos, flag_name)
                core.chat_send_player(player_name, S("Flag at") .. " " .. pos_str .. " " .. S("was removed from the network"))
            end
        end

        local pos = network_obj:get_quest_machine_coords()
        if pos then
            local node = superquest.utils.get_node(pos)
            local meta = core.get_meta(pos)
            if not node or not superquest.quest_machine.node.is_quest_machine(node.name)
                    or meta:get_string("owner") ~= owner_name
                    or meta:get_string("network") ~= network_name then
                network_obj:remove_quest_machine(pos)
                core.chat_send_player(player_name, S("Quest machine at") .. " " .. superquest.storage.coords_to_key(pos) .. " " .. S("was removed from the network"))
            end
        end
    end
}

local admin_command_description = S("Commands") .. ":\n"
for k, v in pairs(admin_commands_list) do
    admin_command_description = admin_command_description .. "/" .. top_admin_command_name .. " " .. k .. " " .. v.params .. "\n" .. v.description .. "\n\n"
end
admin_command_description = admin_command_description:gsub('^%s*(.-)%s*$', '%1')

core.register_chatcommand(top_admin_command_name, {
    params = "<command_name> [<parameters>]",
    description = admin_command_description,
    privs = { [superquest.privileges.admin_priv_name] = true },
    func = function(name, param)
        local command_name
        local subparam
        local space_index = param:find(" ")
        if space_index ~= nil then
            command_name = param:sub(1, space_index - 1)
            subparam = param:sub(space_index + 1, -1)
        else
            command_name = param
            subparam = nil
        end

        local command = admin_commands_list[command_name]
        if (command ~= nil) then
            return command.func(name, subparam)
        else
            core.chat_send_player(name, S("Unknown command") .. ": " .. command_name)
            return true
        end
    end
})
