superquest.teleportation = {}

local S = superquest.S

local teleport_to_per_player = {}

local DestinationType = {
    FLAG = {},
    QUEST_MACHINE = {},
}

local function is_node_suitable_for_tp(node_name)
    if node_name == "ignore" then
        return false
    end

    local node_def = core.registered_nodes[node_name]
    if not node_def then
        return false
    end

    return node_def.drawtype == "airlike" or node_def.walkable == false
end

local function can_teleport_to(pos)
    local node_dest = superquest.utils.get_node(pos)
    if not is_node_suitable_for_tp(node_dest.name) then
        return false
    end

    local node_dest_below = superquest.utils.get_node(pos + vector.new(0, -1, 0))
    local node_dest_above = superquest.utils.get_node(pos + vector.new(0, 1, 0))
    -- It's OK if one of them is not suitable but not both
    if not is_node_suitable_for_tp(node_dest_below.name) and not is_node_suitable_for_tp(node_dest_above.name) then
        return false
    end

    return true
end

superquest.teleportation.set_tp_dest_to_quest_machine = function(player_name, owner, network, reset_after_tp)
    teleport_to_per_player[player_name] = {
        type = DestinationType.QUEST_MACHINE,
        owner = owner,
        network = network,
        reset_after_tp = reset_after_tp,
        allow_tp = true
    }
end

superquest.teleportation.set_tp_dest_to_flag_if_possible = function(player_name, owner, network, pos)
    local tp_dest_pos = superquest.flag.get_tp_destination(owner, network, pos)
    -- If we cannot teleport to the flag, skip it and leave previous one if present
    if not tp_dest_pos then
        return
    end

    if not can_teleport_to(tp_dest_pos) then
        return
    end

    teleport_to_per_player[player_name] = {
        type = DestinationType.FLAG,
        owner = owner,
        network = network,
        pos = pos,
        reset_after_tp = false,
        allow_tp = false
    }
end

superquest.teleportation.reset_tp_destination = function(player_name)
    teleport_to_per_player[player_name] = nil
end

superquest.teleportation.allow_tp_for_flag = function(player_name)
    local tp_dest = teleport_to_per_player[player_name]
    if tp_dest and tp_dest.type == DestinationType.FLAG and not tp_dest.allow_tp then
        teleport_to_per_player[player_name].allow_tp = true
        return true
    end

    return false
end

superquest.teleportation.teleport_to_destination = function(player_name)
    local tp_dest = teleport_to_per_player[player_name]
    if not tp_dest or not tp_dest.allow_tp then
        return
    end

    local tp_dest_pos = nil
    if tp_dest.type == DestinationType.QUEST_MACHINE then
        tp_dest_pos = superquest.quest_machine.node.get_tp_destination(tp_dest.owner, tp_dest.network)
    elseif tp_dest.type == DestinationType.FLAG then
        -- Despite we have already checked it before, we cannot rely on that data since it could have been changed since
        tp_dest_pos = superquest.flag.get_tp_destination(tp_dest.owner, tp_dest.network, tp_dest.pos)
    end

    if tp_dest_pos == nil then
        return
    end

    if not can_teleport_to(tp_dest_pos) then
        core.chat_send_player(player_name, S("There is not enough space to teleport"))
        return
    end

    local player = core.get_player_by_name(player_name)
    if player then
        player:set_pos(tp_dest_pos)
    end

    if tp_dest.reset_after_tp then
        superquest.teleportation.reset_tp_destination(player_name)
    end
end

core.register_on_leaveplayer(function(player)
    superquest.teleportation.reset_tp_destination(player:get_player_name())
end)
