-- PIGS TO SHEEP


local S = minetest.get_translator("mobs_mc")

--------------------------------------------------------------
-- PART 1: HIDE ITEMS FROM CREATIVE INVENTORY (SMART LOOP)
--------------------------------------------------------------
-- Instead of guessing exact names, we search for anything resembling pork or pig eggs.

minetest.after(0, function()
    -- List of exact matches (just in case)
    local force_hide = {
        ["mobs_mc:pig"] = true,
        ["m16b_new_mobs:warm_pig"] = true,
        ["m16b_new_mobs:cold_pig"] = true,
        ["mobs_mc:boar"] = true,
        ["mobs_mc:piglet"] = true
    }

    for name, def in pairs(minetest.registered_items) do
        local should_hide = false

        -- 1. Hide meat (pork)
        if string.find(name, "pork") then
            should_hide = true
        end

        -- 2. Hide eggs of standard pigs, but NOT Piglins
        if (string.find(name, "egg") or string.find(name, "spawn")) and 
           (string.find(name, "pig") or string.find(name, "boar")) then
            
            -- Important: Do not touch Piglins, Hoglins, and Zombies
            if not string.find(name, "piglin") and 
               not string.find(name, "hoglin") and 
               not string.find(name, "zombie") then
                should_hide = true
            end
        end

        -- 3. Hide specific items forced by user
        if force_hide[name] then
            should_hide = true
        end

        -- APPLY HIDING
        if should_hide then
            local new_groups = table.copy(def.groups or {})
            new_groups.creative_inventory = 0
            new_groups.not_in_creative_inventory = 1
            
            minetest.override_item(name, {
                groups = new_groups,
                description = "REMOVED (Pig)"
            })
        end
    end
end)

minetest.register_alias("mcl_mobitems:porkchop",        "mcl_mobitems:mutton")
minetest.register_alias("mcl_mobitems:cooked_porkchop", "mcl_mobitems:cooked_mutton")


--------------------------------------------------------------
-- PART 2: UPGRADE SHEEP (RIDING, VISUALS & DROPS)
--------------------------------------------------------------

local sheep_def = minetest.registered_entities["mobs_mc:sheep"]

if sheep_def then
    -- 1. Enable steering
    sheep_def.steer_class = "follow_item" 
    sheep_def.steer_item = "group:controls_pig" 
    sheep_def._csm_driving_enabled = true
    
    -- 2. Add Carrot on a Stick to follow list
    if not sheep_def.follow then sheep_def.follow = {} end
    table.insert(sheep_def.follow, "mcl_mobitems:carrot_on_a_stick")
    table.insert(sheep_def.follow, "mcl_mobitems:carrot_on_a_stick_enchanted")

    -- HELPER: Apply Saddle Texture (Shifted -7, 0)
    local function apply_saddle_texture(self)
        local props = self.object:get_properties()
        if props and props.textures then
            local current_tex = props.textures[1]
            local saddle_file = "mobs_mc_pig_saddle.png"
            
            if string.find(current_tex, "mobs_mc_pig_saddle.png") then return end
            
            -- EXACT MODIFIER: -7 left, 0 right (No wrapping)
            local shift_modifier = "^[combine:64x32:-7,0=" .. saddle_file .. ":0,0=" .. saddle_file
            
            local new_textures = props.textures
            new_textures[1] = current_tex .. shift_modifier
            self.object:set_properties({textures = new_textures})
        end
    end

    -- 3. Inject "do_custom" for Visual Updates
    local old_do_custom = sheep_def.do_custom
    sheep_def.do_custom = function(self, dtime)
        if old_do_custom then old_do_custom(self, dtime) end
        if self.saddle == "yes" then
            apply_saddle_texture(self)
        end
    end
    
    -- 4. Drop Saddle on Death
    local old_on_die = sheep_def.on_die
    sheep_def.on_die = function(self, pos)
        if old_on_die then old_on_die(self, pos) end
        if self.saddle == "yes" then
            minetest.add_item(self.object:get_pos(), "mcl_mobitems:saddle")
        end
    end

    local original_sheep_rightclick = sheep_def.on_rightclick

    -- 5. Override Right-Click
    sheep_def.on_rightclick = function(self, clicker)
        if not clicker or not clicker:is_player() then return end
        
        local item = clicker:get_wielded_item()
        local item_name = item:get_name()
        local player_name = clicker:get_player_name()

        -- LOGIC A: BOOST
        if self.driver and clicker == self.driver and
           minetest.get_item_group(item_name, "controls_pig") > 0 then
            local vel = self.object:get_velocity()
            if vel then
                local speed = 8
                local yaw = self.object:get_yaw()
                local new_vel = {x = -math.sin(yaw)*speed, y = vel.y, z = math.cos(yaw)*speed}
                self.object:set_velocity(new_vel)
            end
            if not minetest.is_creative_enabled(player_name) then
                if item:get_wear() > 63000 then
                    minetest.sound_play("default_break_tool", {pos=clicker:get_pos(), gain=0.5}, true)
                    clicker:set_wielded_item("mcl_fishing:fishing_rod")
                else
                    item:add_wear(2521)
                    clicker:set_wielded_item(item)
                end
            end
            return
        end

        -- LOGIC B: MOUNT
        if self.saddle == "yes" and not self.driver and clicker:get_wielded_item():is_empty() then
             if not clicker:get_player_control().sneak then
                self.driver_attach_at = {x = 0.0, y = 8.5, z = -2.0} 
                self.driver_eye_offset = {x = 0, y = 3, z = 0}
                self:attach(clicker)
                return
             end
        end

        -- LOGIC C: APPLY SADDLE
        if item_name == "mcl_mobitems:saddle" and self.saddle ~= "yes" and not self.child then
            self.saddle = "yes"
            self.tamed = true
            if not minetest.is_creative_enabled(player_name) then
                item:take_item()
                clicker:set_wielded_item(item)
            end
            minetest.sound_play({name = "mcl_armor_equip_leather"}, {gain=0.5, pos=self.object:get_pos()}, true)
            apply_saddle_texture(self)
            return
        end

        if original_sheep_rightclick then
            original_sheep_rightclick(self, clicker)
        end
    end
    print("[pigs_to_sheep] Sheep upgraded.")
end


--------------------------------------------------------------
-- PART 3: UPDATE PIGLINS (ACCEPT MUTTON)
--------------------------------------------------------------

local piglin_def = minetest.registered_entities["mobs_mc:piglin"]

if piglin_def then
    local function is_mutton(name)
        return name == "mcl_mobitems:mutton" or name == "mcl_mobitems:cooked_mutton"
    end
    local function piglin_loves_item(itemname)
        return itemname == "mcl_core:gold_ingot" 
            or itemname == "mcl_core:gold_nugget"
            or minetest.get_item_group(itemname, "golden") > 0
    end

    piglin_def.should_pick_up = function(self, stack)
        local name = stack:get_name()
        if self.child and name == "mcl_mobitems:leather" then return false end
        if minetest.get_item_group(name, "soul_firelike") > 0 then return false end
        if self._admire_cooldown > 0 and self.attack then return false end
        if name == "mcl_core:gold_ingot" then
            local offhand = self:get_offhand_item()
            return not piglin_loves_item(offhand:get_name())
        else
            local fits = self:has_inventory_space(stack)
            local is_food = (name == "mcl_mobitems:porkchop" or 
                             name == "mcl_mobitems:cooked_porkchop" or 
                             is_mutton(name))
            if is_food then return fits and self._feed_cooldown == 0
            elseif name == "mcl_core:gold_nugget" then return fits
            elseif piglin_loves_item(name) then
                local offhand = self:get_offhand_item()
                return not piglin_loves_item(offhand:get_name()) and fits
            else return self:evaluate_new_item(stack) and fits end
        end
    end

    piglin_def.default_pickup = function(self, object, stack, def, itemname)
        if not self:should_pick_up(stack) then return true end
        self:cancel_navigation()
        self:halt_in_tracks()
        local taken = stack:take_item(itemname == "mcl_core:gold_nugget" and stack:get_count() or 1)
        local name = taken:get_name()
        
        if piglin_loves_item(name) then
            self:set_offhand_item(taken)
            self._admiring_item = 6.0
            self:replace_activity("_admiring_item", false)
        elseif (name == "mcl_mobitems:porkchop" or 
                name == "mcl_mobitems:cooked_porkchop" or 
                is_mutton(name)) and self._feed_cooldown == 0 then
            self._feed_cooldown = 10
        else
            local equipped = self:try_equip_item(taken, def, itemname)
            if not equipped then
                local remainder = self:add_to_inventory(taken)
                if not remainder:is_empty() then
                    local self_pos = self.object:get_pos()
                    local pos = vector.offset(self_pos, 0, 1.16, 0)
                    local obj = minetest.add_item(pos, remainder)
                    if obj then self:chuck_randomly(self_pos, obj) end
                end
            end
        end
        if stack:is_empty() then object:remove() else object:get_luaentity().itemstring = stack:to_string() end
        return true
    end
end


--------------------------------------------------------------
-- PART 4: SAFE REPLACEMENT (INJECT ON_STEP)
--------------------------------------------------------------

local pigs_to_replace = {
    "mobs_mc:pig",
    "mobs_mc:piglet",
    "mobs_mc:boar",
    "m16b_new_mobs:warm_pig",
    "m16b_new_mobs:cold_pig"
}

local replacement_mob = "mobs_mc:sheep"

for _, pig_name in ipairs(pigs_to_replace) do
    local def = minetest.registered_entities[pig_name]

    if def then
        local old_on_step = def.on_step
        def.on_step = function(self, dtime)
            if old_on_step then old_on_step(self, dtime) end
            if self.object and self.object:get_pos() then
                if self.name == pig_name then
                    local pos = self.object:get_pos()
                    self.object:remove()
                    minetest.add_entity(pos, replacement_mob)
                end
            end
        end
        print("[pigs_to_sheep] Injected safe replacement for: " .. pig_name)
    end
end