--[[

  Nether mod for minetest

  Copyright (C) 2013 PilzAdam

  Permission to use, copy, modify, and/or distribute this software for
  any purpose with or without fee is hereby granted, provided that the
  above copyright notice and this permission notice appear in all copies.

  THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
  WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR
  BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
  OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
  WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
  SOFTWARE.

]]--

-- Set DEBUG_FLAGS to determine the behavior of nether.debug():
--   0 = off
--   1 = print(...)
--   2 = minetest.chat_send_all(...)
--   4 = minetest.log("info", ...)
local DEBUG_FLAGS = 0

local S
if minetest.get_translator ~= nil then
	S = minetest.get_translator("abyss")
else
	-- mock the translator function for MT 0.4
	S = function(str, ...)
		local args={...}
		return str:gsub(
			"@%d+",
			function(match) return args[tonumber(match:sub(2))]	end
		)
	end
end

-- Global Nether namespace
abyss                = {}
abyss.mapgen         = {} -- Shared Nether mapgen namespace, for mapgen files to expose functions and constants
abyss.modname        = minetest.get_current_modname()
abyss.path           = minetest.get_modpath(abyss.modname)
abyss.get_translator = S
                     -- nether.useBiomes allows other mods to know whether they can register ores etc. in the Nether.
                     -- See mapgen.lua for an explanation of why minetest.read_schematic is being checked
abyss.useBiomes      = false -- minetest.get_mapgen_setting("mg_name") ~= "v6" and minetest.read_schematic ~= nil
abyss.fogColor = {	           -- only used if climate_api is installed
	netherCaverns = "#1D0504", -- Distance-fog colour for classic nether
	mantle        = "#070916", -- Distance-fog colour for the Mantle region
	geodes        = "#300530"  -- Distance-fog colour for secondary region
}


-- Settings
abyss.DEPTH_CEILING              =  -5000 -- The y location of the Nether's celing
abyss.DEPTH_FLOOR                = -11000 -- The y location of the Nether's floor
abyss.FASTTRAVEL_FACTOR          =      10 -- 10 could be better value for Minetest, since there's no sprint, but ex-Minecraft players will be mathing for 8
abyss.PORTAL_BOOK_LOOT_WEIGHTING =    0 -- Likelyhood of finding the Book of Portals (guide) in dungeon chests. Set to 0 to disable.
abyss.NETHER_REALM_ENABLED       =   true -- Setting to false disables the Nether and Nether portal


-- Override default settings with values from the .conf file, if any are present.
abyss.FASTTRAVEL_FACTOR          = tonumber(minetest.settings:get("nether_fasttravel_factor") or abyss.FASTTRAVEL_FACTOR)
abyss.PORTAL_BOOK_LOOT_WEIGHTING = tonumber(minetest.settings:get("nether_portalBook_loot_weighting") or abyss.PORTAL_BOOK_LOOT_WEIGHTING)
abyss.NETHER_REALM_ENABLED       = minetest.settings:get_bool("nether_realm_enabled", abyss.NETHER_REALM_ENABLED)
abyss.DEPTH_CEILING              = tonumber(minetest.settings:get("nether_depth_ymax") or abyss.DEPTH_CEILING)
abyss.DEPTH_FLOOR                = tonumber(minetest.settings:get("nether_depth_ymin") or abyss.DEPTH_FLOOR)

if abyss.DEPTH_FLOOR + 1000 > abyss.DEPTH_CEILING then
	error("The lower limit of the Nether must be set at least 1000 lower than the upper limit, and more than 3000 is recommended. Set settingtypes.txt, or 'All Settings' -> 'Mods' -> 'nether' -> 'Nether depth'", 0)
end
abyss.DEPTH = abyss.DEPTH_CEILING -- Deprecated, use abyss.DEPTH_CEILING instead.

-- DEPTH_FLOOR_LAYERS gives the bottom Y of all locations that wish to be
-- considered part of the Nether.
-- DEPTH_FLOOR_LAYERS Allows mods to insert extra layers below the
-- Nether, by knowing where their layer ceiling should start, and letting
-- the layers be included in effects which only happen in the Nether.
-- If a mod wishes to add a layer below the Nether it should read
-- nether.DEPTH_FLOOR_LAYERS to find the bottom Y of the Nether and any
-- other layers already under the Nether. The mod should leave a small gap
-- between DEPTH_FLOOR_LAYERS and its ceiling (e.g. use DEPTH_FLOOR_LAYERS - 6
-- for its ceiling Y, so there is room to shift edge-case biomes), then set
-- nether.DEPTH_FLOOR_LAYERS to reflect the mod's floor Y value, and call
-- shift_existing_biomes() with DEPTH_FLOOR_LAYERS as the floor_y argument.
abyss.DEPTH_FLOOR_LAYERS = abyss.DEPTH_FLOOR

-- A debug-print function that understands vectors etc. and does not
-- evaluate when debugging is turned off.
-- Works like string.format(), treating the message as a format string.
-- nils, tables, and vectors passed as arguments to nether.debug() are
-- converted to strings and can be included inside the message with %s
function abyss.debug(message, ...)

	local args = {...}
	local argCount = select("#", ...)

	for i = 1, argCount do
		local arg = args[i]
		if arg == nil then
			-- convert nils to strings
			args[i] = '<nil>'
		elseif type(arg) == "table" then
			local tableCount = 0
			for _,_ in pairs(arg) do tableCount = tableCount + 1 end
			if tableCount == 3 and arg.x ~= nil and arg.y ~= nil and arg.z ~= nil then
				-- convert vectors to strings
				args[i] = minetest.pos_to_string(arg)
			else
				-- convert tables to strings
				-- (calling function can use dump() if a multi-line listing is desired)
				args[i] = string.gsub(dump(arg, ""), "\n", " ")
			end
		end
	end

	local composed_message = "abyss: " .. string.format(message, unpack(args))

	if math.floor(DEBUG_FLAGS / 1) % 2 == 1 then print(composed_message) end
	if math.floor(DEBUG_FLAGS / 2) % 2 == 1 then minetest.chat_send_all(composed_message) end
	if math.floor(DEBUG_FLAGS / 4) % 2 == 1 then minetest.log("info", composed_message) end
end
if DEBUG_FLAGS == 0 then
	-- do as little evaluation as possible
	abyss.debug = function() end
end


-- Load files
dofile(abyss.path .. "/portal_api.lua")
dofile(abyss.path .. "/nodes.lua")
dofile(abyss.path .. "/mapgen_nobiomes.lua")


-- Portals are ignited by right-clicking with a mese crystal fragment
abyss.register_portal_ignition_item(
	"more_mese:mese_crystal_fragment_red",
	{name = "nether_portal_ignition", gain = 0.3}
)


if abyss.NETHER_REALM_ENABLED then
	-- Use the Portal API to add a portal type which goes to the Nether
	-- See portal_api.txt for documentation
	abyss.register_portal("abyss_portal", {
		shape               = abyss.PortalShape_Traditional,
		frame_node_name     = "default:obsidian",
		wormhole_node_color = 0, -- 0 is magenta
		title = S("Nether Portal"),
		book_of_portals_pagetext = S([[Construction requires 14 blocks of obsidian, which we found deep underground where water had solidified molten rock. The finished frame is four blocks wide, five blocks high, and stands vertically, like a doorway.

This opens to a truly hellish place, though for small mercies the air there is still breathable. There is an intriguing dimensional mismatch happening between this realm and ours, as after opening the second portal into it we observed that 10 strides taken in the Nether appear to be an equivalent of @1 in the natural world.

The expedition parties have found no diamonds or gold, and after an experienced search party failed to return from the trail of a missing expedition party, I must conclude this is a dangerous place.]], 10 * abyss.FASTTRAVEL_FACTOR),

		is_within_realm = function(pos) -- return true if pos is inside the Nether
			return pos.y < abyss.DEPTH_CEILING and pos.y > abyss.DEPTH_FLOOR
		end,

		find_realm_anchorPos = function(surface_anchorPos, player_name)
			-- divide x and z by a factor of 8 to implement Nether fast-travel
			local destination_pos = vector.divide(surface_anchorPos, abyss.FASTTRAVEL_FACTOR)
			destination_pos.x = math.floor(0.5 + destination_pos.x) -- round to int
			destination_pos.z = math.floor(0.5 + destination_pos.z) -- round to int
			destination_pos.y = abyss.DEPTH_CEILING - 1 -- temp value so find_nearest_working_portal() returns nether portals

			-- a y_factor of 0 makes the search ignore the altitude of the portals (as long as they are in the Nether)
			local existing_portal_location, existing_portal_orientation =
				abyss.find_nearest_working_portal("abyss_portal", destination_pos, 8, 0)

			if existing_portal_location ~= nil then
				return existing_portal_location, existing_portal_orientation
			else
				local start_y = abyss.DEPTH_CEILING - math.random(500, 1500) -- Search starting altitude
				destination_pos.y = abyss.find_nether_ground_y(destination_pos.x, destination_pos.z, start_y, player_name)
				return destination_pos
			end
		end,

		find_surface_anchorPos = function(realm_anchorPos, player_name)
			-- A portal definition doesn't normally need to provide a find_surface_anchorPos() function,
			-- since find_surface_target_y() will be used by default, but Nether portals also scale position
			-- to create fast-travel.
			-- Defining a custom function also means we can look for existing nearby portals.

			-- Multiply x and z by a factor of 8 to implement Nether fast-travel
			local destination_pos = vector.multiply(realm_anchorPos, abyss.FASTTRAVEL_FACTOR)
			destination_pos.x = math.min(30900, math.max(-30900, destination_pos.x)) -- clip to world boundary
			destination_pos.z = math.min(30900, math.max(-30900, destination_pos.z)) -- clip to world boundary
			destination_pos.y = abyss.DEPTH_CEILING + 1 -- temp value so find_nearest_working_portal() doesn't return nether portals

			-- a y_factor of 0 makes the search ignore the altitude of the portals (as long as they are outside the Nether)
			local existing_portal_location, existing_portal_orientation =
				abyss.find_nearest_working_portal("abyss_portal", destination_pos, 8 * abyss.FASTTRAVEL_FACTOR, 0)

			if existing_portal_location ~= nil then
				return existing_portal_location, existing_portal_orientation
			else
				destination_pos.y = abyss.find_surface_target_y(destination_pos.x, destination_pos.z, "abyss_portal", player_name)
				return destination_pos
			end
		end,

		on_ignite = function(portalDef, anchorPos, orientation)

			-- make some sparks fly
			local p1, p2 = portalDef.shape:get_p1_and_p2_from_anchorPos(anchorPos, orientation)
			local pos = vector.divide(vector.add(p1, p2), 2)

			local textureName = portalDef.particle_texture
			if type(textureName) == "table" then textureName = textureName.name end

			minetest.add_particlespawner({
				amount = 110,
				time   = 0.1,
				minpos = {x = pos.x - 0.5, y = pos.y - 1.2, z = pos.z - 0.5},
				maxpos = {x = pos.x + 0.5, y = pos.y + 1.2, z = pos.z + 0.5},
				minvel = {x = -5, y = -1, z = -5},
				maxvel = {x =  5, y =  1, z =  5},
				minacc = {x =  0, y =  0, z =  0},
				maxacc = {x =  0, y =  0, z =  0},
				minexptime = 0.1,
				maxexptime = 0.5,
				minsize = 0.2 * portalDef.particle_texture_scale,
				maxsize = 0.8 * portalDef.particle_texture_scale,
				collisiondetection = false,
				texture = textureName .. "^[colorize:#F4F:alpha",
				animation = portalDef.particle_texture_animation,
				glow = 8
			})

		end

	})

end -- end of "if nether.NETHER_REALM_ENABLED..."

