local storage = minetest.get_mod_storage()
local S = minetest.get_translator("atl_server_filter")

local banned_words = {}
for key, value in pairs(storage:to_table().fields) do
    banned_words[key] = value
end
local function log_warning(message)
    minetest.log("warning", message)
end

local function log_info(message)
    minetest.log("info", message)
end

local function add_banned_word(word)
    local uppercase_word = word:upper()
    banned_words[word] = uppercase_word
    storage:set_string(word, uppercase_word)
    log_info("Banned word added: " .. word)
end

local function remove_banned_word(word)
    if banned_words[word] then
        banned_words[word] = nil
        storage:set_string(word, "")
        log_info("Banned word removed: " .. word)
        return true
    else
        log_warning("Attempted to remove non-existent banned word: " .. word)
        return false
    end
end

local function contains_banned_word(text)
    local lower_text = text:lower()
    for word, _ in pairs(banned_words) do
        if string.find(lower_text, "%f[%a]" .. word .. "%f[%A]") then
            return word
        end
    end
    return false
end

minetest.register_chatcommand("ban_word", {
    params = "<word>",
    description = S("Ban a word from being used in chat messages."),
    privs = {ban = true},
    func = function(name, param)
        if param == "" then
            return false, S("[ ! ]  Please specify a word to ban.")
        end

        local word = param:lower()
        if banned_words[word] then
            return false, S("[ ! ]  The word '") .. word .. S("' is already banned.")
        end

        add_banned_word(word)
        return true, S("[ ! ]  The word '") .. word .. S("' has been banned.")
    end,
})

minetest.register_chatcommand("ban_word_list", {
    description = "List all banned words.",
    privs = {ban = true},
    func = function(name)
        if next(banned_words) == nil then
            return true, S("[ ! ]  No words are currently banned.")
        end

        local word_list = {}
        for word, _ in pairs(banned_words) do
            table.insert(word_list, word)
        end

        return true, S("[ ! ]  Banned words: ") .. table.concat(word_list, ", ")
    end,
})

minetest.register_chatcommand("ban_word_remove", {
    params = "<word>",
    description = "Remove a word from the banned words list.",
    privs = {ban = true},
    func = function(name, param)
        if param == "" then
            return false, S("[ ! ]  Please specify a word to remove.")
        end

        local word = param:lower()
        if remove_banned_word(word) then
            return true, S("[ ! ]  The word '") .. word .. S("' has been removed from the banned list.")
        else
            return false, S("[ ! ]  The word '") .. word .. S("' is not in the banned list.")
        end
    end,
})

minetest.register_on_chat_message(function(name, message)
    local banned_word = contains_banned_word(message)
    if banned_word then
        minetest.chat_send_player(name, minetest.colorize("red", S("[ ! ]  Your previous message was deleted due to the word '") .. banned_word .. S("' being banned.")))
        log_warning("Player '" .. name .. "' used a banned word: " .. banned_word)
        return true
    end
end)

minetest.register_on_prejoinplayer(function(name, ip)
    local banned_word = contains_banned_word(name)
    if banned_word then
        log_warning("Player '" .. name .. "' tried to join with a banned word in their name: " .. banned_word)
        return S("[ ! ]  Your username contains a banned word: '@1'. Please change it and try again.", banned_word)
    end
end)
