-- 1. Table des intervalles personnalisés
local spike_intervals = {
    ["3d_spikes:wood_spikes"]    = 0.5,
    ["3d_spikes:stone_spikes"]   = 0.4,
    ["3d_spikes:steel_spikes"]   = 0.3,
    ["3d_spikes:gold_spikes"]    = 0.1,
    ["3d_spikes:mese_spikes"]    = 0.25,
    ["3d_spikes:diamond_spikes"] = 0.25,
}

-- 2. Fonction utilitaire pour démarrer le timer correct
local function start_spike_timer(pos, node)
    local interval = spike_intervals[node.name] or 1.0
    minetest.get_node_timer(pos):start(interval)
end

-- 3. Boucle d'enregistrement des piques
local spike_defs = {
    wood_spikes = {damage = 1,   texture = "default_wood.png",       sounds = default.node_sound_wood_defaults(),    group = {choppy=2, oddly_breakable_by_hand=1}},
    stone_spikes = {damage = 2,  texture = "default_stone.png",      sounds = default.node_sound_stone_defaults(),    group = {cracky=2}},
    steel_spikes = {damage = 4,  texture = "default_steel_block.png", sounds = default.node_sound_metal_defaults(),    group = {cracky=2}},
    gold_spikes = {damage = 1,   texture = "default_gold_block.png",  sounds = default.node_sound_metal_defaults(),    group = {cracky=2}},
    mese_spikes = {damage = 5,   texture = "default_mese_block.png",  sounds = default.node_sound_stone_defaults(),    group = {cracky=3}},
    diamond_spikes = {damage = 6,texture = "default_diamond_block.png",sounds = default.node_sound_stone_defaults(),   group = {cracky=3}},
}

for name, def in pairs(spike_defs) do
    local nodename = "3d_spikes:"..name
    minetest.register_node(nodename, {
        description = def.texture:match("default_(.-)%.png"):gsub("^%l", string.upper).." Spikes",
        drawtype = "mesh",
        mesh = "spike.obj",
        tiles = {def.texture},
        inventory_image = def.texture.."^[mask:spike_inv_mask.png",
        wield_image = def.texture.."^[mask:spike_inv_mask.png",
        paramtype = "light",
        sunlight_propagates = true,
        visual_scale = 0.5,
        groups = def.group,
        walkable = true,
        selection_box = {
            type = "fixed",
            fixed = {-0.35, -0.5, -0.35, 0.35, 0.0, 0.35},
        },
        collision_box = {
            type = "fixed",
            fixed = {-0.35, -0.5, -0.35, 0.35, 0.0, 0.35},
        },
        sounds = def.sounds,

        -- démarre le timer quand le node est construit
        on_construct = function(pos)
            start_spike_timer(pos, minetest.get_node(pos))
        end,

        -- quand le timer expire, inflige les dégâts et relance automatiquement
        on_timer = function(pos)
            local node = minetest.get_node(pos)
            local objs = minetest.get_objects_inside_radius(pos, 0.6)
            for _, obj in ipairs(objs) do
                if obj:is_player() or obj:get_luaentity() then
                    obj:punch(obj, 1.0, {
                        full_punch_interval = 1.0,
                        damage_groups = {fleshy = def.damage},
                    }, nil)
                    minetest.sound_play("spike_hit", {pos = pos, max_hear_distance = 8, gain = 1.0})
                    minetest.add_particlespawner({
                        amount = 8, time = 0.1,
                        minpos = {x=pos.x-0.1, y=pos.y+0.2, z=pos.z-0.1},
                        maxpos = {x=pos.x+0.1, y=pos.y+0.4, z=pos.z+0.1},
                        minvel = {x=-2, y=2, z=-2}, maxvel = {x=2, y=4, z=2},
                        minacc = {x=0, y=-6, z=0}, maxacc = {x=0, y=-9, z=0},
                        minexptime = 0.2, maxexptime = 0.4,
                        minsize = 2, maxsize = 4,
                        texture = "blood_particle.png", glow = 5,
                    })
                end
            end
            -- relance le timer avec le même intervalle
            return spike_intervals[nodename]
        end,
    })
end

-- 4. LBM pour démarrer les timers des piques générés par la map
local function table_keys(t)
    local keys = {}
    for k, _ in pairs(t) do
        table.insert(keys, k)
    end
    return keys
end

minetest.register_lbm({
    label = "Start spike timers (mapgen)",
    name = "3d_spikes:start_spike_timer",
    nodenames = table_keys(spike_intervals),
    run_at_every_load = true,
    action = function(pos, node)
        start_spike_timer(pos, node)
    end,
})
