-- Helper function wrapper for bulk module loading with an option to share local
-- variables between them via loadfile.
function voxelgarden.get_include_helper(modname)
	modname = modname or core.get_current_modname()
	-- * `files` is a table containing base filenames without ".lua"
	-- * `basedir` is a table listing out the directories that have to be traversed
	--   to get to the files
	-- * `...` is everything you want to share with the loaded files (vararg)
	return function(files, basedir, ...)
		for _, file in ipairs(files) do
			loadfile(table.concat({
				core.get_modpath(modname),
				table.concat(basedir, DIR_DELIM),
				file .. ".lua"
			}, DIR_DELIM))(...)
		end
	end
	-- Usage example:
	--	local S = core.get_translator("mymod")
	--	local include = voxelgarden.get_include_helper("mymod")
	--	local modules = {"common", "api", "register"}
	--	include(modules, {"src"}, S)
	--		--> src/common.lua, src/api.lua, src/register.lua
	--		--> Inside the files:
	--		-->	local S = ... -- fetch variable(-s) from the loader
end

-- Call `on_rightclick` of the pointed node if defined.
-- Should be compatible with `mcl_util.call_on_rightclick`.
function voxelgarden.call_on_rightclick(itemstack, player, pointed_thing)
	-- Only proceed if the following conditions are met:
	--  * `pointed_thing` isn't nil;
	--  * `pointed_thing` is a node;
	--  * `player` isn't nil;
	--  * `player` is a valid ObjectRef to a player.
	--  * `player` isn't sneaking (sneak-place feature).
	if (not pointed_thing) or (pointed_thing.type ~= "node")
			or (
				player and player:is_player()
				and player:get_player_control().sneak
			) then
		return
	end

	-- If the pointed node isn't loaded (WTF?), don't proceed either.
	local node = core.get_node_or_nil(pointed_thing.under)
	if not node then return end

	-- Finally, call `on_rightclick` if defined.
	-- Return the original itemstack if the pointed node's
	-- `on_rightclick` doesn't return anything.
	local def = core.registered_nodes[node.name]
	if def and def.on_rightclick then
		return def.on_rightclick(pointed_thing.under, node, player, itemstack, pointed_thing)
			or itemstack
	end
end

-- An alternative `on_place` callback for pillar-like "facedir" nodes, for which
-- it isn't desirable to have 3 additional mirror orientations that may affect
-- shading or just be annoying (i.e. when working with a screwdriver).
-- The placed node's direction reflects the axis the placer is looking in
-- (Y for up/down, X for left/right, Z for front/back).
-- Should be compatible with `mcl_util.rotate_axis`.
function voxelgarden.place_axisdir(itemstack, placer, pointed_thing)
	local rc = voxelgarden.call_on_rightclick(itemstack, placer, pointed_thing)
	if rc then return rc end

	local param2
	if pointed_thing.above.y ~= pointed_thing.under.y then -- Y+ or Y-
		param2 = 0
	elseif pointed_thing.above.x ~= pointed_thing.under.x then -- X+ or X-
		param2 = 12
	elseif pointed_thing.above.z ~= pointed_thing.under.z then -- Z+ or Z-
		param2 = 6
	end

	-- The engine will do any other necessary checks for us.
	return core.item_place_node(itemstack, placer, pointed_thing, param2)
end

-- Register an alias for a possibly defined itemstring without emitting warnings.
function voxelgarden.safe_alias(from, to)
	if core.registered_items[from] then
		return
	end
	return core.register_alias(from, to)
end
