-- Local wrapping functions for on_use reuse
local function do_fill(...)
	return bottles.fill(...)
end

local function do_spill(...)
	return bottles.spill(...)
end

-- Local map for liquid category to int for simple comparison
local liquid_map = {
	none = 0,
	source = 1,
	flowing = 2,
}

-- Globals
bottles = {
	-- Settings
	settings = {
		node_fill_limit = 10,
		liquid_fill_unlimited = 1, -- "all"
	},

	-- Registry of filled bottles
	registered_filled_bottles = {},

	-- Index target nodes to their filled bottle types
	target_node_map = {},

	-- Cache node liquid statuses for special liquid handling
	target_liquid_map = {},
}

-- Play a sound whenever a bottle is filled or spilled
function bottles.play_bottle_sound(pos, sound)
	if not sound then return end
	return core.sound_play(sound, {
		pos = pos,
		gain = 0.25,
		pitch = 5.0,
		max_hear_distance = 12,
	}, true)
end

-- Fill an empty glass bottle
function bottles.fill(itemstack, placer, target)
	if target.type ~= "node" or not placer:is_player() then return itemstack, nil end

	-- Get targeted node
	local pos = target.under
	local node = core.get_node(pos)
	local filled_bottle = bottles.target_node_map[node.name]
	if not filled_bottle then return itemstack, nil end

	-- Play contents sound
	bottles.play_bottle_sound(placer:get_pos(), filled_bottle.sound)

	-- Subtract from stack of empty bottles
	local count = itemstack:get_count()
	local retval
	if count == 1 then
		itemstack:clear()
		retval = ItemStack(filled_bottle.name)
	else
		itemstack:set_count(count - 1)
		retval = itemstack
		local leftover = placer:get_inventory():add_item(
			"main",
			ItemStack(filled_bottle.name)
		)
		if not leftover:is_empty() then
			core.add_item(placer:get_pos(),leftover)
		end
	end

	-- Set filled node metadata if enabled; special handling for liquids
	local liquid = bottles.target_liquid_map[node.name]
	if bottles.settings.node_fill_limit > 0
			and liquid < bottles.settings.liquid_fill_unlimited then
		local meta = core.get_meta(pos)
		local limit = meta:get_int("bottles.node_fill_limit")
		limit = limit + 1
		if limit >= bottles.settings.node_fill_limit then
			core.set_node(pos, {name = filled_bottle.replacement})
		else
			meta:set_int("bottles.node_fill_limit", limit)
		end
	end

	-- Return value
	return retval, filled_bottle.name
end

-- Spill the contents out of a filled bottle
function bottles.spill(itemstack, placer)
	if not placer:is_player() then return itemstack end

	-- Play contents sound
	local bdef = bottles.registered_filled_bottles[itemstack:get_name()]
	if bdef and bdef.sound then
		bottles.play_bottle_sound(
			placer:get_pos(),
			bdef.sound
		)
	end

	-- Subtract from stack of filled bottles and set return value
	local count = itemstack:get_count()
	local retval
	if count == 1 then
		itemstack:clear()
		retval = ItemStack("vessels:glass_bottle")
	else
		itemstack:set_count(count - 1)
		retval = itemstack
		local leftover = placer:get_inventory():add_item("main", ItemStack("vessels:glass_bottle"))
		if not leftover:is_empty() then
			core.add_item(placer:get_pos(), leftover)
		end
	end

	return retval
end

-- Register a filled bottle
function bottles.register_filled_bottle(spec)
	-- Validate spec, and set defaults
	if not spec.target then
		return false
	end

	local target_type = type(spec.target)

	if not spec.contents then
		if target_type == "string" then
			spec.contents = spec.target
		elseif target_type == "table" then
			spec.contents = spec.target[1]
		else
			return false
		end
	end

	local contents_node = core.registered_nodes[spec.contents]
	if not contents_node then
		return false
	end

	spec.description = spec.description or (contents_node.description:split("\n")[1] .. " Bottle")

	spec.replacement = spec.replacement or "air"

	if type(contents_node.tiles[1]) == "string" then
		spec.image = contents_node.tiles[1]
	elseif type(contents_node.tiles[1]) == "table" then
		spec.image = contents_node.tiles[1].name
	elseif not spec.image then
		return false
	end
	spec.image = "[combine:16x16:0,0=" .. spec.image .. "^vessels_glass_bottle_mask.png^[makealpha:0,254,0"
	spec.name = "bottles:" .. (spec.name or ("bottle_of_" .. contents_node.name:split(":")[2]))

	-- Ensure that name is not already in use, fail registration if so
	if bottles.registered_filled_bottles[spec.name] then
		return false
	end

	-- Normalize target type
	if target_type == "string" then
		spec.target = {spec.target}
	end

	-- Ensure that target nodes exist and are not already in use, fail registration if so
	-- Also capture brightest light from target nodes which the filled bottle will inherit
	local max_light = 0
	for _, target in ipairs(spec.target) do
		local tnode = core.registered_nodes[target]
		if not tnode or bottles.target_node_map[target] then
			return false
		end
		if (tnode.light_source or 0) > max_light then
			max_light = tnode.light_source
		end
	end

	-- Extract node footstep sounds if possible and if no other sounds are provided
	if not spec.sound and contents_node.sounds and contents_node.sounds.footstep then
		spec.sound = contents_node.sounds.footstep.name
	end

	-- Map target nodes and liquid status to spec
	for _, target in ipairs(spec.target) do
		bottles.target_node_map[target] = spec
		bottles.target_liquid_map[target] = liquid_map[core.registered_nodes[target].liquidtype or "none"]
	end

	-- Put bottle into map of registered filled bottles
	bottles.registered_filled_bottles[spec.name] = spec

	-- Register new bottle node
	core.register_node(":" .. spec.name,{
		description = spec.description,
		drawtype = "plantlike",
		tiles = {spec.image},
		use_texture_alpha = "blend",
		inventory_image = spec.image,
		wield_image = spec.image,
		paramtype = "light",
		light_source = max_light > 0 and max_light or nil,
		is_ground_content = false,
		walkable = false,
		selection_box = {
			type = "fixed",
			fixed = {-0.25, -0.5, -0.25, 0.25, 0.3, 0.25}
		},
		groups = {vessel = 1, dig_immediate = 3, attached_node = 1},
		sounds = default.node_sound_glass_defaults(),
		on_use = do_spill
	})

	-- Successful registration
	return true
end

-- Unregister a filled bottle
function bottles.unregister_filled_bottle(name)
	-- Only unregister if it's already registered
	local spec = bottles.registered_filled_bottles[name]
	if not spec then
		return false
	end

	-- Remove from registered bottles list and target index
	bottles.registered_filled_bottles[name] = nil
	for _, target in ipairs(spec.target) do
		bottles.target_node_map[target] = nil
	end

	-- Unregister node
	core.unregister_item(name)

	-- Successfully unregistered filled bottle
	return true
end

-- Make empty glass bottles able to 'point and fill'
core.override_item("vessels:glass_bottle", {
	liquids_pointable = true,
	on_use = do_fill,
})

-- Nodes that have been partially filled have a chance to drop nothing when dug
if bottles.settings.node_fill_limit > 0 then
	local old_handle_node_drops = core.handle_node_drops
	function core.handle_node_drops(pos, drops, digger)
		local meta = core.get_meta(pos)
		local limit = meta:get_int("bottles.node_fill_limit")
		if limit <= 0 or math.random(1, bottles.settings.node_fill_limit) > limit then
			return old_handle_node_drops(pos, drops, digger)
		end
	end
end
