-- SPDX-FileCopyrightText: 2024 DS
--
-- SPDX-License-Identifier: Apache-2.0

local settings = {}

local function update_settings()
	settings.rate =
			tonumber(core.settings:get("lavaplop.rate")) or 0.05
	settings.check_radius =
			tonumber(core.settings:get("lavaplop.check_radius")) or 20.0
	settings.check_radius_y_scale =
			tonumber(core.settings:get("lavaplop.check_radius_y_scale")) or 0.5
	settings.check_y_offset =
			tonumber(core.settings:get("lavaplop.check_y_offset")) or 0.0
	settings.check_radius_fallof_range =
			tonumber(core.settings:get("lavaplop.check_radius_fallof_range")) or 5.0
	settings.particle_size_mu =
			tonumber(core.settings:get("lavaplop.particle_size_mu")) or 0.7
	settings.particle_collisiondetection =
			core.settings:get_bool("lavaplop.particle_collisiondetection", true)
	settings.sound_gain_mu =
			tonumber(core.settings:get("lavaplop.sound_gain_mu")) or 0.3
end

update_settings()

local function rand_sym(radius)
	return (math.random() - 0.5) * radius * 2
end

-- used for rate limiting sounds, per playername
local num_sound_slots_used = {}

local function trigger_plop(pos, playername, node, plopperdef)
	local p0 = pos + vector.new(rand_sym(0.5), 0.4 + rand_sym(0.1), rand_sym(0.5))
	core.add_particle({
		pos = p0,
		velocity = vector.new(rand_sym(0.7), 0.8 + rand_sym(0.2), rand_sym(0.7)) * 5,
		acceleration = vector.new(0, -10, 0),
		expirationtime = 1,
		size = (1 + rand_sym(0.5)) * settings.particle_size_mu,
		collisiondetection = settings.particle_collisiondetection,
		playername = playername,
		glow = 10,
		node = node,
		drag = vector.new(1, 1, 1),
	})

	if num_sound_slots_used[playername] < 50 then
		core.sound_play(plopperdef.sound or "lavaplop_plop", {
				gain = (1 + rand_sym(0.3)) * settings.sound_gain_mu,
				pitch = 1 + rand_sym(0.1),
				pos = p0,
				to_player = playername,
			}, true)
		num_sound_slots_used[playername] = num_sound_slots_used[playername] + 1
	end
end

local function try_single_node(pos, playername)
	local node = core.get_node(pos)
	local ndef = core.registered_nodes[node.name]

	if not (ndef and ndef._lavaplop and ndef._lavaplop.plopper) then
		return
	end

	local node_abv = core.get_node(pos:offset(0, 1, 0))
	if node_abv.name ~= "air" then -- TODO: less restrictive check
		return
	end

	trigger_plop(pos, playername, node, ndef._lavaplop.plopper)
end

local accumulated_num_nodes = 0

-- performance counters
local cntr_checks = 0
local cntr_t = 0
local cntr_n = 0
local cntr_perf = 0

core.register_globalstep(function(dtime)
	local t0 = core.get_us_time()

	update_settings()

	local radius = settings.check_radius
	local radius_y = radius * settings.check_radius_y_scale
	local radius_y_scale_reci = 1 / settings.check_radius_y_scale
	local radius_fallof_range_reci = 1 / settings.check_radius_fallof_range
	local y_offset = settings.check_y_offset

	-- find out how many nodes we need to check per player
	local volume = radius * radius_y * radius * 8
	local num_nodes_unrounded = settings.rate * dtime * volume
			+ accumulated_num_nodes
	local num_nodes = math.floor(num_nodes_unrounded)
	accumulated_num_nodes = num_nodes_unrounded - num_nodes

	local players = core.get_connected_players()
	for _, player in ipairs(players) do
		local playername = player:get_player_name()
		local ppos = player:get_pos()
		local min_pos = ppos:offset(-radius, -radius_y + y_offset, -radius):round()
		local max_pos = ppos:offset(radius, radius_y + y_offset, radius):round()

		num_sound_slots_used[playername] =
				(num_sound_slots_used[playername] or 0) * 0.5^dtime

		for _i = 1, num_nodes do
			-- TODO: try using a low discrepancy sequence
			local pos = vector.combine(min_pos, max_pos, math.random)

			local posdiff = pos - ppos
			posdiff.y = 0 -- cylinder
			local dist = posdiff:length()
			local y_dist = math.abs(pos.y + y_offset - ppos.y) * radius_y_scale_reci
			dist = math.max(dist, y_dist) -- cylinder with top and bottom
			local chance = (radius - dist) * radius_fallof_range_reci
			if chance >= 1 or (chance > 0 and chance >= math.random()) then
				try_single_node(pos, playername)
				cntr_checks = cntr_checks + 1
			end
		end
	end

	cntr_t = cntr_t + dtime
	cntr_n = cntr_n + 1
	cntr_perf = cntr_perf + (core.get_us_time() - t0)
end)

local function print_perf() -- luacheck: ignore (unused function)
	core.log("num checks (s^-1): "..(cntr_checks / cntr_t))
	core.log("time per step (ms): "..(cntr_perf / cntr_n * 1.0e-3))
	cntr_checks = 0
	cntr_t = 0
	cntr_n = 0
	cntr_perf = 0
	core.after(2, print_perf)
end
--~ print_perf()

core.register_on_leaveplayer(function(player)
	local playername = player:get_player_name()
	num_sound_slots_used[playername] = nil
end)
