function spears_throw (itemstack, player, pointed_thing)
	local spear = itemstack:get_name() .. '_entity'
	local player_pos = player:get_pos()
	local head_pos = vector.new(player_pos.x, player_pos.y + player:get_properties().eye_height, player_pos.z)
	local direction = player:get_look_dir()
	local throw_pos = vector.add(head_pos, vector.multiply(direction,0.5))
	local pitch = player:get_look_vertical()
	local yaw = player:get_look_horizontal()
	local rotation = vector.new(0, yaw + math.pi/2, pitch + math.pi/6)
	local wear = itemstack:get_wear()
	local pointed_a = pointed_thing.above
	local pointed_b = pointed_thing.under	
	if pointed_thing.type == "node" and vector.distance(pointed_a, throw_pos) < 1 then -- Stick into node
		local node = core.get_node(pointed_b)
		local check_node = spears_check_node(node.name)
		if check_node == nil then
			return false
		elseif check_node == 'cracky' then
			core.sound_play("default_metal_footstep", {pos = pointed_a}, true)
			return false
		elseif check_node == 'sticky' then
			local spear_object = core.add_entity(vector.divide(vector.add(vector.multiply(pointed_a, 2), pointed_b), 3), spear)
			spear_object:set_rotation(rotation)
			spear_object:get_luaentity()._wear = wear
			spear_object:get_luaentity()._stickpos = pointed_b
			core.sound_play("default_place_node", {pos = pointed_a}, true)
			return false
		end
	else -- Throw
		local throw_speed = tonumber(core.settings:get("spears_throw_speed"))
		local player_vel = player:get_velocity()
		local spear_object = core.add_entity(throw_pos, spear)
		spear_object:set_velocity(vector.add(player_vel, vector.multiply(direction, throw_speed)))
		spear_object:set_rotation(rotation)
		core.sound_play("spears_throw", {pos = player_pos}, true)
		spear_object:get_luaentity()._wear = wear
		spear_object:get_luaentity()._stickpos = nil
		spear_object:get_luaentity()._owner = player:get_luaentity()
		return true
	end
end

function spears_set_entity(spear_type, base_damage, toughness)
	local SPEAR_ENTITY={
		initial_properties = {
			physical = false,
			visual = "item",
			visual_size = {x = 0.3, y = 0.3, z = 0.3},
			wield_item = "spears:spear_" .. spear_type,
			collisionbox = {-0.3, -0.3, -0.3, 0.3, 0.3, 0.3},
		},

		on_activate = function (self, staticdata, dtime_s)
			self.object:set_armor_groups({immortal = 1})
		end,
		
		on_punch = function (self, puncher)
			if puncher:is_player() then -- Grab the spear
				local stack = {name='spears:spear_' .. spear_type, wear = self._wear}
				local inv = puncher:get_inventory()
				if inv:room_for_item("main", stack) then
					inv:add_item("main", stack)
					self.object:remove()
				end
			end
		end,

		on_step = function(self, dtime)
			local wear = self._wear
			if wear == nil then
				self.object:remove()
				return false
			end
			local pos = self.object:get_pos()
			local velocity = self.object:get_velocity()
			local speed = vector.length(velocity)
			if self._stickpos ~= nil then -- Spear is stuck
				local node = core.get_node(self._stickpos)
				local check_node = spears_check_node(node.name)
				if check_node ~= 'sticky' then -- Fall when node is removed
					self.object:remove()
					core.add_item(pos, {name='spears:spear_' .. spear_type, wear = wear})
					return false
				end
			else -- Spear is flying
				local direction = vector.normalize(velocity)
				local yaw = core.dir_to_yaw(direction)
				local pitch = math.acos(velocity.y/speed) - math.pi/3
				local spearhead_pos = vector.add(pos, vector.multiply(direction, 0.5))
				self.object:set_rotation({x = 0, y = yaw + math.pi/2, z = pitch})
				-- Hit a target?
				local objects_in_radius = core.get_objects_inside_radius(spearhead_pos, 1)
				for _,object in ipairs(objects_in_radius) do
					if spears_check_target(self, object, spearhead_pos) and object:get_armor_groups().fleshy then
						local damage = (speed + base_damage)^1.15 - 20
						object:punch(self.object, 1.0, {full_punch_interval = 1.0, damage_groups = {fleshy=damage},}, direction)
						self.object:remove()
						core.sound_play("spears_hit", {pos = pos}, true)
						wear = spears_wear(wear, toughness)
						core.add_item(pos, {name='spears:spear_' .. spear_type, wear = wear})
						return true
					end
				end
				-- Hit a node?
				local node = core.get_node(spearhead_pos)
				local check_node = spears_check_node(node.name)
				if check_node == nil then
					self.object:remove()
					core.add_item(pos, {name='spears:spear_' .. spear_type, wear = wear})
				elseif check_node ~= 'through' then
					wear = spears_wear(wear, toughness)
					if wear >= 65535 then
						core.sound_play("default_tool_breaks", {pos = pos}, true)
						self.object:remove()
						core.add_item(pos, {name='defaut:stick'})
						return false
					elseif check_node == 'cracky' then
						core.sound_play("default_metal_footstep", {pos = pos}, true)
						self.object:remove()
						core.add_item(pos, {name='spears:spear_' .. spear_type, wear = wear})
						return false
					elseif check_node == 'sticky' then
						self.object:set_acceleration({x = 0, y = 0, z = 0})
						self.object:set_velocity({x = 0, y = 0, z = 0})
						core.sound_play("default_place_node", {pos = pos}, true)
						self._stickpos = spearhead_pos
						self._wear = wear
					end
				else -- Get drag
					local viscosity = core.registered_nodes[node.name].liquid_viscosity
					local drag_coeff = tonumber(core.settings:get("spears_drag_coeff"))
					local drag = math.max(viscosity, drag_coeff)
					local acceleration = vector.multiply(velocity, -drag)
					acceleration.y = acceleration.y - 10 * ((7 - drag) / 7)
					self.object:set_acceleration(acceleration)
				end
			end
		end,
	}
	return SPEAR_ENTITY
end

function spears_check_node(node_name)
	local node = core.registered_nodes[node_name]
	local cracky_limit = tonumber(core.settings:get("spears_node_cracky_limit"))
	if node == nil then
		return nil
	elseif node.groups.cracky ~= nil and node.groups.cracky < cracky_limit then
		return 'cracky'
	elseif node.walkable and not node.buildable_to then
		return 'sticky'
	else
		return 'through'
	end
end

function spears_wear(initial_wear, toughness)
	if not core.settings:get_bool("creative_mode") then
		local wear = initial_wear + 65535/toughness
		return wear
	else
		local wear = initial_wear
		return wear
	end
end

function spears_check_target(self, object, spearhead_pos)
	if object:get_luaentity() == self then
		return false
	elseif isPointInsideEntity(object, spearhead_pos) then
		-- print(self.name .. " hit " .. object:get_luaentity().name)
		return true
	else
		-- print(self.name .. " missed " .. object:get_luaentity().name)
		return false
	end
end

-- Function to check if a point is inside an entity
function isPointInsideEntity(object, point)
    local pos = object:get_pos()
    local collisionbox = object:get_properties().collisionbox
    local yaw = object:get_yaw()

    -- Adjust position for the center of the collision box
    local center = {
        x = pos.x + (collisionbox[1] + collisionbox[4]) / 2,
        y = pos.y + (collisionbox[2] + collisionbox[5]) / 2,
        z = pos.z + (collisionbox[3] + collisionbox[6]) / 2
    }

    -- Rotate the point around the entity's center
    local rotatedPoint = rotatePoint(point, center, yaw)

    -- Check if the rotated point is inside the collision box
    return rotatedPoint.x >= pos.x + collisionbox[1] and
           rotatedPoint.x <= pos.x + collisionbox[4] and
           rotatedPoint.y >= pos.y + collisionbox[2] and
           rotatedPoint.y <= pos.y + collisionbox[5] and
           rotatedPoint.z >= pos.z + collisionbox[3] and
           rotatedPoint.z <= pos.z + collisionbox[6]
end

-- Function to rotate a point around a center and yaw angle
function rotatePoint(point, center, yaw)
    local cosYaw = math.cos(yaw)
    local sinYaw = math.sin(yaw)

    local translatedX = point.x - center.x
    local translatedZ = point.z - center.z

    local rotatedX = translatedX * cosYaw - translatedZ * sinYaw
    local rotatedZ = translatedX * sinYaw + translatedZ * cosYaw

    return {
        x = rotatedX + center.x,
        y = point.y,
        z = rotatedZ + center.z
    }
end

