-- RUNETHERA - BALANCE SYSTEM
-- Prevents OP spell combinations and rune spam

runethera.balance = {}

-- Rune usage multipliers (each duplicate increases cost)
-- Formula: base_cost * (1 + duplicates * multiplier)
local DUPLICATE_MULTIPLIER = 0.5  -- Each duplicate adds 50% more cost/cooldown

-- OP combination penalties
local OP_COMBINATIONS = {
    -- Piercing + AoE damage is extremely powerful
    {runes = {"pierce", "explosion"}, mana_mult = 3.0, cooldown_mult = 2.5},
    {runes = {"pierce", "meteor"}, mana_mult = 4.0, cooldown_mult = 3.0},
    {runes = {"pierce", "quake"}, mana_mult = 3.5, cooldown_mult = 2.5},
    
    -- Multi-hit + strong effects
    {runes = {"multicast", "meteor"}, mana_mult = 5.0, cooldown_mult = 4.0},
    {runes = {"multicast", "explosion"}, mana_mult = 4.0, cooldown_mult = 3.0},
    {runes = {"split", "meteor"}, mana_mult = 3.0, cooldown_mult = 2.5},
    
    -- Gravity burst + damage is area denial
    {runes = {"gravity_burst", "meteor"}, mana_mult = 4.5, cooldown_mult = 3.5},
    {runes = {"gravity_burst", "explosion"}, mana_mult = 3.5, cooldown_mult = 3.0},
    
    -- Chain reactions
    {runes = {"chain", "lightning"}, mana_mult = 3.0, cooldown_mult = 2.0},
    {runes = {"bounce", "explosion"}, mana_mult = 3.5, cooldown_mult = 2.5},
    
    -- Homing + strong damage
    {runes = {"homing", "meteor"}, mana_mult = 3.5, cooldown_mult = 2.5},
    {runes = {"homing", "lightning"}, mana_mult = 2.5, cooldown_mult = 2.0},
}

-- Calculate duplicate penalty
function runethera.balance.calculate_duplicate_penalty(rune_list)
    local counts = {}
    local max_duplicates = 0
    
    -- Count each rune
    for _, rune_name in ipairs(rune_list) do
        counts[rune_name] = (counts[rune_name] or 0) + 1
        if counts[rune_name] > max_duplicates then
            max_duplicates = counts[rune_name]
        end
    end
    
    -- Calculate total penalty
    local total_penalty = 0
    for rune_name, count in pairs(counts) do
        if count > 1 then
            -- Each duplicate adds penalty
            -- 2 same runes: 1.5x, 3 same: 2.0x, 4 same: 2.5x
            local duplicates = count - 1
            total_penalty = total_penalty + (duplicates * DUPLICATE_MULTIPLIER)
        end
    end
    
    return 1.0 + total_penalty
end

-- Check if spell contains OP combination
function runethera.balance.check_op_combinations(rune_list)
    local max_mana_mult = 1.0
    local max_cooldown_mult = 1.0
    
    -- Check each OP combination
    for _, combo in ipairs(OP_COMBINATIONS) do
        local has_all = true
        
        -- Check if spell contains all runes in this combo
        for _, required_rune in ipairs(combo.runes) do
            local found = false
            for _, spell_rune in ipairs(rune_list) do
                if spell_rune == required_rune then
                    found = true
                    break
                end
            end
            if not found then
                has_all = false
                break
            end
        end
        
        -- If spell has this OP combo, apply penalty
        if has_all then
            if combo.mana_mult > max_mana_mult then
                max_mana_mult = combo.mana_mult
            end
            if combo.cooldown_mult > max_cooldown_mult then
                max_cooldown_mult = combo.cooldown_mult
            end
        end
    end
    
    return max_mana_mult, max_cooldown_mult
end

-- Calculate total spell cost with all penalties
-- Returns adjusted_mana_cost, adjusted_cooldown
function runethera.balance.calculate_spell_cost(rune_list, base_mana, base_cooldown)
    -- 1. Duplicate penalty
    local dup_mult = runethera.balance.calculate_duplicate_penalty(rune_list)
    
    -- 2. OP combination penalty
    local op_mana_mult, op_cd_mult = runethera.balance.check_op_combinations(rune_list)
    
    -- 3. Apply both multipliers
    local final_mana = base_mana * dup_mult * op_mana_mult
    local final_cooldown = base_cooldown * dup_mult * op_cd_mult
    
    -- 4. Round to reasonable values
    final_mana = math.ceil(final_mana)
    final_cooldown = math.ceil(final_cooldown * 10) / 10  -- Round to 0.1
    
    return final_mana, final_cooldown
end

-- Get warning message for OP spell
-- Returns warning string or nil
function runethera.balance.get_op_warning(rune_list)
    local op_mana_mult, op_cd_mult = runethera.balance.check_op_combinations(rune_list)
    
    if op_mana_mult > 2.0 or op_cd_mult > 2.0 then
        return string.format(
            "⚠ OP Combination! Cost ×%.1f, Cooldown ×%.1f",
            op_mana_mult, op_cd_mult
        )
    end
    
    return nil
end

print("[Runethera] Balance system loaded")
