-- ZAUBERBUCH MIT PNG-ANZEIGE + SCROLL
-- Jede Kategorie zeigt Runen-PNGs mit Namen, scrollbar bei > 16 Runen

-- Hilfsfunktion: Runen einer Kategorie (mit Paging)
local function pool_of_type(pname, rtype, page)
    local pool = runethera.get_pool(pname)
    local list = {}
    for rname, unlocked in pairs(pool) do
        if unlocked then
            local def = runethera.runes[rname]
            if def and def.type == rtype then
                table.insert(list, {name=rname, def=def})
            end
        end
    end
    table.sort(list, function(a, b) return a.def.description < b.def.description end)
    
    local per_page = 16  -- 4x4 Grid
    local start_idx = page * per_page + 1
    local page_items = {}
    for i = start_idx, math.min(start_idx + per_page - 1, #list) do
        table.insert(page_items, list[i])
    end
    
    local total_pages = math.max(1, math.ceil(#list / per_page))
    return page_items, #list, total_pages
end

function runethera.get_spellbook_formspec(player)
    local pname = player:get_player_name()
    local data  = runethera.players[pname]
    if not data then return "" end

    data.active_tab    = data.active_tab or "trigger"
    data.category_page = data.category_page or {trigger=0,effect=0,powerup=0,customizable=0}

    local sel    = data.selected_spell or 1
    local spell  = data.spells[sel]
    local max_rs = data.max_rune_slots  or 7
    local max_ss = data.max_spell_slots or 7

    local W, H  = 21, 12
    local fs = "formspec_version[6]size[" .. W .. "," .. H .. "]" ..
               "bgcolor[#0d0b1a;true]"

    -- LINKS: Pool-Panel mit PNG-Runen (0..9.5)
    local px, pw = 0.1, 9.4
    fs = fs .. "box[" .. px .. ",0.1;" .. pw .. "," .. (H-0.2) .. ";#111128]"

    -- Tabs
    local tabs   = {"trigger","effect","powerup","customizable"}
    local tlabel = {"Auslöser","Effekte","Powerups","Anpassbar"}
    local tbg    = {"#1a2a5a","#3a1010","#3a2c00","#2a1040"}
    local tsel   = "#334466"
    local tw     = (pw - 0.1) / 4

    for i, t in ipairs(tabs) do
        local tx  = px + (i-1) * tw + 0.05
        local col = data.active_tab == t and tsel or tbg[i]
        fs = fs .. "box["    .. tx .. ",0.15;" .. (tw-0.05) .. ",0.55;" .. col .. "]"
              .. "button[" .. tx .. ",0.15;" .. (tw-0.05) .. ",0.55;tab_" .. t .. ";" .. tlabel[i] .. "]"
    end

    local at   = data.active_tab
    local pg   = data.category_page[at] or 0
    local items, total, pages = pool_of_type(pname, at, pg)

    fs = fs .. "label[" .. (px+0.1) .. ",0.85;Freiunlocked: " .. total .. " (" .. at .. ")]"
    fs = fs .. "box[" .. px .. ",0.95;" .. pw .. ",8.8;#090918]"

    if total == 0 then
        fs = fs .. "label[" .. (px+0.5) .. ",3.5;Keine Runen freiunlocked!]"
        fs = fs .. "label[" .. (px+0.5) .. ",4.0;Crafte + Linksklick = dauerhaft freischalten]"
    else
        -- 4x4 Grid mit PNG + Namen
        local cell_size = (pw - 0.4) / 4
        for i, item in ipairs(items) do
            local col = (i-1) % 4
            local row = math.floor((i-1) / 4)
            local ix  = px + 0.2 + col * cell_size
            local iy  = 1.1 + row * (cell_size + 0.1)
            
            -- Button with PNG as background
            local btn_size = 1.5
            fs = fs .. "image_button[" .. ix .. "," .. iy .. ";" .. btn_size .. "," .. btn_size .. 
                       ";" .. item.def.texture .. ";rune_sel_" .. item.name .. ";;true;false;" .. 
                       item.def.texture .. "]"
            fs = fs .. "tooltip[rune_sel_" .. item.name .. ";" .. item.def.description .. "\n" ..
                       item.def.mana_cost .. " Mana]"
            
            -- Name (abgeschnitten bei langen Namen)
            local short_name = item.def.description
            if #short_name > 12 then short_name = short_name:sub(1,11) .. "." end
            fs = fs .. "label[" .. (ix + cell_size*0.42) .. "," .. (iy+0.05) .. ";" ..
                       minetest.formspec_escape(short_name) .. "]"
            
            -- Mana-Cost
            fs = fs .. "label[" .. (ix + cell_size*0.42) .. "," .. (iy+0.32) .. ";" ..
                       item.def.mana_cost .. " Mana]"
            
            -- Button (unsichtbar aber klickbar)
            fs = fs .. "button[" .. ix .. "," .. iy .. ";" .. cell_size .. "," .. (cell_size*0.7) ..
                       ";padd_" .. item.name .. ";]"
        end
    end

    -- Paging-Buttons
    if pg > 0 then
        fs = fs .. "button[" .. (px+0.2) .. ",9.85;1.8,0.6;pprev_" .. at .. ";◄ Zurück]"
    end
    if pages > 1 and pg < pages-1 then
        fs = fs .. "button[" .. (px+pw-2.0) .. ",9.85;1.8,0.6;pnext_" .. at .. ";Weiter ►]"
    end
    if pages > 1 then
        fs = fs .. "label[" .. (px+pw/2-0.6) .. ",10.1;Seite " .. (pg+1) .. "/" .. pages .. "]"
    end

    -- MITTE: Aktiver Zauber (PNG-Slots)
    local mx, mw = px + pw + 0.1, 5.0
    fs = fs .. "box[" .. mx .. ",0.1;" .. mw .. "," .. (H-0.2) .. ";#141030]"

    fs = fs .. "label[" .. (mx+0.1) .. ",0.5;Zauber " .. sel .. ":]"
    fs = fs .. "field[" .. (mx+0.1) .. ",0.7;" .. (mw-0.9) .. ",0.6;sname;;" ..
               minetest.formspec_escape(spell.name) .. "]"
    fs = fs .. "button[" .. (mx+mw-0.75) .. ",0.7;0.65,0.6;sname_ok;OK]"

    fs = fs .. "label[" .. (mx+0.1) .. ",1.55;Runen (" .. #spell.runes .. "/" .. max_rs .. "):]"
    
    local ss = 1.2  -- Larger for visibility  -- Slot-Größe
    for i = 1, 12 do
        local c = (i-1) % 2
        local r = math.floor((i-1) / 2)
        local sx = mx + 0.1 + c * (ss + 0.15)
        local sy = 1.9 + r * (ss + 0.15)
        
        if i > max_rs then
            -- Gesperrt
            fs = fs .. "box[" .. sx .. "," .. sy .. ";" .. ss .. "," .. ss .. ";#2a0000]"
            fs = fs .. "label[" .. (sx+0.25) .. "," .. (sy+0.35) .. ";LOCK]"
        elseif spell.runes[i] then
            -- Gefüllter Slot: PNG-Textur zeigen (wie im Inventar!)
            local rd = runethera.runes[spell.runes[i]]
            if rd then
                -- Button with PNG as background (for removal)
                fs = fs .. "image_button[" .. sx .. "," .. sy .. ";" .. ss .. "," .. ss ..
                           ";" .. rd.texture .. ";rrem_" .. i .. ";;true;false;" .. rd.texture .. "]"
                fs = fs .. "tooltip[" .. sx .. "," .. sy .. ";" .. ss .. "," .. ss ..
                           ";Rechtsklick: Entfernen\n" .. rd.description .. "\n" ..
                           rd.mana_cost .. " Mana]"
            end
        else
            -- Leerer Slot
            fs = fs .. "box[" .. sx .. "," .. sy .. ";" .. ss .. "," .. ss .. ";#0a0a22]"
            fs = fs .. "label[" .. (sx+0.42) .. "," .. (sy+0.35) .. ";+]"
        end
    end

    fs = fs .. "box[" .. mx .. ",9.0;" .. mw .. ",2.8;#0a0820]"
    fs = fs .. "label[" .. (mx+0.1) .. ",9.2;REGELN:]"
    fs = fs .. "label[" .. (mx+0.1) .. ",9.5;1. Auslöser ZUERST (Pflicht!)]"
    fs = fs .. "label[" .. (mx+0.1) .. ",9.8;2. Max. 1 Auslöser pro Zauber]"
    fs = fs .. "label[" .. (mx+0.1) .. ",10.1;3. Dann Effekte + Powerups]"
    fs = fs .. "label[" .. (mx+0.1) .. ",10.4;4. Farb-Runen ändern Partikel!]"
    fs = fs .. "label[" .. (mx+0.1) .. ",10.7;5. Split = parallele Projektile]"
    fs = fs .. "label[" .. (mx+0.1) .. ",11.0;6. Kegel = 2s Partikel-Spray]"

    -- RECHTS: Zauber-Liste
    local rx, rw = mx + mw + 0.1, W - (mx + mw + 0.2)
    fs = fs .. "box[" .. rx .. ",0.1;" .. rw .. "," .. (H-0.2) .. ";#100e28]"
    fs = fs .. "label[" .. (rx+0.1) .. ",0.5;Zauber-Liste:]"

    for i = 1, 10 do
        local sy  = 0.8 + (i-1) * 1.1
        local sp2 = data.spells[i]
        if i > max_ss then
            fs = fs .. "box[" .. (rx+0.05) .. "," .. sy .. ";" .. (rw-0.1) .. ",0.95;#200000]"
            fs = fs .. "label[" .. (rx+0.3) .. "," .. (sy+0.3) .. ";Gesperrt]"
        else
            local bg  = i == sel and "#1e4a1e" or "#161626"
            local pre = i == sel and "> " or "  "
            fs = fs .. "box[" .. (rx+0.05) .. "," .. sy .. ";" .. (rw-0.1) .. ",0.95;" .. bg .. "]"
            fs = fs .. "button[" .. (rx+0.05) .. "," .. sy .. ";" .. (rw-0.1) ..
                       ",0.95;ssel_" .. i .. ";" .. pre .. i .. ". " ..
                       minetest.formspec_escape(sp2.name) ..
                       " (" .. #sp2.runes .. "/" .. max_rs .. ")]"
        end
    end

    -- Calculate balanced costs for active spell
    local balanced_info = ""
    if active_spell and active_spell.runes and #active_spell.runes > 0 then
        local balanced_mana, balanced_cd = runethera.balance.calculate_spell_cost(
            active_spell.runes, 
            active_spell.mana_cost or 10, 
            active_spell.cooldown or 1
        )
        
        -- Show if different from base
        if balanced_mana > (active_spell.mana_cost or 10) * 1.2 then
            balanced_info = string.format("⚠ Balanced: %d mana, %.1fs CD", 
                balanced_mana, balanced_cd)
        end
    end
    
    if balanced_info ~= "" then
        fs = fs .. "label[" .. (rx+0.1) .. ",10.9;" .. 
                   minetest.formspec_escape(balanced_info) .. "]"
    end
    
    fs = fs .. "box[" .. (rx+0.05) .. ",11.15;" .. (rw-0.1) .. ",0.65;#001030]"
    fs = fs .. "label[" .. (rx+0.1) .. ",11.4;Mana: " .. math.floor(data.mana) ..
               "/" .. data.max_mana .. "  +" .. data.mana_regen .. "/s]"

    return fs
end

-- FIELDS (unverändert von Original, nur Scroll-Fix)
minetest.register_on_player_receive_fields(function(player, formname, fields)
    if formname ~= "runethera:spellbook" then return end
    local pname = player:get_player_name()
    local data  = runethera.players[pname]
    if not data then return end

    local refresh = false
    data.category_page = data.category_page or {trigger=0,effect=0,powerup=0,customizable=0}

    for _, t in ipairs({"trigger","effect","powerup","customizable"}) do
        if fields["tab_" .. t]   then data.active_tab = t; refresh = true end
        if fields["pnext_" .. t] then data.category_page[t] = (data.category_page[t] or 0) + 1; refresh = true end
        if fields["pprev_" .. t] then data.category_page[t] = math.max(0,(data.category_page[t] or 0)-1); refresh = true end
    end

    for rname in pairs(runethera.runes) do
        if fields["padd_" .. rname] then
            local spell  = data.spells[data.selected_spell]
            local max_rs = data.max_rune_slots or 7
            local def    = runethera.runes[rname]

            if #spell.runes >= max_rs then
                minetest.chat_send_player(pname, "Kein Platz! Upgrade benötigt.")
                refresh=true; break
            end

            if def.type == "trigger" then
                for _, r in ipairs(spell.runes) do
                    if runethera.runes[r] and runethera.runes[r].type == "trigger" then
                        minetest.chat_send_player(pname, "Nur 1 Auslöser pro Zauber!")
                        refresh=true; goto skip_add
                    end
                end
                if #spell.runes > 0 then
                    minetest.chat_send_player(pname, "Auslöser muss an erster Stelle stehen!")
                    refresh=true; goto skip_add
                end
            else
                local has_trigger = false
                for _, r in ipairs(spell.runes) do
                    if runethera.runes[r] and runethera.runes[r].type == "trigger" then
                        has_trigger = true; break
                    end
                end
                if not has_trigger then
                    minetest.chat_send_player(pname, "Zuerst einen Auslöser einsetzen!")
                    refresh=true; goto skip_add
                end
            end

            if runethera.pool_has(pname, rname) then
                table.insert(spell.runes, rname)
                runethera.save_player_data(pname)
            end
            ::skip_add::
            refresh=true; break
        end
    end

    for i = 1, 12 do
        if fields["rrem_" .. i] then
            local spell = data.spells[data.selected_spell]
            if spell.runes[i] then
                table.remove(spell.runes, i)
                runethera.save_player_data(pname)
            end
            refresh=true; break
        end
    end

    for i = 1, 10 do
        if fields["ssel_" .. i] then
            if i <= (data.max_spell_slots or 7) then
                data.selected_spell = i
                runethera.update_hud(player)
            end
            refresh=true; break
        end
    end

    if fields["sname_ok"] and fields["sname"] then
        data.spells[data.selected_spell].name = fields["sname"]
        runethera.update_hud(player)
        refresh=true
    end

    if refresh then
        minetest.show_formspec(pname, "runethera:spellbook",
            runethera.get_spellbook_formspec(player))
    end
end)

-- ZAUBERBUCH ITEM (unverändert von vorher)
minetest.register_tool("runethera:spellbook", {
    description = "Zauberbuch\n\n" ..
                  "Linksklick = Zauber wirken\n" ..
                  "Rechtsklick = Nächster Zauber\n" ..
                  "Schleichen + Rechtsklick = Vorheriger Zauber\n" ..
                  "/spellbook = Zauberbuch öffnen",
    inventory_image = "runethera_spellbook.png",
    wield_image     = "runethera_spellbook.png",
    stack_max = 1,
    groups = {not_repairab=1},

    on_use = function(itemstack, player, pointed_thing)
        local pname = player:get_player_name()
        local data  = runethera.players[pname]
        if not data then return itemstack end

        if (data.spell_cooldown or 0) > 0 then
            minetest.chat_send_player(pname,
                "Cooldown: " .. string.format("%.1f", data.spell_cooldown) .. "s")
            return itemstack
        end

        local spell = data.spells[data.selected_spell]
        if spell and #spell.runes > 0 then
            runethera.cast_spell(player, spell, pointed_thing)
        else
            minetest.chat_send_player(pname,
                "Kein Zauber aktiv! Öffne /spellbook und setze Runen ein.")
        end
        return itemstack
    end,

    on_place = function(itemstack, player, pointed_thing)
        local pname  = player:get_player_name()
        local data   = runethera.players[pname]
        if not data then return itemstack end

        local ctrl   = player:get_player_control()
        local max_ss = data.max_spell_slots or 7

        if ctrl.sneak then
            data.selected_spell = data.selected_spell - 1
            if data.selected_spell < 1 then data.selected_spell = max_ss end
        else
            data.selected_spell = data.selected_spell + 1
            if data.selected_spell > max_ss then data.selected_spell = 1 end
        end

        runethera.update_hud(player)

        local sp = data.spells[data.selected_spell]
        local rune_count = sp and #sp.runes or 0
        minetest.chat_send_player(pname,
            "Zauber " .. data.selected_spell .. ": " ..
            (sp and sp.name or "?") ..
            " (" .. rune_count .. " Runen)")

        return itemstack
    end,

    on_secondary_use = function(itemstack, player, pointed_thing)
        local pname = player:get_player_name()
        local data  = runethera.players[pname]
        if not data then return itemstack end

        local ctrl   = player:get_player_control()
        local max_ss = data.max_spell_slots or 7

        if ctrl.sneak then
            data.selected_spell = data.selected_spell - 1
            if data.selected_spell < 1 then data.selected_spell = max_ss end
        else
            data.selected_spell = data.selected_spell + 1
            if data.selected_spell > max_ss then data.selected_spell = 1 end
        end

        runethera.update_hud(player)
        local sp = data.spells[data.selected_spell]
        minetest.chat_send_player(pname,
            "Zauber " .. data.selected_spell .. ": " ..
            (sp and sp.name or "?") ..
            " (" .. (sp and #sp.runes or 0) .. " Runen)")
        return itemstack
    end,
})

minetest.register_chatcommand("spellbook", {
    description = "Zauberbuch öffnen",
    func = function(name, _)
        local p = minetest.get_player_by_name(name)
        if p then
            minetest.show_formspec(name, "runethera:spellbook",
                runethera.get_spellbook_formspec(p))
        end
    end
})

print("[Runethera] Spellbook mit PNG-Anzeige geladen!")
