-- RUNETHERA - Zauber-System

-- Cooldown pro Zauber berechnen (basierend auf Runen-Anzahl)
local function calc_cooldown(spell, spell_data)
    local base = 0.5  -- Minimum-Cooldown in Sekunden
    for _, rname in ipairs(spell.runes) do
        local rd = runethera.runes[rname]
        if rd then base = base + (rd.cooldown or 0) end
    end
    -- Split erhoehe CD leicht
    if (spell_data.split or 1) > 1 then
        base = base * 1.2
    end
    return math.max(0.5, base)
end

function runethera.cast_spell(player, spell, pointed_thing)
    local pname = player:get_player_name()
    local data  = runethera.players[pname]
    if not data or not spell or #spell.runes == 0 then return false end

    -- Check cooldown (spell_cooldown ist in Sekunden als Float)
    if (data.spell_cooldown or 0) > 0 then
        -- Wird schon in on_use abgefangen, aber sicherheitshalber nochmal
        return false
    end

    -- Mana berechnen
    local total_mana = 0
    for _, rname in ipairs(spell.runes) do
        local rd = runethera.runes[rname]
        if rd then total_mana = total_mana + rd.mana_cost end
    end
    if not runethera.consume_mana(player, total_mana) then return false end

    -- Runen sortieren
    local trigger    = nil
    local effects    = {}
    local powerups   = {}
    local wait_delay = 0

    for _, rname in ipairs(spell.runes) do
        local rd = runethera.runes[rname]
        if rd then
            if rd.type == "trigger" then
                if rd.effect_id == "wait" then
                    wait_delay = wait_delay + (rd.params[1] or 2)
                else
                    trigger = rd
                end
            elseif rd.type == "effect"       then table.insert(effects, rd)
            elseif rd.type == "powerup"      then table.insert(powerups, rd)
            elseif rd.type == "customizable" then table.insert(powerups, rd)
            end
        end
    end

    if not trigger then
        minetest.chat_send_player(pname, "Kein Ausloser im Zauber!")
        runethera.add_mana(player, total_mana)
        return false
    end

    -- Spell-Daten
    local spell_data = {
        power          = 1.0,
        duration       = 5,
        range          = 50,
        split          = 1,
        pierce         = false,
        aoe_radius     = 5,
        speed          = 20,
        particle_color = nil,
    }

    -- Powerups anwenden
    for _, pu in ipairs(powerups) do
        if pu.on_cast then pu.on_cast(player, spell_data) end
    end

    -- Cooldown SETZEN (bevor Zauber gewirkt wird)
    local cd = calc_cooldown(spell, spell_data)
    data.spell_cooldown = cd

    -- Trigger ausfuehren
    local function do_cast()
        local ttype, tdata = trigger.on_cast(player, pointed_thing)
        tdata = tdata or {}

        -- BUGFIX Split: Nur Einzel-Trigger vervielfachen
        local single_triggers = {projectile=true, self=true, touch=true, trap=true}
        local times = 1
        if single_triggers[ttype] then
            times = math.min(spell_data.split or 1, 8)
            -- DEBUG
            if (spell_data.split or 1) > 1 then
                minetest.chat_send_player(player:get_player_name(),
                    "Split aktiv: " .. times .. "x " .. ttype)
            end
        end

        if ttype == "projectile" then
            for i = 1, times do
                runethera.fire_projectile(player, spell_data, effects, i, times)
            end
        elseif ttype == "self" then
            for i = 1, times do
                runethera.cast_self(player, spell_data, effects)
            end
        elseif ttype == "touch" then
            for i = 1, times do
                runethera.cast_touch(player, pointed_thing, spell_data, effects)
            end
        elseif ttype == "trap" then
            for i = 1, times do
                runethera.cast_trap(player, pointed_thing, spell_data, effects)
            end
        elseif ttype == "aoe" then
            local radius = tdata.radius or spell_data.aoe_radius or 5
            runethera.cast_aoe(player, radius, spell_data, effects)
        elseif ttype == "beam" then
            runethera.cast_beam(player, tdata.length or 20, spell_data, effects)
        elseif ttype == "orbit" then
            runethera.cast_orbit(player, tdata.count or 3, spell_data, effects)
        elseif ttype == "chain" then
            runethera.cast_chain(player, pointed_thing, tdata.max_chains or 3, spell_data, effects)
        elseif ttype == "rain" then
            runethera.cast_rain(player, tdata.count or 5, spell_data, effects)
        elseif ttype == "cone" then
            runethera.cast_cone(player, spell_data, effects)
        elseif ttype == "burst" then
            runethera.cast_burst(player, tdata.count or 8, spell_data, effects)
        elseif ttype == "ground_slam" then
            runethera.cast_ground_slam(player, tdata.radius or 4, spell_data, effects)
        else
            runethera.fire_projectile(player, spell_data, effects, 1, 1)
        end
    end

    if wait_delay > 0 then
        minetest.after(wait_delay, do_cast)
    else
        do_cast()
    end

    runethera.update_hud(player)
    return true
end

-- PROJEKTIL
function runethera.fire_projectile(player, spell_data, effects, idx, total)
    local pos = player:get_pos()
    pos.y = pos.y + 1.5
    local dir = player:get_look_dir()

    if total > 1 then
        -- Split: Projektile nebeneinander (nicht rotiert, nur versetzt)
        local offset_distance = (idx - (total+1)/2) * 0.8
        local right = vector.normalize({x = dir.z, y = 0, z = -dir.x})
        pos = vector.add(pos, vector.multiply(right, offset_distance))
        -- Richtung bleibt gleich (keine Rotation!)
    end

    local speed = (spell_data.speed or 20) * (spell_data.speed_mult or 1)
    local obj = minetest.add_entity(pos, "runethera:spell_projectile")
    if obj then
        local ent = obj:get_luaentity()
        if ent then
            ent.owner      = player:get_player_name()
            ent.effects    = effects
            ent.spell_data = spell_data
            ent.pierce     = spell_data.pierce or false
        end
        obj:set_velocity(vector.multiply(dir, speed))
    end
end

-- SELBST
function runethera.cast_self(player, spell_data, effects)
    local pos = player:get_pos()
    minetest.add_particlespawner({
        amount=40, time=0.8,
        minpos=vector.subtract(pos,1), maxpos=vector.add(pos,{x=1,y=2,z=1}),
        minvel={x=-2,y=0,z=-2}, maxvel={x=2,y=4,z=2},
        minexptime=0.4, maxexptime=1,
        minsize=1, maxsize=3,
        texture="runethera_particle_magic.png", glow=12,
    })
    for _, eff in ipairs(effects) do
        if eff.on_cast then eff.on_cast(player, player, {effects=effects, spell_data=spell_data}) end
    end
end

-- BERUEHRUNG
function runethera.cast_touch(player, pointed_thing, spell_data, effects)
    if not pointed_thing or pointed_thing.type == "nothing" then return end
    local target = nil
    local tpos   = nil
    if pointed_thing.type == "object" then
        target = pointed_thing.ref
        tpos   = target:get_pos()
    elseif pointed_thing.type == "node" then
        tpos = pointed_thing.under
    end
    if not tpos then return end
    for _, eff in ipairs(effects) do
        if eff.on_cast then eff.on_cast(player, target or tpos) end
    end
end

-- FALLE
function runethera.cast_trap(player, pointed_thing, spell_data, effects)
    local pos
    if pointed_thing and pointed_thing.type == "node" then
        pos = vector.new(pointed_thing.above)
    else
        pos = vector.round(player:get_pos())
    end
    if minetest.get_node(pos).name ~= "air" then pos.y = pos.y + 1 end

    local stored = {}
    for _, eff in ipairs(effects) do
        table.insert(stored, eff.effect_id or "damage")
    end

    minetest.set_node(pos, {name="runethera:rune_trap_block", param2=0})
    local meta = minetest.get_meta(pos)
    meta:set_string("owner",   player:get_player_name())
    meta:set_string("effects", minetest.serialize(stored))
    minetest.get_node_timer(pos):start(0.2)

    minetest.add_particlespawner({
        amount=8, time=60,
        minpos=pos, maxpos=vector.add(pos,{x=1,y=0.1,z=1}),
        minvel={x=0,y=0.1,z=0}, maxvel={x=0,y=0.3,z=0},
        minexptime=1, maxexptime=2, minsize=1, maxsize=2,
        texture="runethera_particle_magic.png", glow=6,
    })
end

-- FLAECHE (AOE)
function runethera.cast_aoe(player, radius, spell_data, effects)
    local pos = player:get_pos()
    minetest.add_particlespawner({
        amount=60, time=0.5,
        minpos=vector.subtract(pos,radius), maxpos=vector.add(pos,radius),
        minvel={x=-3,y=-3,z=-3}, maxvel={x=3,y=3,z=3},
        minexptime=0.5, maxexptime=1.2, minsize=2, maxsize=5,
        texture="runethera_particle_magic.png", glow=12,
    })
    local objs = minetest.get_objects_inside_radius(pos, radius)
    for _, obj in ipairs(objs) do
        if obj ~= player then
            for _, eff in ipairs(effects) do
                if eff.on_cast then eff.on_cast(player, obj, {effects=effects, spell_data=spell_data}) end
            end
        end
    end
end

-- STRAHL (BEAM)
function runethera.cast_beam(player, length, spell_data, effects)
    local pos = player:get_pos(); pos.y = pos.y + 1.5
    local dir = player:get_look_dir()
    local hit_objs = {}
    for i = 1, length do
        local p = vector.add(pos, vector.multiply(dir, i))
        minetest.add_particle({pos=p, velocity={x=0,y=0,z=0},
            expirationtime=0.3, size=3,
            texture="runethera_particle_magic.png", glow=14})
        local node = minetest.get_node(p)
        if node.name ~= "air" and node.name ~= "ignore" then break end
        local objs = minetest.get_objects_inside_radius(p, 1)
        for _, obj in ipairs(objs) do
            if obj ~= player and not hit_objs[obj] then
                hit_objs[obj] = true
                for _, eff in ipairs(effects) do
                    if eff.on_cast then eff.on_cast(player, obj, {effects=effects, spell_data=spell_data}) end
                end
                if not spell_data.pierce then break end
            end
        end
    end
end

-- ORBITAL
function runethera.cast_orbit(player, count, spell_data, effects)
    local pname = player:get_player_name()
    for i = 1, count do
        local obj = minetest.add_entity(player:get_pos(), "runethera:orbit_projectile")
        if obj then
            local ent = obj:get_luaentity()
            if ent then
                ent.owner        = pname
                ent.effects      = effects
                ent.spell_data   = spell_data
                ent.orbit_angle  = (i-1) * (math.pi * 2 / count)
                ent.orbit_radius = 2.5
            end
        end
    end
end

-- KETTE
function runethera.cast_chain(player, pointed_thing, max_chains, spell_data, effects)
    local cur = player:get_pos()
    if pointed_thing and pointed_thing.type == "object" then
        cur = pointed_thing.ref:get_pos()
    end
    local hit = {}
    for i = 1, max_chains do
        local objs = minetest.get_objects_inside_radius(cur, 8)
        local next_obj = nil
        for _, obj in ipairs(objs) do
            if obj ~= player and not hit[obj] then
                next_obj = obj; hit[obj] = true; break
            end
        end
        if next_obj then
            for _, eff in ipairs(effects) do
                if eff.on_cast then eff.on_cast(player, next_obj) end
            end
            cur = next_obj:get_pos()
        else break end
    end
end

-- REGEN
function runethera.cast_rain(player, count, spell_data, effects)
    local pos = player:get_pos()
    for i = 1, count do
        local tpos = vector.add(pos, {x=math.random(-5,5), y=15, z=math.random(-5,5)})
        minetest.after(i * 0.3, function()
            local obj = minetest.add_entity(tpos, "runethera:spell_projectile")
            if obj then
                local ent = obj:get_luaentity()
                if ent then
                    ent.owner=player:get_player_name()
                    ent.effects=effects
                    ent.spell_data=spell_data
                end
                obj:set_velocity({x=0,y=-20,z=0})
            end
        end)
    end
end

-- KEGEL
function runethera.cast_cone(player, spell_data, effects)
    local dir = player:get_look_dir()
    local ppos = player:get_pos()
    ppos.y = ppos.y + 1.5
    
    local hit_entities = {}
    local hit_count = 0
    local max_hits = 5
    local duration = 2.0
    local range = 5
    local cone_angle = math.pi / 3  -- 60 Grad Öffnungswinkel
    
    -- Particles-Spray über 2 Sekunden
    local function spawn_cone_particles()
        for _ = 1, 12 do  -- 12 Partikel pro Frame
            -- Zufälliger Winkel im Kegel
            local h_angle = (math.random() - 0.5) * cone_angle
            local v_angle = (math.random() - 0.5) * cone_angle * 0.7
            
            -- Richtung berechnen
            local right = {x=dir.z, y=0, z=-dir.x}
            local spray_dir = {
                x = dir.x * math.cos(h_angle) + right.x * math.sin(h_angle),
                y = dir.y + math.sin(v_angle),
                z = dir.z * math.cos(h_angle) + right.z * math.sin(h_angle),
            }
            spray_dir = vector.normalize(spray_dir)
            
            -- Particles-Position zufällig entlang des Strahls
            local dist = math.random() * range
            local ppos_ray = vector.add(ppos, vector.multiply(spray_dir, dist))
            
            -- Farbe aus spell_data
            local color = spell_data.particle_color or {r=150, g=100, b=255}
            local size = (spell_data.particle_size or 1.0) * math.random(0.8, 1.5)
            
            minetest.add_particle({
                pos = ppos_ray,
                velocity = vector.multiply(spray_dir, math.random(2,6)),
                expirationtime = math.random(0.3, 0.8),
                size = size,
                texture = string.format("runethera_particle_magic.png^[colorize:#%02x%02x%02x:180",
                    color.r, color.g, color.b),
                glow = 10,
            })
        end
    end
    
    -- Kollisions-Check (10x pro Sekunde für 2 Sekunden)
    local function check_hits(tick)
        if tick > 20 or hit_count >= max_hits then return end
        
        -- Prüfe Kegel-Bereich
        for i = 1, range do
            local check_pos = vector.add(ppos, vector.multiply(dir, i))
            local objs = minetest.get_objects_inside_radius(check_pos, 1.5 + i * 0.3)
            
            for _, obj in ipairs(objs) do
                if obj ~= player and not hit_entities[obj] then
                    local obj_name = obj:is_player() and obj:get_player_name() or 
                                    (obj:get_luaentity() and obj:get_luaentity().name or "unknown")
                    if obj_name ~= "runethera:spell_projectile" and obj_name ~= player:get_player_name() then
                        hit_entities[obj] = true
                        hit_count = hit_count + 1
                        
                        -- Effects anwenden
                        for _, eff in ipairs(effects) do
                            if eff.on_cast then eff.on_cast(player, obj, {effects=effects, spell_data=spell_data}) end
                        end
                        
                        if hit_count >= max_hits then return end
                    end
                end
            end
        end
        
        minetest.after(0.1, function() check_hits(tick + 1) end)
    end
    
    -- Starte Spray
    for i = 0, 19 do
        minetest.after(i * 0.1, spawn_cone_particles)
    end
    check_hits(0)
end

-- AUSBRUCHend

-- AUSBRUCH
function runethera.cast_burst(player, count, spell_data, effects)
    local pos = vector.add(player:get_pos(), {x=0,y=1.5,z=0})
    for i = 1, count do
        local angle = (i-1) * (math.pi*2 / count)
        local dir   = {x=math.cos(angle), y=0, z=math.sin(angle)}
        local obj   = minetest.add_entity(pos, "runethera:spell_projectile")
        if obj then
            local ent = obj:get_luaentity()
            if ent then
                ent.owner=player:get_player_name()
                ent.effects=effects
                ent.spell_data=spell_data
            end
            obj:set_velocity(vector.multiply(dir, spell_data.speed or 15))
        end
    end
end

-- GROUND_SLAM (Bodenschlag - Schockwelle)
function runethera.cast_ground_slam(player, radius, spell_data, effects)
    local pos = player:get_pos()
    if not pos then return end
    
    -- Schockwellen-Partikel
    for i = 1, 24 do
        local angle = i * math.pi / 12
        local r = radius
        minetest.add_particle({
            pos = {x=pos.x + math.cos(angle)*r, y=pos.y+0.2, z=pos.z + math.sin(angle)*r},
            velocity = {x=math.cos(angle)*5, y=0.5, z=math.sin(angle)*5},
            expirationtime = 0.5,
            size = 4,
            texture = "runethera_particle_magic.png^[colorize:#ffffff:180",
            glow = 14,
        })
    end
    
    minetest.sound_play("tnt_explode", {pos=pos, gain=0.8, pitch=0.8}, true)
    
    -- Damage + Knockback
    local objs = minetest.get_objects_inside_radius(pos, radius)
    local caster_name = player:get_player_name()
    for _, obj in ipairs(objs) do
        local is_caster = (obj:is_player() and obj:get_player_name() == caster_name)
        if obj ~= player and not is_caster then
            -- Effects anwenden
            for _, eff in ipairs(effects) do
                if eff.on_cast then eff.on_cast(player, obj, {effects=effects, spell_data=spell_data}) end
            end
            
            -- Knockback nach außen
            local opos = obj:get_pos()
            if opos then
                local dir = vector.direction(pos, opos)
                dir.y = 0.5  -- Leicht nach oben
                obj:add_velocity(vector.multiply(dir, 6))
            end
        end
    end
end

-- FALLEN-NODE
minetest.register_node("runethera:rune_trap_block", {
    description   = "Rune Trap (2D)",
    drawtype      = "nodebox",
    tiles         = {"runethera_trap.png"},
    paramtype     = "light",
    paramtype2    = "wallmounted",
    walkable      = false,  -- Walk through it
    pointable     = false,
    diggable      = false,
    groups        = {not_in_creative_inventory=1},
    light_source  = 8,
    node_box      = {
        type = "wallmounted",
        wall_top    = {-0.5, 0.45, -0.5, 0.5, 0.5, 0.5},   -- On top of block
        wall_bottom = {-0.5, -0.5, -0.5, 0.5, -0.45, 0.5},
        wall_side   = {-0.5, -0.5, -0.5, -0.45, 0.5, 0.5},
    },
    selection_box = {
        type = "wallmounted",
        wall_top    = {-0.5, 0.45, -0.5, 0.5, 0.5, 0.5},
        wall_bottom = {-0.5, -0.5, -0.5, 0.5, -0.45, 0.5},
        wall_side   = {-0.5, -0.5, -0.5, -0.45, 0.5, 0.5},
    },
    on_construct  = function(pos)
        minetest.get_node_timer(pos):start(0.1)
    end,
    on_timer = function(pos, elapsed)
        -- Prüfe ob jemand AUF dem Block steht (pos.y+1)
        local above = {x=pos.x, y=pos.y+1, z=pos.z}
        local objs = minetest.get_objects_inside_radius(above, 0.7)
        
        for _, obj in ipairs(objs) do
            if obj:is_player() then
                local opos = obj:get_pos()
                -- Player muss auf dem Block stehen (y zwischen pos.y+0.5 und pos.y+1.5)
                if opos and opos.y >= pos.y+0.5 and opos.y <= pos.y+1.5 then
                    local meta     = minetest.get_meta(pos)
                    local owner    = minetest.get_player_by_name(meta:get_string("owner"))
                    local eff_list = minetest.deserialize(meta:get_string("effects")) or {}
                    
                    -- Effects auslösen
                    for _, eid in ipairs(eff_list) do
                        local fn = runethera.effects[eid]
                        if fn then fn(owner or obj, obj, {}) end
                    end
                    
                    -- NUR Partikel, KEIN Explosions-Sound
                    minetest.add_particlespawner({
                        amount=25, time=0.3,
                        minpos=pos, maxpos=vector.add(pos,0.5),
                        minvel={x=-1,y=1,z=-1}, maxvel={x=1,y=2,z=1},
                        minexptime=0.3, maxexptime=0.7,
                        minsize=1, maxsize=2.5,
                        texture="runethera_particle_magic.png", glow=10,
                    })
                    
                    -- Sanfter Trigger-Sound
                    minetest.sound_play("default_place_node", {pos=pos, gain=0.4, pitch=2.5}, true)
                    minetest.remove_node(pos)
                    return false
                end
            end
        end
        return true
    end,
})

-- PROJEKTIL-ENTITY
minetest.register_entity("runethera:spell_projectile", {
    initial_properties = {
        visual="sprite", textures={"runethera_projectile.png"},
        visual_size={x=0.5,y=0.5}, collisionbox={-0.15,-0.15,-0.15,0.15,0.15,0.15},
        physical=false, pointable=false, glow=14,
        collide_with_objects=false,  -- Immun gegen Knockback!
    },
    owner="", effects={}, spell_data={}, pierce=false, timer=0, has_split=false,
    on_activate = function(self) self.object:set_armor_groups({immortal=1}) end,
    on_step = function(self, dtime)
        self.timer = self.timer + dtime
        if self.timer > 8 then self.object:remove(); return end
        
        -- SPLIT-MECHANIK: Nach 0.5s teilt sich das Projektil
        if self.spell_data and self.spell_data.split and 
           self.spell_data.split > 1 and not self.has_split and 
           self.timer > 0.5 then
            self.has_split = true
            local pos = self.object:get_pos()
            local vel = self.object:get_velocity()
            if pos and vel then
                local split_count = math.min(self.spell_data.split - 1, 7)
                
                -- Berechne "rechts"-Vektor (senkrecht zur Flugrichtung)
                local right = vector.normalize({x=vel.z, y=0, z=-vel.x})
                
                for i = 1, split_count do
                    -- Distribute evenly with 10 block spacing
                    -- 4 splits: -15, -5, +5, +15
                    -- 7 splits: -30, -20, -10, 0, +10, +20, +30
                    local offset = (i - (split_count+1)/2) * 10
                    local new_pos = vector.add(pos, vector.multiply(right, offset))
                    
                    local proj = minetest.add_entity(new_pos, "runethera:spell_projectile")
                    if proj then
                        local le = proj:get_luaentity()
                        if le then
                            le.owner = self.owner
                            le.effects = self.effects
                            le.spell_data = self.spell_data
                            le.pierce = self.pierce
                            le.has_split = true  -- Verhindere endloses Splitten
                        end
                        proj:set_velocity(vel)
                    end
                end
                
                -- Particles-Effekt
                minetest.add_particlespawner({
                    amount = 20,
                    time = 0.2,
                    minpos = pos,
                    maxpos = pos,
                    minvel = {x=-3, y=-3, z=-3},
                    maxvel = {x=3, y=3, z=3},
                    minexptime = 0.3,
                    maxexptime = 0.6,
                    minsize = 1,
                    maxsize = 2,
                    texture = "runethera_particle_magic.png",
                    glow = 12,
                })
            end
        end
        local pos = self.object:get_pos()
        if not pos then self.object:remove(); return end
        -- Particles mit Farbe aus spell_data
        local color = self.spell_data and self.spell_data.particle_color
        local size = self.spell_data and self.spell_data.particle_size or 1.0
        local tex = "runethera_particle_magic.png"
        if color then
            tex = string.format("runethera_particle_magic.png^[colorize:#%02x%02x%02x:180",
                color.r or 150, color.g or 100, color.b or 255)
        end
        minetest.add_particle({pos=pos, velocity={x=0,y=0,z=0},
            expirationtime=0.3, size=2*size,
            texture=tex, glow=10})
        local objs = minetest.get_objects_inside_radius(pos, 0.8)
        for _, obj in ipairs(objs) do
            if obj ~= self.object then
                local is_target = false
                if obj:is_player() and obj:get_player_name() ~= self.owner then
                    is_target = true
                elseif obj:get_luaentity() and
                       obj:get_luaentity().name ~= "runethera:spell_projectile" and
                       obj:get_luaentity().name ~= "runethera:orbit_projectile" then
                    is_target = true
                end
                if is_target then
                    local owner_p = minetest.get_player_by_name(self.owner)
                    for _, eff in ipairs(self.effects) do
                        if eff.on_cast then eff.on_cast(owner_p, obj, {effects=effects, spell_data=spell_data}) end
                    end
                    if not self.pierce then self.object:remove(); return end
                end
            end
        end
        local node = minetest.get_node(pos)
        if node.name ~= "air" and node.name ~= "ignore" and
           not node.name:find("runethera:") then
            local owner_p = minetest.get_player_by_name(self.owner)
            for _, eff in ipairs(self.effects) do
                if eff.on_cast then eff.on_cast(owner_p, pos, {effects=effects, spell_data=spell_data}) end
            end
            if not self.pierce then self.object:remove() end
        end
    end,
})

-- ORBITAL-ENTITY
minetest.register_entity("runethera:orbit_projectile", {
    initial_properties = {
        visual="sprite", textures={"runethera_projectile.png"},
        visual_size={x=0.4,y=0.4}, collisionbox={-0.1,-0.1,-0.1,0.1,0.1,0.1},
        physical=false, pointable=false, glow=12,
    },
    owner="", effects={}, spell_data={}, orbit_angle=0, orbit_radius=2.5, timer=0,
    on_activate = function(self) self.object:set_armor_groups({immortal=1}) end,
    on_step = function(self, dtime)
        self.timer = self.timer + dtime
        if self.timer > 20 then self.object:remove(); return end
        local owner = minetest.get_player_by_name(self.owner)
        if not owner then self.object:remove(); return end
        self.orbit_angle = self.orbit_angle + dtime * 2
        local opos = owner:get_pos(); opos.y = opos.y + 1.5
        self.object:set_pos({
            x = opos.x + math.cos(self.orbit_angle) * self.orbit_radius,
            y = opos.y,
            z = opos.z + math.sin(self.orbit_angle) * self.orbit_radius,
        })
        local pos = self.object:get_pos()
        local objs = minetest.get_objects_inside_radius(pos, 0.7)
        for _, obj in ipairs(objs) do
            if obj ~= self.object and obj ~= owner then
                if obj:is_player() or obj:get_luaentity() then
                    for _, eff in ipairs(self.effects) do
                        if eff.on_cast then eff.on_cast(owner, obj) end
                    end
                    self.object:remove(); return
                end
            end
        end
    end,
})

print("[Runethera] Zauber-System geladen!")
