-- Load the biosphere object
dofile(core.get_modpath(core.get_current_modname()) .. "/biosphere.lua")

-- Noise parameters for the terrain.
-- Only 2d is used
local np_terrain = {
	offset = 0,
	scale = 1,
	spread = {x = 100, y = 100, z = 100},
	seed = 5900033,
	octaves = 5,
	persist = 0.63,
	lacunarity = 2.0,
	--flags = ""
}

-- Load biomegen
dofile(core.get_modpath(core.get_current_modname()) .. "/biomegen.lua")

-- Get node used
local c_air = core.CONTENT_AIR
local c_stone = core.get_content_id("mapgen_stone")
local c_water = core.get_content_id("mapgen_river_water_source") -- Use river water source instead of water as pool of water are more like lake than sea
local c_glass = core.get_content_id(biosphere.glass_node)
local c_bridge = core.get_content_id(biosphere.bridge_node)

-- Initialize noise object to nil. It will be created once only during the
-- generation of the first mapchunk, to minimise memory use.
local nobj_terrain = nil

-- Localise noise buffer table outside the loop, to be re-used for all
-- mapchunks, therefore minimising memory use.
local nvals_terrain = {}

-- Localise data buffer table outside the loop, to be re-used for all
-- mapchunks, therefore minimising memory use.
local data = {}
local param2 = {}

-- Door stuff
-- There is no easy way to place a door from mapgen.
-- The main issue is that the doors in Minetest Game are implemented as craft items and special nodes,
-- so we have to simulate what is done by the on_place callback of the craft item.
-- Furthermore, other games and mods provide different definition for doors.
-- Try to detect which door API is used.
local doors_mod = core.get_modpath("doors")
local gameinfo = core.get_game_info()
local door_api = {
	-- Minetest Game. Doors Redo also use the same representation.
	minetest_game = (gameinfo.title == "Minetest Game") and (doors_mod ~= nil)
}

-- Set a left door at a given position
local add_left_door = function(area, data, param2, vindex, orientation, is_bottom)
	-- minetest game
	if door_api.minetest_game then
		if is_bottom then
			data[vindex] = core.get_content_id(biosphere.door_node.."_a")
			param2[vindex] = orientation
		else
			data[vindex + area.ystride] = core.get_content_id("doors:hidden")
			param2[vindex + area.ystride] = orientation
		end
	-- put air in other cases
	else
		data[vindex] = c_air
	end
end
-- Set a right door at a given position
local add_right_door = function(area, data, param2, vindex, orientation, is_bottom)
	-- minetest game
	if door_api.minetest_game then
		if is_bottom then
			data[vindex] = core.get_content_id(biosphere.door_node.."_b")
			param2[vindex] = orientation
		else
			data[vindex + area.ystride] = core.get_content_id("doors:hidden")
			param2[vindex + area.ystride] = (orientation + 3) % 4
		end
	-- put air in other cases
	else
		data[vindex] = c_air
	end
end

-- Register the generation function
-- 'minp' and 'maxp' are the minimum and maximum positions of the mapchunk that define the 3D volume.
core.register_on_generated(function(vmanip, minp, maxp, blockseed)
	-- If outside the y range, skip
	if (maxp.y < biosphere:min_y()) or (minp.y > biosphere:max_y()) then
		return
	end
	-- Side length of mapchunk.
	local sidelen = maxp.x - minp.x + 1
	-- Required dimensions of the 2D noise perlin map.
	local permapdims2d = {x = sidelen, y = sidelen}
	-- Create the perlin map noise object once only, during the generation of
	-- the first mapchunk when 'nobj_terrain' is 'nil'.
	nobj_terrain = nobj_terrain or core.get_perlin_map(np_terrain, permapdims2d)
	-- Create a flat array of noise values from the perlin map, with the
	-- minimum point being 'minp'.
	-- Set the buffer parameter to use and reuse 'nvals_terrain' for this.
	nobj_terrain:get_2d_map_flat({x = minp.x, y = minp.z}, nvals_terrain)

	-- Area
	local emin, emax = vmanip:get_emerged_area()
	-- 'area' is used later to get the voxelmanip indexes for positions.
	local area = VoxelArea:new{MinEdge = emin, MaxEdge = emax}
	-- Get the content ID data from the voxelmanip in the form of a flat array.
	-- Set the buffer parameter to use and reuse 'data' for this.
	vmanip:get_data(data)
	vmanip:get_param2_data(param2)

	-- Generation loop.

	-- Process the content IDs in 'data'.
	-- We use a ZXY order.
	-- The goal is to create multiple biospheres, a sphere of glass half-filled with stone.
	-- An heightmap is applied at the top of the half-sphere to avoid a flat map inside a biosphere.
	-- Depressions are filled with water.
	for z = minp.z, maxp.z do
	for x = minp.x, maxp.x do
		-- Voxelmanip index for the flat array of content IDs.
		-- Initialise to first node in this y column.
		local vi = area:index(x, minp.y, z)
		-- The local coordinates are the coordinates relative to the closest biosphere center (located at 0,0,0 in local coordinates)
		-- The coordinates are defined so that the center of the world (0,0,0) is at the center of a biosphere
		local local_x = biosphere:local_x(x)
		local local_z = biosphere:local_z(z)
		-- Altitude at the position
		local ni = (x - minp.x) + sidelen * (z - minp.z)
		local noiseval = biosphere.noise_amplitude * nvals_terrain[ni + 1]
		-- use a linear interpolation with the distance to the center
		-- so that at the center, the altitude corresponds to the noise
		-- but at the borders, the altitude is 0
		local dist_2d = biosphere:cylindrical_distance_to_center(x, 0, z)
		local altitude = (biosphere.radius - dist_2d) * noiseval / biosphere.radius
		-- Y loop
		for y = minp.y, maxp.y do
			local local_y = biosphere:local_y(y)
			-- Distance to the closest center
			local dist = biosphere:distance_to_center(x, y, z)
			-- If dist < biosphere.radius, use the altitude
			if dist < biosphere.radius then
				-- If below altitude, put stone
				if local_y < altitude then
					data[vi] = c_stone
				-- if above altitude but below water level => put water
				elseif local_y < biosphere.water_level then
					data[vi] = c_water
				-- else put air to overwrite what could have been previously generated
				elseif biosphere.clear then
					data[vi] = c_air
				end
			-- bridge between bioshperes
			-- those are tubes of glass with a ground made of bridge
			elseif (((local_z > -4) and (local_z < 4)) or ((local_x > -4) and (local_x < 4))) and (local_y > -2) and (local_y < 6) then
				-- bridge
				if local_y < 1 then
					data[vi] = c_bridge
				-- Door at biosphere entrances
				elseif (math.abs(local_x) == biosphere.radius) or (math.abs(local_x) == biosphere.radius + 1) or (math.abs(local_z) == biosphere.radius) or (math.abs(local_z) == biosphere.radius + 1) then
					-- wall around and above doors
					if (local_y > 2) or (local_x == 0) or (math.abs(local_x) == 3) or (local_z == 0) or (math.abs(local_z) == 3) then
						data[vi] = c_bridge
					-- air space before doors
					elseif (math.abs(local_x) == biosphere.radius) or (math.abs(local_z) == biosphere.radius) then
						data[vi] = c_air
					-- doors themselves
					elseif (math.abs(local_x) == biosphere.radius + 1) or (math.abs(local_z) == biosphere.radius + 1) then
						local is_bottom = (local_y == 1)
						if (local_z > 0) and (local_x == 1 or local_x == -2) then
							add_left_door(area, data, param2, vi, 0, is_bottom)
						elseif (local_z > 0) and (local_x == 2 or local_x == -1) then
							add_right_door(area, data, param2, vi, 0, is_bottom)
						elseif (local_z < 0) and (local_x == 1 or local_x == -2) then
							add_right_door(area, data, param2, vi, 2, is_bottom)
						elseif (local_z < 0) and (local_x == 2 or local_x == -1) then
							add_left_door(area, data, param2, vi, 2, is_bottom)
						elseif (local_x > 0) and (local_z == 1 or local_z == -2) then
							add_right_door(area, data, param2, vi, 1, is_bottom)
						elseif (local_x > 0) and (local_z == 2 or local_z == -1) then
							add_left_door(area, data, param2, vi, 1, is_bottom)
						elseif (local_x < 0) and (local_z == 1 or local_z == -2) then
							add_left_door(area, data, param2, vi, 3, is_bottom)
						elseif (local_x < 0) and (local_z == 2 or local_z == -1) then
							add_right_door(area, data, param2, vi, 3, is_bottom)
						end
					end
				-- glass part
				elseif (local_y == 5) or (math.abs(local_x) == 3) or (math.abs(local_z) == 3) then
					data[vi] = c_glass
				-- else air
				else
					data[vi] = c_air
				end
			-- Glass dome above and bridge dome below
			elseif dist < biosphere.radius + 2 then
				if local_y > 0 then
					data[vi] = c_glass
				else
					data[vi] = c_bridge
				end
			end
			-- Increment voxelmanip index along y row.
			-- The voxelmanip index increases by ystride when
			-- moving by 1 node in the +y direction.
			vi = vi + area.ystride
		end
	end
	end

	-- Set the param2 table
	vmanip:set_param2_data(param2)

	-- Generate biomes, decorations, ores and dust using biomegen.
	-- It will also call :set_data() for us.
	biomegen.generate_all(data, area, vmanip, minp, maxp, blockseed, biosphere)

	-- Calculate lighting for what has been created.
	vmanip:calc_lighting()
	-- Liquid nodes were placed so set them flowing.
	vmanip:update_liquids()
end)
