local gs_cfg = {}
gs_cfg.__index = gs_cfg

ConfigFile = gs_cfg

local function read_file(file_path)
    local file = io.open(file_path, "r")
    if file then
        local content = file:read("*a")
        file:close()
        return content
    else
        return false
    end
end

local function convert_value(value)
    value = value:match("^%s*(.-)%s*$")

    if value == "nil" then
        return nil
    end

    if value == "true" then
        return true
    elseif value == "false" then
        return false
    end

    if value:match("^-?%d+$") then
        return tonumber(value)
    end

    if value:match("^-?%d+%.%d+$") then
        return tonumber(value)
    end

    if value:match("^%b{}$") then
        local func, err = load("return " .. value)
        if func then
            local success, result = pcall(func)
            if success then
                return result
            end
        end
    end

    return value
end

function gs_cfg.new()
    local new_cfg = setmetatable({}, gs_cfg)
    return new_cfg
end

function gs_cfg.load(file_path)
    if type(file_path) == "string" then
        local file_content = read_file(file_path)
        if file_content ~= false then
            local cfg_data_table = setmetatable({}, gs_cfg)
            local current_section = nil

            for line in file_content:gmatch("[^\r\n]+") do
                line = line:match("^%s*(.-)%s*$")
                local section = line:match("^%[(.-)%]$")
                if section then
                    current_section = section
                    cfg_data_table[current_section] = {}
                elseif current_section and line ~= "" and not line:match("^;") and not line:match("^#") then
                    local key, value = line:match("^(.-)%s*=%s*(.-)$")
                    if key and value then
                        cfg_data_table[current_section][key] = convert_value(value)
                    end
                end
            end
            return cfg_data_table
        else
            error("Could not read file: " .. file_path)
        end
    else
        error("Invalid file path")
    end
end

function gs_cfg:set_value(section, key, value)
    if not self[section] then
        self[section] = {}
    end
    self[section][key] = value
end

function gs_cfg:get_value(section, key)
    if self[section] and self[section][key] then
        return self[section][key]
    else
        return nil
    end
end

function gs_cfg:save(file_path)
    local file = io.open(file_path, "w")
    if not file then
        error("Could not open file for writing: " .. file_path)
    end

    file:write(self:to_string())

    file:close()
end

function gs_cfg:get_sections()
    local sections = {}
    for section_name, _ in pairs(self) do
        table.insert(sections, section_name)
    end
    return sections
end

function gs_cfg:has_section(section)
    return self[section] ~= nil
end

function gs_cfg:has_key(section, key)
    return self[section] and self[section][key] ~= nil
end

function gs_cfg:delete_section(section)
    self[section] = nil
end

function gs_cfg:delete_key(section, key)
    if self[section] then
        self[section][key] = nil
    end
end

function gs_cfg:get_keys(section)
    local keys = {}
    if self[section] then
        for key in pairs(self[section]) do
            table.insert(keys, key)
        end
    end
    return keys
end

function gs_cfg:clear()
    for section in pairs(self) do
        self[section] = nil
    end
end

function gs_cfg:to_string()
    local function serialize_value(value)
        if type(value) == "string" then
            return "\"" .. value .. "\""
        elseif type(value) == "number" or type(value) == "boolean" then
            return tostring(value)
        elseif type(value) == "table" then
            local table_str = "{"
            local is_array = true
            for key, _ in pairs(value) do
                if type(key) ~= "number" then
                    is_array = false
                    break
                end
            end
            if is_array then
                for _, v in ipairs(value) do
                    table_str = table_str .. serialize_value(v) .. ", "
                end
                if #value > 0 then
                    table_str = table_str:sub(1, -3)
                end
            else
                for key, v in pairs(value) do
                    table_str = table_str .. key .. "=" .. serialize_value(v) .. ", "
                end
                if next(value) then
                    table_str = table_str:sub(1, -3)
                end
            end

            table_str = table_str .. "}"
            return table_str
        else
            return tostring(value)
        end
    end

    local result = ""
    for section, keys in pairs(self) do
        if type(keys) == "table" then
            result = result .. "[" .. section .. "]\n"
            for key, value in pairs(keys) do
                result = result .. key .. "=" .. serialize_value(value) .. "\n"
            end
            result = result .. "\n"
        end
    end
    return result
end

function gs_cfg:get_type(section, key)
    local value = self:get_value(section, key)
    return value and type(value) or nil
end
