#include <lua.h>
#include <lauxlib.h>
#include <libpq-fe.h>
#include <stdlib.h>
#include <string.h>

static int pg_connect(lua_State *L) {
	const char *host = luaL_checkstring(L, 1);
	const char *dbname = luaL_checkstring(L, 2);
	const char *user = luaL_checkstring(L, 3);
	const char *password = luaL_checkstring(L, 4);

	char conninfo[512];
	snprintf(conninfo, sizeof(conninfo),
	         "host=%s dbname=%s user=%s password=%s",
	         host, dbname, user, password);

	PGconn *conn = PQconnectdb(conninfo);
	if (PQstatus(conn) != CONNECTION_OK) {
		PQfinish(conn);
		lua_pushnil(L);
		return 1;
	}

	lua_pushlightuserdata(L, conn);
	return 1;
}

static int pg_escape(lua_State *L) {
	PGconn *conn = (PGconn *)lua_touserdata(L, 1);
	const char *input = luaL_checkstring(L, 2);

	size_t len = strlen(input);
	char *escaped = malloc(len * 2 + 1);
	if (!escaped) {
		lua_pushnil(L);
		lua_pushstring(L, "malloc failed");
		return 2;
	}

	int err;
	size_t escaped_len = PQescapeStringConn(conn, escaped, input, len, &err);
	if (err) {
		free(escaped);
		lua_pushnil(L);
		lua_pushstring(L, "escape failed");
		return 2;
	}

	lua_pushlstring(L, escaped, escaped_len);
	free(escaped);
	return 1;
}

static int pg_execute(lua_State *L) {
	PGconn *conn = (PGconn *)lua_touserdata(L, 1);
	const char *query = luaL_checkstring(L, 2);

	PGresult *res = PQexec(conn, query);
	ExecStatusType status = PQresultStatus(res);

	if (status != PGRES_TUPLES_OK && status != PGRES_COMMAND_OK) {
		PQclear(res);
		lua_pushnil(L);
		return 1;
	}

	if (status == PGRES_COMMAND_OK) {
		const char *status_msg = PQcmdStatus(res);
		lua_pushstring(L, status_msg);
		PQclear(res);
		return 1;
	}

	int nrows = PQntuples(res);
	int ncols = PQnfields(res);

	size_t buffer_size = 0;
	for (int i = 0; i < nrows; i++) {
		for (int j = 0; j < ncols; j++) {
			const char *value = PQgetvalue(res, i, j);
			buffer_size += strlen(value) + 1;
		}
		buffer_size += 1;
	}

	char *result = malloc(buffer_size + 1);
	if (!result) {
		PQclear(res);
		lua_pushnil(L);
		lua_pushstring(L, "malloc failed");
		return 2;
	}

	result[0] = '\0';

	for (int i = 0; i < nrows; i++) {
		for (int j = 0; j < ncols; j++) {
			strcat(result, PQgetvalue(res, i, j));
			if (j < ncols - 1) {
				strcat(result, "\t");
			}
		}
		strcat(result, "\n");
	}

	lua_pushstring(L, result);
	free(result);
	PQclear(res);
	return 1;
}

static int lua_luapg_select(lua_State *L) {
	PGconn *conn = (PGconn *)lua_touserdata(L, 1);
	const char *query = luaL_checkstring(L, 2);

	PGresult *res = PQexec(conn, query);
	if (PQresultStatus(res) != PGRES_TUPLES_OK) {
		PQclear(res);
		lua_pushnil(L);
		return 1;
	}

	int nrows = PQntuples(res);
	int ncols = PQnfields(res);

	lua_newtable(L);

	for (int i = 0; i < nrows; i++) {
		lua_newtable(L);
		for (int j = 0; j < ncols; j++) {
			const char *colname = PQfname(res, j);
			const char *value = PQgetvalue(res, i, j);
			lua_pushstring(L, value);
			lua_setfield(L, -2, colname);
		}
		lua_rawseti(L, -2, i + 1);
	}

	PQclear(res);
	return 1;
}

int luaopen_luapg(lua_State *L) {
	luaL_Reg funcs[] = {
		{"connect", pg_connect},
		{"escape", pg_escape},
		{"execute", pg_execute},
		{"select", lua_luapg_select},
		{NULL, NULL}
	};

	luaL_newlib(L, funcs);
	return 1;
}

//gcc -O2 -Wall -shared -fPIC -o luapg.so luapg.c -I/usr/include/lua5.4 -I/usr/include/postgresql -llua5.4 -lpq
