local connection_data = {"host", "dbname", "user", "password"}
local user_home_dir = os.getenv("HOME")
if not user_home_dir then
	minetest.log("error", "[postgresql] HOME environment variable not found.")
	return
end
local config_path = user_home_dir .. "/.lt_postgresql.conf"
local postgresql_conf = Settings(config_path)
if not postgresql_conf or not next(postgresql_conf:to_table()) then
	minetest.log("error", "[postgresql] Config file not found or empty at " .. config_path)
	return
end
for _, conf_key in ipairs(connection_data)  do
	local conf_value = postgresql_conf:get(conf_key)
	if not conf_value then
		minetest.log("error", "[postgresql] Key \"" .. conf_key .. "\" in \"" .. config_path .. "\" not found.")
		return
	end
	connection_data[conf_key] = conf_value
end

--every value in ~/.lt_postgresql.conf found :)

postgresql = {}
local luapg = assert(package.loadlib(minetest.get_modpath("postgresql") .. "/luapg.so", "luaopen_luapg"))()

function postgresql.connect(host, dbname, user, password)
	local connection = luapg.connect(host, dbname, user, password)
	if not connection then
		return false
	end
	postgresql.conn = connection
	return true
end

function postgresql.execute_query(query)
	if not postgresql.conn then
		return false
	end
	local ok = luapg.execute(postgresql.conn, query)
	return ok
end

function postgresql.escape(text, escaping_type)
	if not postgresql.conn then
		return false
	end
	if escaping_type ~= "identifier" and escaping_type ~= "value" then
		escaping_type = "value"
	end
	return luapg.escape(postgresql.conn, text, escaping_type)
end

function postgresql.execute_query_with_args(sql_query, parameters)
	if not postgresql.conn then
		return false
	end

	if type(parameters) ~= "table" then
		return false
	end

	local escaped_parameters = {}
	for index, value in ipairs(parameters) do
		local escaped_value = postgresql.escape(tostring(value), "value")
		if not escaped_value then
			return false
		end
		table.insert(escaped_parameters, "'" .. escaped_value .. "'")
	end

	local parameter_index = 0
	local query_with_values = sql_query:gsub("%?", function()
		parameter_index = parameter_index + 1
		return escaped_parameters[parameter_index] or "NULL"
	end)

	local success = postgresql.execute_query(query_with_values)
	return success
end

function postgresql.select_query_with_args(sql_query, parameters)
	if not postgresql.conn then
		return false
	end
	if type(parameters) ~= "table" then
		return false
	end

	local escaped_parameters = {}
	for _, value in ipairs(parameters) do
		local escaped_value = postgresql.escape(tostring(value), "value")
		if not escaped_value then
			return false
		end
		table.insert(escaped_parameters, "'" .. escaped_value .. "'")
	end

	local parameter_index = 0
	local query_with_values = sql_query:gsub("%?", function()
		parameter_index = parameter_index + 1
		return escaped_parameters[parameter_index] or "NULL"
	end)

	return luapg.select(postgresql.conn, query_with_values)
end

local connection_ok = postgresql.connect(connection_data.host, connection_data.dbname, connection_data.user, connection_data.password)

if not connection_ok then
	postgresql = nil
	minetest.log("error", "Did not connect to postgresql! Check the file " .. config_path)
	return
end

function postgresql.create_table(table_name, column_definition)
	local escaped_table = postgresql.escape(table_name, "identifier")
	local query = "CREATE TABLE IF NOT EXISTS " .. escaped_table .. " (" .. column_definition .. ");"
	return postgresql.execute_query(query)
end

function postgresql.drop_table(table_name)
	local escaped_table = postgresql.escape(table_name, "identifier")
	local query = "DROP TABLE IF EXISTS " .. escaped_table .. ";"
	return postgresql.execute_query(query)
end
