postgresql = {}
local luapg = assert(package.loadlib(minetest.get_modpath(minetest.get_current_modname()) .. "/luapg.so", "luaopen_luapg"))()

PostgresConnection = {}
PostgresConnection.__index = PostgresConnection

---@param host string The hostname of the PostgreSQL server
---@param dbname string The name of the database
---@param user string The database username
---@param password string The database password
---@return PostgresConnection|nil, string|nil Returns a PostgresConnection object or nil + error message
function PostgresConnection.new(host, dbname, user, password)
    local conn = luapg.connect(host, dbname, user, password)
    if not conn then
        return nil, "Connection failed"
    end
    local self = setmetatable({}, PostgresConnection)
    self._conn = conn
    return self
end

---@param config_path string Path to the configuration file
---@return PostgresConnection|nil, string|nil Returns a PostgresConnection object or nil + error message
function PostgresConnection.new_from_conf(config_path)
    local postgresql_conf = Settings(config_path)
    if not postgresql_conf or not next(postgresql_conf:to_table()) then
        return nil, "Config file not found or empty at " .. config_path
    end
    local host = postgresql_conf:get("host")
    local dbname = postgresql_conf:get("dbname")
    local user = postgresql_conf:get("user")
    local password = postgresql_conf:get("password")
    return PostgresConnection.new(host, dbname, user, password)
end

---@param query string SQL statement to execute
---@return boolean, string|nil Returns true if successful, false + error message on failure
function PostgresConnection:execute(query)
    local ok, err = luapg.execute(self._conn, query)
    if not ok then
        return false, err
    end
    return true
end

---@param query string SQL SELECT statement
---@return table|nil, string|nil Returns a table of rows or nil + error message on failure
function PostgresConnection:select(query)
    local rows, err = luapg.select(self._conn, query)
    if not rows then
        return nil, err
    end
    return rows
end

---@param text string Text to escape
---@param escaping_type string "value" or "identifier"
---@return string|nil, string|nil Returns the escaped text or nil + error message
function PostgresConnection:escape(text, escaping_type)
	if escaping_type ~= "identifier" and escaping_type ~= "value" then
		escaping_type = "value"
	end
    local result, err = luapg.escape(self._conn, text, escaping_type)
    if not result then
        return nil, err
    end
    return result
end
