#include <lua.h>
#include <lauxlib.h>
#include <libpq-fe.h>
#include <stdlib.h>
#include <string.h>

static int l_connect(lua_State *L) {
    const char *host = luaL_checkstring(L, 1);
    const char *dbname = luaL_checkstring(L, 2);
    const char *user = luaL_checkstring(L, 3);
    const char *password = luaL_checkstring(L, 4);

    char conninfo[512];
    snprintf(conninfo, sizeof(conninfo), "host=%s dbname=%s user=%s password=%s", host, dbname, user, password);

    PGconn *conn = PQconnectdb(conninfo);
    if (PQstatus(conn) != CONNECTION_OK) {
        PQfinish(conn);
        lua_pushnil(L);
        return 1;
    }

    lua_pushlightuserdata(L, conn);
    return 1;
}

static int l_escape(lua_State *L) {
    PGconn *conn = (PGconn *)lua_touserdata(L, 1);
    const char *input = luaL_checkstring(L, 2);
    const char *mode = luaL_checkstring(L, 3);

    if (strcmp(mode, "value") == 0) {
        size_t len = strlen(input);
        char *escaped = (char *)malloc(len * 2 + 1);
        if (!escaped) {
            lua_pushnil(L);
            lua_pushstring(L, "malloc failed");
            return 2;
        }
        int err = 0;
        size_t out_len = PQescapeStringConn(conn, escaped, input, len, &err);
        if (err) {
            free(escaped);
            lua_pushnil(L);
            lua_pushstring(L, "escape failed");
            return 2;
        }
        lua_pushlstring(L, escaped, out_len);
        free(escaped);
        return 1;
    } else if (strcmp(mode, "identifier") == 0) {
        char *escaped_id = PQescapeIdentifier(conn, input, strlen(input));
        if (!escaped_id) {
            lua_pushnil(L);
            lua_pushstring(L, "escape failed");
            return 2;
        }
        lua_pushstring(L, escaped_id);
        PQfreemem(escaped_id);
        return 1;
    } else {
        lua_pushnil(L);
        lua_pushstring(L, "invalid escape mode");
        return 2;
    }
}

static int l_execute(lua_State *L) {
    PGconn *conn = (PGconn *)lua_touserdata(L, 1);
    const char *query = luaL_checkstring(L, 2);

    PGresult *res = PQexec(conn, query);
    if (!res) {
        lua_pushnil(L);
        lua_pushstring(L, "PQexec returned NULL");
        return 2;
    }

    ExecStatusType status = PQresultStatus(res);
    if (status == PGRES_TUPLES_OK || status == PGRES_COMMAND_OK) {
        PQclear(res);
        lua_pushboolean(L, 1);
        return 1;
    } else {
        const char *err = PQresultErrorMessage(res);
        lua_pushnil(L);
        lua_pushstring(L, err ? err : "unknown error");
        PQclear(res);
        return 2;
    }
}

static int l_select(lua_State *L) {
    PGconn *conn = (PGconn *)lua_touserdata(L, 1);
    const char *query = luaL_checkstring(L, 2);

    PGresult *res = PQexec(conn, query);
    if (!res) {
        lua_pushnil(L);
        lua_pushstring(L, "PQexec returned NULL");
        return 2;
    }

    if (PQresultStatus(res) != PGRES_TUPLES_OK) {
        const char *err = PQresultErrorMessage(res);
        lua_pushnil(L);
        lua_pushstring(L, err ? err : "query did not return tuples");
        PQclear(res);
        return 2;
    }

    int nrows = PQntuples(res);
    int ncols = PQnfields(res);

    lua_newtable(L);
    for (int i = 0; i < nrows; i++) {
        lua_newtable(L);
        for (int j = 0; j < ncols; j++) {
            const char *colname = PQfname(res, j);
            if (PQgetisnull(res, i, j)) {
                lua_pushstring(L, "");
            } else {
                lua_pushstring(L, PQgetvalue(res, i, j));
            }
            lua_setfield(L, -2, colname);
        }
        lua_rawseti(L, -2, i + 1);
    }

    PQclear(res);
    return 1;
}

int luaopen_luapg(lua_State *L) {
    luaL_Reg funcs[] = {
        {"connect", l_connect},
        {"escape", l_escape},
        {"execute", l_execute},
        {"select", l_select},
        {NULL, NULL}
    };
#if LUA_VERSION_NUM >= 502
    luaL_newlib(L, funcs);
#else
    luaL_register(L, "luapg", funcs);
#endif
    return 1;
}

//gcc -O2 -Wall -shared -fPIC -o luapg.so luapg.c -I/usr/include/lua5.4 -I/usr/include/postgresql -llua5.4 -lpq
