wind_charges = {}

local registered_nodes = minetest.registered_nodes

local function is_solid_node_near(pos)
	local radius = 0.12
	local offsets = {
		{x = radius, y = 0, z = 0}, {x = -radius, y = 0, z = 0},
		{x = 0, y = radius, z = 0}, {x = 0, y = -radius, z = 0},
		{x = 0, y = 0, z = radius}, {x = 0, y = 0, z = -radius}
	}
	for i = 1, #offsets do
		local p = vector.add(pos, offsets[i])
		local node = minetest.get_node_or_nil(p)
		local def = node and registered_nodes[node.name]
		if def and def.walkable then
			return true
		end
	end
	return false
end

minetest.register_entity("wind_charges:wind_projectile", {
	initial_properties = {
		physical = true,
		collisionbox = {-0.1, -0.1, -0.1, 0.1, 0.1, 0.1},
		visual = "sprite",
		textures = {"wind_charge.png"},
		visual_size = {x = 0.5, y = 0.5},
		collide_with_objects = false,
	},

	lifetime = 8,
	timer = 0,
	spawn_time = 0,

	explode = function(self)
		local pos = self.object:get_pos()
		minetest.sound_play("wind_charge_explode", {
			pos = pos,
			max_hear_distance = 24,
			gain = 1.0,
		})

		minetest.add_particlespawner({
			amount = 8,
			time = 0.1,
			minpos = pos,
			maxpos = pos,
			minvel = {x = -5, y = 0, z = -5},
			maxvel = {x = 5, y = 2, z = 5},
			minacc = {x = 0, y = 0, z = 0},
			maxacc = {x = 0, y = -5, z = 0},
			minexptime = 0.1,
			maxexptime = 0.2,
			minsize = 5,
			maxsize = 8,
			texture = "wind_charge_explode_particle.png",
			glow = 5,
		})

		local objs = minetest.get_objects_inside_radius(pos, 4)
		for _, obj in ipairs(objs) do
			if obj:is_player() then
				local obj_pos = obj:get_pos()
				local dist = vector.distance(pos, obj_pos)
				local dir = vector.direction(pos, obj_pos)
				local power = 16 * (1 - math.min(dist / 4, 1))
				dir.y = 1
				obj:add_player_velocity(vector.multiply(dir, power))
			end
		end

		self.object:remove()
	end,

	on_punch = function(self, puncher, time_from_last_punch, tool_capabilities, dir)
		self:explode()
		return 0
	end,

	on_activate = function(self, staticdata)
		local data = minetest.deserialize(staticdata or "") or {}
		self.spawn_time = data.spawn_time or os.time()
	end,

	get_staticdata = function(self)
		return minetest.serialize({
			spawn_time = self.spawn_time,
		})
	end,

	on_step = function(self, dtime)
		self.timer = self.timer + dtime
		if os.time() - self.spawn_time >= self.lifetime then
			self:explode()
			return
		end

		local pos = self.object:get_pos()
		if is_solid_node_near(pos) then
			self:explode()
			return
		end
	end,
})

local players_wind_charge_cooldown = {}

minetest.register_craftitem("wind_charges:wind_charge", {
	description = "Wind Charge",
	inventory_image = "wind_charge.png",
	on_use = function(itemstack, user, pointed_thing)
		local user_name = user:get_player_name()

		local current_time = minetest.get_gametime()
		local last_used_time = players_wind_charge_cooldown[user_name] or 0
		local cooldown_time = 0.02

		if current_time - last_used_time < cooldown_time then
			return itemstack
		end

		players_wind_charge_cooldown[user_name] = current_time

		local pos = user:get_pos()
		local props = user:get_properties()
		local eye_height = props and props.eye_height or 1.6
		pos.y = pos.y + eye_height

		local dir = user:get_look_dir()
		local player_velocity = user:get_velocity() or {x = 0, y = 0, z = 0}

		local obj = minetest.add_entity(pos, "wind_charges:wind_projectile", minetest.serialize({spawn_time = os.time(),}))
		if obj then
			local base_speed = vector.multiply(dir, 18)
			local total_velocity = vector.add(base_speed, player_velocity)
			obj:set_velocity(total_velocity)

			minetest.sound_play("wind_charge_throw", {
				pos = pos,
				max_hear_distance = 16,
				gain = 0.8
			})
		end
		itemstack:take_item()
		return itemstack
	end
})
