local mod_name = minetest.get_current_modname()
local mod_path = minetest.get_modpath(mod_name)

local SPAWN_HEIGHT = 50  
local FIRE_TRAIL_DURATION = 6  
local FIRE_PARTICLES = true 
local DAMAGE_ON_LANDING = false 

local fire_trails = {}
local spawn_timers = {}  

local function verificar_50_bloques_arriba(pos)
    if not pos then
        return false, "Posición inválida"
    end
    
    local pos_actual = {
        x = math.floor(pos.x),
        y = math.floor(pos.y),
        z = math.floor(pos.z)
    }

    for i = 1, SPAWN_HEIGHT do
        local pos_verificar = {
            x = pos_actual.x,
            y = pos_actual.y + i,
            z = pos_actual.z
        }
        
        local nodo = minetest.get_node(pos_verificar)

        if nodo.name ~= "air" and nodo.name ~= "ignore" then
            return false, i 
        end
    end
    
    return true, SPAWN_HEIGHT  
end

local function create_fire_particles(pos)
    if not FIRE_PARTICLES then return end
    
    minetest.add_particlespawner({
        amount = 15,
        time = 0.5,
        minpos = {x = pos.x - 0.3, y = pos.y - 0.5, z = pos.z - 0.3},
        maxpos = {x = pos.x + 0.3, y = pos.y - 0.2, z = pos.z + 0.3},
        minvel = {x = -0.5, y = 0.5, z = -0.5},
        maxvel = {x = 0.5, y = 1.5, z = 0.5},
        minacc = {x = 0, y = 0.5, z = 0},
        maxacc = {x = 0, y = 0.5, z = 0},
        minexptime = 1,
        maxexptime = 2,
        minsize = 5,
        maxsize = 10,
        collisiondetection = false,
        texture = "fire_basic_flame.png",
        glow = 14
    })
end

local function create_fire_trail(player)
    local player_name = player:get_player_name()
    
    if fire_trails[player_name] then
        minetest.after(0, function()
            local p = minetest.get_player_by_name(player_name)
            if p then
                p:set_detach()
            end
        end)
        local p = minetest.get_player_by_name(player_name)
        if p then
            p:set_eye_offset({x=0, y=0, z=0}, {x=0, y=0, z=0})
        end
        fire_trails[player_name] = nil
    end
    
    local pos = player:get_pos()
    local obj = minetest.add_entity(pos, "entrance:fire_trail")
    
    if obj then
        obj:set_attach(player, "", {x=0, y=0, z=0}, {x=0, y=0, z=0})
        fire_trails[player_name] = {
            object = obj,
            timer = 0
        }
    end
end

local function cleanup_player_effects(player_name)
    local player = minetest.get_player_by_name(player_name)
    if player then
        player:set_physics_override({gravity = 1})
    end
    
    if fire_trails[player_name] then
        if fire_trails[player_name].object then
            fire_trails[player_name].object:remove()
        end
        fire_trails[player_name] = nil
    end
    
    if spawn_timers[player_name] then
        spawn_timers[player_name] = nil
    end
end

local function check_landing(player_name, start_time)
    local player = minetest.get_player_by_name(player_name)
    if not player then
        cleanup_player_effects(player_name)
        return
    end
    
    local current_time = minetest.get_us_time() / 1000000
    local elapsed_time = current_time - start_time
    
    local player_pos = player:get_pos()
    local node_below = minetest.get_node({x=player_pos.x, y=player_pos.y-0.5, z=player_pos.z})
    
    local on_ground = node_below.name ~= "air" and node_below.name ~= "ignore"
    
    if on_ground or elapsed_time >= FIRE_TRAIL_DURATION then
        player:set_physics_override({gravity = 1})
        
        if fire_trails[player_name] then
            if fire_trails[player_name].object then
                fire_trails[player_name].object:remove()
            end
            fire_trails[player_name] = nil
        end
    
        if on_ground then
            minetest.add_particlespawner({
                amount = 30,
                time = 0.5,
                minpos = {x=player_pos.x-1, y=player_pos.y, z=player_pos.z-1},
                maxpos = {x=player_pos.x+1, y=player_pos.y+0.5, z=player_pos.z+1},
                minvel = {x=-2, y=2, z=-2},
                maxvel = {x=2, y=4, z=2},
                minacc = {x=0, y= -5, z=0},
                maxacc = {x=0, y= -5, z=0},
                minexptime = 1,
                maxexptime = 2,
                minsize = 3,
                maxsize = 8,
                collisiondetection = true,
                texture = "fire_basic_flame.png",
                glow = 14
            })
            
            if DAMAGE_ON_LANDING then
                local hp = player:get_hp()
                if hp > 5 then
                    player:set_hp(hp - 5)
                end
            end
        end
        
        spawn_timers[player_name] = nil
        return
    end

    if elapsed_time < FIRE_TRAIL_DURATION then
        create_fire_particles(player_pos)

        minetest.after(0.1, function()
            if spawn_timers[player_name] then
                check_landing(player_name, start_time)
            end
        end)
    end
end

local function entrance(player)
    local player_name = player:get_player_name()
    if not minetest.check_player_privs(player_name, {entrance = true}) then
        return
    end

    -- Verificar espacio antes de teletransportar
    local pos_actual = player:get_pos()
    local hay_espacio, altura = verificar_50_bloques_arriba(pos_actual)
    
    if not hay_espacio then
        return
    end

    cleanup_player_effects(player_name)
    local pos = player:get_pos()
    pos.y = pos.y + SPAWN_HEIGHT
    player:set_pos(pos)
    player:set_physics_override({
        gravity = 0.8 
    })

    create_fire_trail(player)

    minetest.add_particle({
        pos = pos,
        velocity = {x=0, y=0, z=0},
        acceleration = {x=0, y=0, z=0},
        expirationtime = 1.5,
        size = 15,
        texture = "fire_basic_flame.png",
        glow = 14
    })

    minetest.sound_play("fire_fire", {
        pos = pos,
        gain = 1.0,
        max_hear_distance = 20
    })

    spawn_timers[player_name] = true
    local start_time = minetest.get_us_time() / 1000000
    
    minetest.after(0.1, function()
        if spawn_timers[player_name] then
            check_landing(player_name, start_time)
        end
    end)
    
    minetest.after(1, function()
        local player_pos = player:get_pos()
        minetest.sound_play("explosion", {
            pos = player_pos,
            gain = 1,
            max_hear_distance = 15
        })
    end)
end

minetest.register_privilege("entrance", {
    description = "Permite aparecer cayendo épicamente con estela de fuego",
    give_to_singleplayer = true
})

minetest.register_on_joinplayer(function(player)
    minetest.after(0.5, function()
        entrance(player)
    end)
end)

minetest.register_on_leaveplayer(function(player)
    local player_name = player:get_player_name()
    cleanup_player_effects(player_name)
end)

minetest.register_on_dieplayer(function(player)
    local player_name = player:get_player_name()
    cleanup_player_effects(player_name)
    return true
end)

minetest.register_entity("entrance:fire_trail", {
    initial_properties = {
        visual = "cube",
        visual_size = {x = 0.5, y = 2, z = 0.5},
        textures = {
            "fire_basic_flame.png", "fire_basic_flame.png",
            "fire_basic_flame.png", "fire_basic_flame.png",
            "fire_basic_flame.png", "fire_basic_flame.png"
        },
        collisionbox = {0, 0, 0, 0, 0, 0},
        physical = false,
        pointable = false,
        glow = 14,
        static_save = false
    },
    
    on_activate = function(self, staticdata)
        self.object:set_armor_groups({immortal = 1})
        -- Autodestrucción por seguridad
        minetest.after(10, function()
            if self.object then
                self.object:remove()
            end
        end)
    end,
    
    on_step = function(self, dtime)
        local pos = self.object:get_pos()
        if pos then
            minetest.add_particlespawner({
                amount = 5,
                time = 0.1,
                minpos = {x = pos.x - 0.2, y = pos.y - 1, z = pos.z - 0.2},
                maxpos = {x = pos.x + 0.2, y = pos.y - 0.5, z = pos.z + 0.2},
                minvel = {x = -0.5, y = 1, z = -0.5},
                maxvel = {x = 0.5, y = 2, z = 0.5},
                minacc = {x = 0, y = 0.5, z = 0},
                maxacc = {x = 0, y = 0.5, z = 0},
                minexptime = 0.5,
                maxexptime = 1,
                minsize = 1,
                maxsize = 3,
                collisiondetection = false,
                texture = "fire_basic_flame.png",
                glow = 14
            })
        end
    end
})