
local modname = minetest.get_current_modname()
local modpath = minetest.get_modpath(modname)

local texgen = assert(loadfile(modpath.."/texgen.lua"))()

local name_format = minetest.settings:get("texttext_name_format") or "name_item"
local name_format_choices = { "name_full", "name_item", "desc_full", "desc_short", }

local cache_enabled = (minetest.settings:get("texttext_cache_enabled") or "false") == "true"

local insecure = nil
local existing_textures = {}
if cache_enabled then
	insecure = minetest.request_insecure_environment()
	if not insecure then
		error("Text Text requires an insecure env for texture caching."
			.." Either disable cache or add texttext to trusted mods.", 0)
	end
	
	for i, format in ipairs(name_format_choices) do
		local path = modpath.."/textures/"..format
		minetest.mkdir(path)
		for j, filename in ipairs(minetest.get_dir_list(path, false)) do
			existing_textures[filename] = true
		end
	end
end

minetest.register_on_mods_loaded(function()
	if cache_enabled then
		for name, def in pairs(minetest.registered_nodes) do
			local filename = "texttext__"..name_format.."__"..name:gsub(":", "8")..".png"
			minetest.override_item(name, { tiles={filename} })
			
			if not existing_textures[filename] then
				existing_textures[filename] = true
				local png = texgen(name, def)
				local filepath = modpath.."/textures/"..name_format.."/"..filename
				-- would be nice if mintest.safe_file_write worked in an insecure env
				local file = insecure.io.open(filepath, "wb")
				file:write(png)
				file:close()
			end
		end
	else
		for name, def in pairs(minetest.registered_nodes) do
			minetest.override_item(name, {
				tiles = { "^[png:"..minetest.encode_base64(texgen(name, def)) },
			})
		end
	end
end)

minetest.register_chatcommand("texttext_clear", {
	params = "",
	description = "Clear the Text Text image cache."
		.. "\nWill cause textures to be re-generated next load.",
	privs = { server=true, },
	func = function(name, param)
		if not cache_enabled then
			return false, "Cache is disabled. Nothing changed."
		end
		-- would be nice if minetest.rmdir worked in an insecure env
		for i, format in ipairs(name_format_choices) do
			local path = modpath.."/textures/"..format.."/"
			for j, filename in pairs(minetest.get_dir_list(path, false)) do
				insecure.os.remove(path..filename)
				existing_textures[filename] = nil
			end
		end
	end,
})
