local nc = nodecore
local source = "nc_terrain:water_source"
local flow = "nc_terrain:water_flowing"

-- Among the default map generators, only 'valleys' generates river water.
-- The "rivers" in 'v7' and 'carpathian' are really canals with normal water in them.
-- So, ignore "nc_terrain:river_water".

-- Disable vanilla water renewal.
core.registered_nodes[source].liquid_renewable = false
core.registered_nodes[flow].liquid_renewable = false

local min_connected = tonumber(core.settings:get("flood_min_connected")) or 100
-- If min_connected is set to 1 or lower, the whole world will flood. Don't do it. Seriously.
if min_connected < 2 then min_connected = 2 end
local cache_timeout = tonumber(core.settings:get("flood_cache_timeout")) or 10

-- Directions in which flooding occurs. Not up.
local flood_dirs = {
	{x = 0, y =-1, z = 0}, -- down
	{x = 0, y = 0, z = 1}, -- north
	{x = 0, y = 0, z =-1}, -- south
	{x = 1, y = 0, z = 0}, -- east
	{x =-1, y = 0, z = 0}  -- west
}

local flood_cache = {}

-- Once an area is completely flooded, cached positions don't get checked anymore,
-- so expired cached values may accumulate during long play sessions.
-- So, just delete the whole cache every 10 minutes.
local delete_cache_interval = 600 -- seconds
local delete_cache_time = nc.gametime or 0 + delete_cache_interval

-- This ABM targets water sources which are adjacent to flowing water.
-- Since this ABM replaces most such flowing water with more water sources,
-- the ABM runs out of valid targets fairly quickly and ceases using (much) CPU time.
core.register_abm({
	label = "flood water",
	nodenames = {source},
	neighbors = {flow},
	interval = 1,
	chance = 1,
	action = function(pos1, node1)
		-- Figure out which, if any, directions contain flowing water.
		local flows = {}
		for _, dir in ipairs(flood_dirs) do
			local pos2 = vector.add(pos1, dir)
			if core.get_node(pos2).name == flow then
				flows[#flows + 1] = pos2
			end
		end
		if #flows <= 0 then return end

		local n = 0 -- Number of water sources connected to this one.
		local t = nc.gametime + cache_timeout -- Cache expiry time, in seconds since the world was created.

		-- Count water sources connected to this one until the threshold is reached or there are no more.
		nc.scan_flood(pos1, min_connected, function(pos2)
			-- Use the cache, or clear it if it's expired.
			local hash2 = core.hash_node_position(pos2)
			local cached = flood_cache[hash2]
			if cached then
				if cached.t > nc.gametime then
					n = cached.n
					t = cached.t
					if n >= min_connected then
						-- Flood!
						for _, pos in ipairs(flows) do core.set_node(pos, node1) end
					end
					return true -- End the scan.
				end
				flood_cache[hash2] = nil
			end
			-- Only water at the same level or above contributes to flooding.
			-- Otherwise a source just above an ocean or canal could raise the global sea level by 1 node.
			if pos2.y < pos1.y then return false end
			-- Stop scanning in this direction if there's no water here.
			if core.get_node(pos2).name ~= source then return false end
			n = n + 1
			-- Continue scanning if the count isn't high enough yet.
			if n < min_connected then return end
			-- Flood!
			for _, pos in ipairs(flows) do core.set_node(pos, node1) end
			return true -- End the scan.
		end)
		if nc.gametime > delete_cache_time then
			flood_cache = {}
			delete_cache_time = nc.gametime + delete_cache_interval
		end
		local hash1 = core.hash_node_position(pos1)
		if not flood_cache[hash1] then flood_cache[hash1] = {n = n, t = t} end
	end
})
