-- SPDX-FileCopyrightText: 2025 IanTwenty <https://gitlab.com/IanTwenty>
--
-- SPDX-License-Identifier: AGPL-3.0-or-later
--
-- A game mod for Mineclonia - adds more chocolate items to the game
--
-- For full copyright information see the AUTHORS file at the top-level
-- directory of this distribution or at
-- [AUTHORS](https://gitlab.com/IanTwenty/allthingschocolate/AUTHORS.md)
--
-- This program is free software: you can redistribute it and/or modify it under
-- the terms of the GNU Affero General Public License as published by the Free
-- Software Foundation, either version 3 of the License, or (at your option) any
-- later version.
--
-- This program is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
-- FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
-- details.
--
-- You should have received a copy of the GNU Affero General Public License
-- along with this program. If not, see <https://www.gnu.org/licenses/>.

local CAKE_HUNGER_POINTS = 4

local S = minetest.get_translator(minetest.get_current_modname())

local cake_texture = {
	"allthingschocolate_cake_top.png",
	"allthingschocolate_cake_bottom.png",
	"allthingschocolate_cake_inner.png",
	"allthingschocolate_cake_side.png",
	"allthingschocolate_cake_side.png",
	"allthingschocolate_cake_side.png",
}
local slice_1 = { -7 / 16, -8 / 16, -7 / 16, -5 / 16, 0 / 16, 7 / 16 }
local slice_2 = { -7 / 16, -8 / 16, -7 / 16, -3 / 16, 0 / 16, 7 / 16 }
local slice_3 = { -7 / 16, -8 / 16, -7 / 16, -1 / 16, 0 / 16, 7 / 16 }
local slice_4 = { -7 / 16, -8 / 16, -7 / 16, 1 / 16, 0 / 16, 7 / 16 }
local slice_5 = { -7 / 16, -8 / 16, -7 / 16, 3 / 16, 0 / 16, 7 / 16 }
local slice_6 = { -7 / 16, -8 / 16, -7 / 16, 5 / 16, 0 / 16, 7 / 16 }

local full_cake = { -7 / 16, -8 / 16, -7 / 16, 7 / 16, 0 / 16, 7 / 16 }

minetest.register_craft({
	output = "mcl_allthingschocolate:cake",
	recipe = {
		{ "mcl_mobitems:milk_bucket", "mcl_mobitems:milk_bucket", "mcl_mobitems:milk_bucket" },
		{ "mcl_core:sugar", "mcl_throwing:egg", "mcl_core:sugar" },
		{ "mcl_cocoas:cocoa_beans", "mcl_farming:wheat_item", "mcl_cocoas:cocoa_beans" },
	},
	replacements = {
		{ "mcl_mobitems:milk_bucket", "mcl_buckets:bucket_empty" },
		{ "mcl_mobitems:milk_bucket", "mcl_buckets:bucket_empty" },
		{ "mcl_mobitems:milk_bucket", "mcl_buckets:bucket_empty" },
	},
})

minetest.register_node("mcl_allthingschocolate:cake", {
	description = S("Cake"),
	_tt_help = S("With 7 tasty slices!") .. "\n" .. S("Hunger points: +@1 per slice", CAKE_HUNGER_POINTS),
	_doc_items_longdesc = S(
		"Cakes can be placed and eaten to restore hunger points. A cake has 7 slices. Each slice restores 4 hunger points and 0.6 saturation points. Cakes will be destroyed when dug or when the block below them is broken."
	),
	_doc_items_usagehelp = S(
		"Place the cake anywhere, then rightclick it to eat a single slice. You can't eat from the cake when your hunger bar is full."
	),
	tiles = {
		"allthingschocolate_cake_top.png",
		"allthingschocolate_cake_bottom.png",
		"allthingschocolate_cake_side.png",
		"allthingschocolate_cake_side.png",
		"allthingschocolate_cake_side.png",
		"allthingschocolate_cake_side.png",
	},
	use_texture_alpha = minetest.features.use_texture_alpha_string_modes and "opaque" or false,
	inventory_image = "allthingschocolate_cake.png",
	wield_image = "allthingschocolate_cake.png",
	paramtype = "light",
	is_ground_content = false,
	drawtype = "nodebox",
	selection_box = {
		type = "fixed",
		fixed = full_cake,
	},
	node_box = {
		type = "fixed",
		fixed = full_cake,
	},
	stack_max = 1,
	groups = {
		handy = 1,
		attached_node = 1,
		dig_by_piston = 1,
		comparator_signal = 14,
		cake = 7,
		food = 2,
		no_eat_delay = 1,
		compostability = 100,
	},
	drop = "",
	on_rightclick = function(pos, node, clicker, itemstack)
		-- Cake is subject to protection
		local name = clicker:get_player_name()
		if minetest.is_protected(pos, name) then
			minetest.record_protection_violation(pos, name)
			return
		end
		-- TODO: Is this 2 correct? Should it be CAKE_HUNGER_POINTS?
		local newcake = minetest.do_item_eat(
			2,
			ItemStack("mcl_allthingschocolate:cake_6"),
			ItemStack("mcl_allthingschocolate:cake"),
			clicker,
			{ type = "nothing" }
		)
		-- Check if we were allowed to eat
		if
			newcake:get_name() ~= "mcl_allthingschocolate:cake"
			or minetest.is_creative_enabled(clicker:get_player_name())
		then
			minetest.add_node(pos, { type = "node", name = "mcl_allthingschocolate:cake_6", param2 = 0 })
		end
	end,
	sounds = mcl_sounds.node_sound_leaves_defaults(),

	_food_particles = false,
	_mcl_saturation = 0.6,
	_mcl_blast_resistance = 0.5,
	_mcl_hardness = 0.5,
})

local register_slice = function(level, nodebox, desc)
	local this = "mcl_allthingschocolate:cake_" .. level
	local after_eat = "mcl_allthingschocolate:cake_" .. (level - 1)
	local on_rightclick
	if level > 1 then
		on_rightclick = function(pos, node, clicker, itemstack)
			local name = clicker:get_player_name()
			if minetest.is_protected(pos, name) then
				minetest.record_protection_violation(pos, name)
				return
			end
			local newcake = minetest.do_item_eat(
				CAKE_HUNGER_POINTS,
				ItemStack(after_eat),
				ItemStack(this),
				clicker,
				{ type = "nothing" }
			)
			-- Check if we were allowed to eat
			if newcake:get_name() ~= this or minetest.is_creative_enabled(clicker:get_player_name()) then
				minetest.add_node(pos, { type = "node", name = after_eat, param2 = 0 })
			end
		end
	else
		-- Last slice
		on_rightclick = function(pos, node, clicker, itemstack)
			local name = clicker:get_player_name()
			if minetest.is_protected(pos, name) then
				minetest.record_protection_violation(pos, name)
				return
			end
			local newcake = minetest.do_item_eat(
				CAKE_HUNGER_POINTS,
				ItemStack("mcl_allthingschocolate:cake 0"),
				ItemStack("mcl_allthingschocolate:cake_1"),
				clicker,
				{ type = "nothing" }
			)
			-- Check if we were allowed to eat
			if newcake:get_name() ~= this or minetest.is_creative_enabled(clicker:get_player_name()) then
				minetest.remove_node(pos)
				minetest.check_for_falling(pos)
			end
		end
	end

	minetest.register_node(this, {
		description = desc,
		_doc_items_create_entry = false,
		tiles = cake_texture,
		use_texture_alpha = minetest.features.use_texture_alpha_string_modes and "opaque" or false,
		paramtype = "light",
		is_ground_content = false,
		drawtype = "nodebox",
		selection_box = {
			type = "fixed",
			fixed = nodebox,
		},
		node_box = {
			type = "fixed",
			fixed = nodebox,
		},
		groups = {
			handy = 1,
			attached_node = 1,
			not_in_creative_inventory = 1,
			dig_by_piston = 1,
			cake = level,
			comparator_signal = level * 2,
			food = 2,
			no_eat_delay = 1,
		},
		drop = "",
		on_rightclick = on_rightclick,
		sounds = mcl_sounds.node_sound_leaves_defaults(),

		_food_particles = false,
		_mcl_saturation = 0.6,
		_mcl_blast_resistance = 0.5,
		_mcl_hardness = 0.5,
	})

	if minetest.get_modpath("doc") then
		doc.add_entry_alias("nodes", "mcl_allthingschocolate:cake", "nodes", "mcl_allthingschocolate:cake_" .. level)
	end
end

register_slice(6, slice_6, S("Cake (6 Slices Left)"))
register_slice(5, slice_5, S("Cake (5 Slices Left)"))
register_slice(4, slice_4, S("Cake (4 Slices Left)"))
register_slice(3, slice_3, S("Cake (3 Slices Left)"))
register_slice(2, slice_2, S("Cake (2 Slices Left)"))
register_slice(1, slice_1, S("Cake (1 Slice Left)"))
