-- SPDX-License-Identifier: MIT
local get_current_modname = core.get_current_modname
local settings = core.settings
local settings_get = settings.get
local gmatch = string.gmatch
local trim = string.trim

local IE = core.request_insecure_environment()
if IE == nil then
    error("insecure environment not available, add unsafe_require to secure.trusted_mods")
end

--- code from [luanti issue 10877](https://github.com/luanti-org/luanti/issues/10877#issuecomment-1060033815)
---
--- the same as `IE.require(...)`, but sets the env to IE
local function require_with_IE_env(...)
    -- be sure that there is no hook, otherwise one could get IE via getfenv
    IE.debug.sethook()

    local old_thread_env = IE.getfenv(0)
    local old_string_metatable = IE.debug.getmetatable("")

    -- set env of thread
    -- (the loader used by IE.require will probably use the thread env for
    -- the loaded functions)
    IE.setfenv(0, IE)

    -- also set the string metatable because the lib might use it while loading
    -- (actually, we probably have to do this every time we call a `require()`d
    -- function, but for performance reasons we only do it if the function
    -- uses the string metatable)
    -- (Maybe it would make sense to set the string metatable __index field
    -- to a function that grabs the string table from the thread env.)
    IE.debug.setmetatable("", { __index = IE.string })

    -- (IE.require's env is neither _G, nor IE. we need to leave it like this,
    -- otherwise it won't find the loaders (it uses the global `loaders`, not
    -- `package.loaders` btw. (see luajit/src/lib_package.c)))

    -- we might be pcall()ed, so we need to pcall to make sure that we reset
    -- the thread env afterwards
    local ok, ret = IE.pcall(IE.require, ...)

    -- reset env of thread
    IE.setfenv(0, old_thread_env)

    -- also reset the string metatable
    IE.debug.setmetatable("", old_string_metatable)

    if not ok then
        IE.error(ret)
    end
    return ret
end

local function splitAndTrim(s)
    local ret = {}
    for part in gmatch(s, "([^,]+)") do
        local value = trim(part)
        if value ~= "" then
            ret[value] = true
        end
    end
    return ret
end

local function internal(name, callback, ...)
    local mod_name = get_current_modname()
    if mod_name == nil then
        error("unsafe_require can only be used in init.lua during startup")
    end
    local settings_key = "unsafe_require.allowed_for." .. mod_name
    local allowed_requires = core.settings:get(settings_key)
    if allowed_requires == nil then
        error("setting " .. settings_key .. " not found")
    end
    allowed_requires = splitAndTrim(allowed_requires)
    if allowed_requires[name] ~= true then
        error("setting " .. settings_key .. " does not contain '" .. name .. "'")
    end
    local mod = require_with_IE_env(name)
    return callback(mod, ...)
end

---@param name string lua module to load via require(name)
---@param callback function callback to be called with the module
---@param ... any additional parameters to be passed to the callback
---@return any returns the value returned by the callback
---@see require
function unsafe_require(name, callback, ...)
    return internal(name, callback, ...)
end
