nebula_admin = {}

local player_info = {}
local ie = core.request_insecure_environment()

local function get_mod_title(mod_name)
    local path = core.get_modpath(mod_name)
    if path then
        local conf = Settings(path .. "/mod.conf")
        if conf then
            local title = conf:get("title")
            if title then
                return title, true
            end
        end
    end

    return mod_name, false
end

function get_mod_settings(mod_name)
    local settings = {}

    local file = ie.io.open(core.get_modpath(mod_name) .. "/settingtypes.txt", "r")
    if not file then return settings end

    for line in file:lines() do
        -- Skip comments, empty lines, and sections
        if line:sub(1,1) ~= "#" and line:trim() ~= "" and line:sub(1,1) ~= "[" then
            local name, title, setting_type, default = line:match("^(%S+)%s+%(([^%)]+)%)%s+(%S+)%s+(%S+)")
            if name then
                local is_default = false
                local value = core.settings:get(name)

                if value == nil then
                    is_default = true
                    value = default
                end

                -- We use an array so we can keep the order of the settings
                table.insert(settings, {
                    name = name,
                    title = title,
                    value = value,
                    is_default = is_default,
                    default = default
                })
            end
        end
    end

    file:close()
    return settings
end

local function get_mods()
    local mods = {}

    for _, mod_name in ipairs(core.get_modnames()) do
        local title, has_title = get_mod_title(mod_name)

        local settings = get_mod_settings(mod_name)
        if #settings > 0 then
            table.insert(mods, {
                name = mod_name,
                title = title,
                has_title = has_title,
                settings = settings
            }) 
        end
    end

    table.sort(mods, function(a, b)
        if a.has_title and not b.has_title then
            return true
        elseif not a.has_title and b.has_title then
            return false
        end

        return a.title < b.title
    end)

    return mods
end


local function show_admin_panel_mods(player, selected_mod_name, scroll_pos)
    local mods = get_mods()
    if #mods > 0 then
        selected_mod_name = selected_mod_name
            or player_info[player:get_player_name()].admin_panel_mods_last_selected
            or mods[1].name
    end
    player_info[player:get_player_name()].admin_panel_mods_last_selected = selected_mod_name
    player_info[player:get_player_name()].admin_panel_mods_scroll_pos = scroll_pos or 0

    local formspec = "formspec_version[7]" ..
                    "size[24,16]" ..
                    "box[0,0;24,16;#333333]" ..
                    "box[0,0;3.8,16;#222222]"

    formspec = formspec .. "textlist[0.2,0.2;3.4,15.6;mod_list;"
    local selected_mod_index = 1
    for i, mod in ipairs(mods) do
        if i > 1 then
            formspec = formspec .. ","
        end
        formspec = formspec .. core.formspec_escape(mod.title)
        if mod.name == selected_mod_name then
            selected_mod_index = i
        end
    end
    formspec = formspec .. ";" .. selected_mod_index .. "]"

    if selected_mod_name then
        -- Hidden element used to query the selected mod when we recieve fields
        formspec = formspec .. "field[0,0;0,0;selected_mod_name;;" .. selected_mod_name .. "]"

        formspec = formspec ..
            "label[4,0.5;" .. core.formspec_escape(mods[selected_mod_index].title) .. "]" ..
            "box[4,1.2;19.5,0.1;#444444]" -- Seperator

        local settings = mods[selected_mod_index].settings
        local y_pos = 0.2
        local content_height = #settings * 10

        formspec = formspec .. "scrollbaroptions[min=0;max=" .. content_height .. ";smallstep=1;largestep=3]" ..
            "scrollbar[23.5,2;0.5,14;vertical;settings_scroll;" .. (scroll_pos or 0) .. "]" ..
            "scroll_container[4,2;19.5,14;settings_scroll;vertical]"

        for _, setting in ipairs(settings) do
            formspec = formspec ..
                "label[0," .. y_pos .. ";" ..core.formspec_escape(setting.title) .. "]"

            formspec = formspec ..
                "field[0," .. y_pos + 0.2 .. ";15.5,0.8;setting_" ..
                core.formspec_escape(setting.name) .. ";;" ..
                core.formspec_escape(tostring(setting.value))  .. "]"

            formspec = formspec .. "button[15.5," .. y_pos + 0.2 .. ";2,0.8;set_" .. setting.name .. ";Set]"
            if setting.is_default then
                formspec = formspec .. "button[17.5," .. y_pos + 0.2 .. ";2,0.8;noop_" .. setting.name .. ";" .. core.formspec_escape("[Default]") .. "]"
            else
                formspec = formspec .. "button[17.5," .. y_pos + 0.2 .. ";2,0.8;reset_"  .. setting.name .. ";Reset]"
            end

            y_pos = y_pos + 1.5
        end

        formspec = formspec .. "scroll_container_end[]"
    end

    core.show_formspec(player:get_player_name(), "nebula_admin:admin_panel_mods", formspec)
end

core.register_privilege("nebula_admin_panel", {
    description = "Allows access to the admin panel"
})

core.register_chatcommand("admin_panel", {
    description = "Shows the admin panel",
    privs = { nebula_admin_panel = true },
    func = function(name, param)
        if not ie then
            return false, "Nebula Admin needs to be trusted to use the admin panel."
        end

        local mods = get_mods()
        show_admin_panel_mods(core.get_player_by_name(name))
        return true
    end,
})

core.register_on_player_receive_fields(function(player, formname, fields)
    if formname ~= "nebula_admin:admin_panel_mods" then return false end
    if not core.check_player_privs(player:get_player_name(), { nebula_admin_panel = true }) then return false end

    if fields.mod_list then
        local event = core.explode_textlist_event(fields.mod_list)
        if event.type == "CHG" then
            local mods = get_mods()
            local selected_mod = mods[event.index]
            show_admin_panel_mods(player, selected_mod.name)
        end
    end
    
    if fields.selected_mod_name then
        local scroll_pos
        if fields.settings_scroll then
            local event = core.explode_scrollbar_event(fields.settings_scroll)
            if event.type == "CHG" then
                scroll_pos = event.value
            end
        end

        if scroll_pos then
            player_info[player:get_player_name()].admin_panel_mods_scroll_pos = scroll_pos
        else
            scroll_pos = player_info[player:get_player_name()].admin_panel_mods_scroll_pos or 0
        end

        for field_name, _ in pairs(fields) do
            if string.match(field_name, "^set_") then
                local setting_name = string.sub(field_name, 5) -- Removes the set_ prefix
                local setting_value = fields["setting_" .. setting_name]
                core.settings:set(setting_name, setting_value)
                core.settings:write()

                show_admin_panel_mods(player, fields.selected_mod_name, scroll_pos)
            end

            if string.match(field_name, "^reset_") then
                local setting_name = string.sub(field_name, 7) -- Removes the reset_ prefix
                core.settings:remove(setting_name)
                core.settings:write()

                show_admin_panel_mods(player, fields.selected_mod_name, scroll_pos)
            end
        end
    end

    return true
end)


core.register_on_joinplayer(function(player, last_login)
    player_info[player:get_player_name()] = {
        admin_panel_mods_scroll_pos = 0,
        admin_panel_mods_last_selected = nil
    }
end)

core.register_on_leaveplayer(function(player, timed_out)
    player_info[player:get_player_name()] = nil
end)
