nebula_admin = {}

local player_info = {}
local ie = core.request_insecure_environment()

local function get_mod_title(mod_name)
    local path = core.get_modpath(mod_name)
    if path then
        local conf = Settings(path .. "/mod.conf")
        if conf then
            local title = conf:get("title")
            if title then
                return title, true
            end
        end
    end

    return mod_name, false
end

function get_mod_settings(mod_name)
    local settings = {}

    local file = ie.io.open(core.get_modpath(mod_name) .. "/settingtypes.txt", "r")
    if not file then return settings end

    for line in file:lines() do
        -- Skip comments, empty lines, and sections
        if line:sub(1,1) ~= "#" and line:trim() ~= "" and line:sub(1,1) ~= "[" then
            local name, title, setting_type, default = line:match("^(%S+)%s+%(([^%)]+)%)%s+(%S+)%s+(%S+)")
            if name then
                local is_default = false
                local value = core.settings:get(name)

                if value == nil then
                    is_default = true
                    value = default
                end

                -- We use an array so we can keep the order of the settings
                table.insert(settings, {
                    name = name,
                    title = title,
                    value = value,
                    is_default = is_default,
                    default = default
                })
            end
        end
    end

    file:close()
    return settings
end

local function get_mods()
    local mods = {}

    for _, mod_name in ipairs(core.get_modnames()) do
        local title, has_title = get_mod_title(mod_name)

        local settings = get_mod_settings(mod_name)
        if #settings > 0 then
            table.insert(mods, {
                name = mod_name,
                title = title,
                has_title = has_title,
                settings = settings
            }) 
        end
    end

    table.sort(mods, function(a, b)
        if a.has_title and not b.has_title then
            return true
        elseif not a.has_title and b.has_title then
            return false
        end

        return a.title < b.title
    end)

    return mods
end


local function config_has_value(key)
    local file = ie.io.open(core.get_user_path() .. "/luanti.conf", "r")
    file = file or ie.io.open(core.get_user_path() .. "/minetest.conf", "r")
    if not file then return false end
    
    for line in file:lines() do
        -- Skip comments, empty lines, and sections
        if line:sub(1,1) ~= "#" and line:trim() ~= "" and line:sub(1,1) ~= "[" then
            local conf_key = line:match("^([^%s=#]+)")
            if conf_key and conf_key == key then
                file:close()
                return true
            end
        end
    end
    
    file:close()
    return false
end

local function get_server_config()
    local mods = {
        {
            name = "server",
            title = "Server",
            has_title = true,
            settings = {
                {
                    name = "name",
                    title = "Admin name"
                },
                {
                    name = "server_name",
                    title = "Server name"
                },
                {
                    name = "server_description",
                    title = "Server description"
                },
                {
                    name = "server_address",
                    title = "Server address"
                },
                {
                    name = "server_url",
                    title = "Server URL"
                },
                {
                    name = "server_announce",
                    title = "Announce server"
                },
                {
                    name = "server_announce_send_players",
                    title = "Send player names to the server list"
                },
                {
                    name = "serverlist_url",
                    title = "Serverlist URL"
                },
                {
                    name = "motd",
                    title = "Message of the day"
                },
                {
                    name = "max_users",
                    title = "Maximum users"
                },
                {
                    name = "static_spawnpoint",
                    title = "Static spawn point"
                },
                {
                    name = "port",
                    title = "Server port"
                },
                {
                    name = "bind_address",
                    title = "Bind address"
                },
                {
                    name = "strict_protocol_version_checking",
                    title = "Strict protocol checking"
                },
                {
                    name = "protocol_version_min",
                    title = "Protocol version minimum"
                },
                {
                    name = "remote_media",
                    title = "Remote media"
                },
                {
                    name = "ipv6_server",
                    title = "IPv6 server"
                }
            }
        },
        {
            name = "server_security",
            title = "Server Security",
            has_title = true,
            settings = {
                {
                    name = "default_password",
                    title = "Default password"
                },
                {
                    name = "disallow_empty_password",
                    title = "Disallow empty passwords"
                },
                {
                    name = "default_privs",
                    title = "Default privileges"
                },
                {
                    name = "basic_privs",
                    title = "Basic privileges",
                    default = "interact, shout"
                },
                {
                    name = "anticheat_flags",
                    title = "Anticheat flags"
                },
                {
                    name = "anticheat_movement_tolerance",
                    title = "Anticheat movement tolerance"
                },
                {
                    name = "enable_rollback_recording",
                    title = "Rollback recording"
                },
                {
                    name = "csm_restriction_flags",
                    title = "Client side modding restrictions"
                },
                {
                    name = "csm_restriction_noderange",
                    title = "Client-side node lookup range restriction"
                },
                {
                    name = "strip_color_codes",
                    title = "Strip color codes"
                },
                {
                    name = "chat_message_max_size",
                    title = "Chat message max length"
                },
                {
                    name = "chat_message_limit_per_10sec",
                    title = "Chat message count limit"
                },
                {
                    name = "chat_message_limit_trigger_kick",
                    title = "Chat message kick threshold"
                }
            }
        },
        {
            name = "server_gameplay",
            title = "Server Gameplay",
            has_title = true,
            settings = {
                {
                    name = "time_speed",
                    title = "Time speed"
                },
                {
                    name = "world_start_time",
                    title = "World start time"
                },
                {
                    name = "item_entity_ttl",
                    title = "Item entity TTL",
                    default = 900
                },
                {
                    name = "default_stack_max",
                    title = "Default stack size",
                    default = 99
                },
                {
                    name = "movement_acceleration_default",
                    title = "Default acceleration"
                },
                {
                    name = "movement_acceleration_air",
                    title = "Acceleration in air"
                },
                {
                    name = "movement_acceleration_fast",
                    title = "Fast mode acceleration"
                },
                {
                    name = "movement_speed_walk",
                    title = "Walking speed"
                },
                {
                    name = "movement_speed_crouch",
                    title = "Sneaking speed"
                },
                {
                    name = "movement_speed_fast",
                    title = "Fast mode speed"
                },
                {
                    name = "movement_speed_climb",
                    title = "Climbing speed"
                },
                {
                    name = "movement_speed_jump",
                    title = "Jumping speed"
                },
                {
                    name = "movement_liquid_fluidity",
                    title = "Liquid fluidity"
                },
                {
                    name = "movement_liquid_fluidity_smooth",
                    title = "Liquid fluidity smoothing"
                },
                {
                    name = "movement_liquid_sink",
                    title = "Liquid sinking"
                },
                {
                    name = "movement_gravity",
                    title = "Gravity"
                }
            }
        },
        {
            name = "advanced",
            title = "Advanced",
            has_title = true,
            settings = {
                {
                    name = "enable_ipv6",
                    title = "IPv6"
                },
                {
                    name = "ignore_world_load_errors",
                    title = "Ignore world errors"
                },
                {
                    name = "chat_message_format",
                    title = "Chat message format"
                },
                {
                    name = "chatcommand_msg_time_threshold",
                    title = "Chat command time message threshold",
                    default = 0.1
                },
                {
                    name = "kick_msg_shutdown",
                    title = "Shutdown message"
                },
                {
                    name = "kick_msg_crash",
                    title = "Crash message"
                },
                {
                    name = "ask_reconnect_on_crash",
                    title = "Ask to reconnect after crash"
                },
                {
                    name = "dedicated_server_step",
                    title = "Dedicated server step"
                },
                {
                    name = "unlimited_player_transfer_distance",
                    title = "Unlimited player transfer distance (Deprecated)",
                    default = true
                },
                {
                    name = "player_transfer_distance",
                    title = "Player transfer distance"
                },
                {
                    name = "active_object_send_range_blocks",
                    title = "Active object send range"
                },
                {
                    name = "active_block_range",
                    title = "Active block range"
                },
                {
                    name = "max_block_send_distance",
                    title = "Max block send distance"
                },
                {
                    name = "max_forceloaded_blocks",
                    title = "Maximum forceloaded blocks",
                    default = 16
                },
                {
                    name = "time_send_interval",
                    title = "Time send interval"
                },
                {
                    name = "server_map_save_interval",
                    title = "Map save interval"
                },
                {
                    name = "server_unload_unused_data_timeout",
                    title = "Unload unused server data"
                },
                {
                    name = "max_objects_per_block",
                    title = "Maximum objects per block"
                },
                {
                    name = "active_block_mgmt_interval",
                    title = "Active block management interval"
                },
                {
                    name = "abm_interval",
                    title = "ABM interval"
                },
                {
                    name = "abm_time_budget",
                    title = "ABM time budget"
                },
                {
                    name = "nodetimer_interval",
                    title = "NodeTimer interval"
                },
                {
                    name = "liquid_loop_max",
                    title = "Liquid loop max"
                },
                {
                    name = "liquid_queue_purge_time",
                    title = "Liquid queue purge time"
                },
                {
                    name = "liquid_update",
                    title = "Liquid update tick"
                },
                {
                    name = "block_send_optimize_distance",
                    title = "Block send optimize distance"
                },
                {
                    name = "server_side_occlusion_culling",
                    title = "Server-side occlusion culling"
                },
                {
                    name = "block_cull_optimize_distance",
                    title = "Block cull optimize distance"
                }
            }
        }
    }

    for _, mod in ipairs(mods) do
        for _, setting in ipairs(mod.settings) do
            setting.value = core.settings:get(setting.name)

            -- Some settings give us the default back, while others give us nil
            -- which is why we have this check
            if setting.value == nil then
                setting.value = setting.default
                setting.is_default = true
            else
                setting.is_default = not config_has_value(setting.name)
                if setting.is_default then
                    setting.default = setting.value
                end
            end
        end
    end

    return mods
end


local function show_admin_panel_mods(player, selected_mod_name, scroll_pos, server)
    local mods
    if server then
        mods = get_server_config()
        selected_mod_name = selected_mod_name or player_info[player:get_player_name()].admin_panel_server_last_selected
        player_info[player:get_player_name()].admin_panel_server_last_selected = selected_mod_name
    else
        mods = get_mods()
        selected_mod_name = selected_mod_name or player_info[player:get_player_name()].admin_panel_mods_last_selected
        player_info[player:get_player_name()].admin_panel_mods_last_selected = selected_mod_name
    end

    if #mods > 0 then
        selected_mod_name = selected_mod_name or mods[1].name
    end

    player_info[player:get_player_name()].admin_panel_mods_scroll_pos = scroll_pos or 0

    local formspec = "formspec_version[7]" ..
                    "size[24,16]" ..
                    "box[0,0;24,16;#333333]" ..
                    "box[0,0;3.8,16;#222222]"

    formspec = formspec .. "textlist[0.2,0.2;3.4,15.6;mod_list;"
    local selected_mod_index = 1
    for i, mod in ipairs(mods) do
        if i > 1 then
            formspec = formspec .. ","
        end
        formspec = formspec .. core.formspec_escape(mod.title)
        if mod.name == selected_mod_name then
            selected_mod_index = i
        end
    end
    formspec = formspec .. ";" .. selected_mod_index .. "]"

    if selected_mod_name then
        -- Hidden element used to query the selected mod when we receive fields
        formspec = formspec .. "field[0,0;0,0;selected_mod_name;;" .. selected_mod_name .. "]"

        formspec = formspec ..
            "label[4,0.5;" .. core.formspec_escape(mods[selected_mod_index].title) .. "]" ..
            "box[4,1.2;19.5,0.1;#444444]" -- Seperator

        local settings = mods[selected_mod_index].settings
        local y_pos = 0.2
        local content_height = #settings * 10

        formspec = formspec .. "scrollbaroptions[min=0;max=" .. content_height .. ";smallstep=1;largestep=3]" ..
            "scrollbar[23.5,2;0.5,14;vertical;settings_scroll;" .. (scroll_pos or 0) .. "]" ..
            "scroll_container[4,2;19.5,14;settings_scroll;vertical]"

        for _, setting in ipairs(settings) do
            formspec = formspec ..
                "label[0," .. y_pos .. ";" ..core.formspec_escape(setting.title) .. "]"

            formspec = formspec ..
                "field[0," .. y_pos + 0.2 .. ";15.5,0.8;setting_" ..
                core.formspec_escape(setting.name) .. ";;" ..
                core.formspec_escape(tostring(setting.value))  .. "]"

            formspec = formspec .. "button[15.5," .. y_pos + 0.2 .. ";2,0.8;set_" .. setting.name .. ";Set]"
            if setting.is_default then
                formspec = formspec .. "button[17.5," .. y_pos + 0.2 .. ";2,0.8;noop_" .. setting.name .. ";" .. core.formspec_escape("[Default]") .. "]"
            else
                formspec = formspec .. "button[17.5," .. y_pos + 0.2 .. ";2,0.8;reset_"  .. setting.name .. ";Reset]"
            end

            y_pos = y_pos + 1.5
        end

        formspec = formspec .. "scroll_container_end[]"
    end

    if server then
        core.show_formspec(player:get_player_name(), "nebula_admin:admin_panel_server", formspec)
    else
        core.show_formspec(player:get_player_name(), "nebula_admin:admin_panel_mods", formspec)
    end
end


core.register_on_player_receive_fields(function(player, formname, fields)
    if formname ~= "nebula_admin:admin_panel_mods" and formname ~= "nebula_admin:admin_panel_server" then return false end
    if not core.check_player_privs(player:get_player_name(), { nebula_admin_panel = true }) then return false end

    if fields.mod_list then
        local event = core.explode_textlist_event(fields.mod_list)
        if event.type == "CHG" then
            local mods
            if formname == "nebula_admin:admin_panel_server" then
                mods = get_server_config()
            else
                mods = get_mods()
            end

            local selected_mod = mods[event.index]
            show_admin_panel_mods(player, selected_mod.name, nil, formname == "nebula_admin:admin_panel_server")
        end
    end
    
    if fields.selected_mod_name then
        local scroll_pos
        if fields.settings_scroll then
            local event = core.explode_scrollbar_event(fields.settings_scroll)
            if event.type == "CHG" then
                scroll_pos = event.value
            end
        end

        if scroll_pos then
            player_info[player:get_player_name()].admin_panel_mods_scroll_pos = scroll_pos
        else
            scroll_pos = player_info[player:get_player_name()].admin_panel_mods_scroll_pos or 0
        end

        for field_name, _ in pairs(fields) do
            if string.match(field_name, "^set_") then
                local setting_name = string.sub(field_name, 5) -- Removes the set_ prefix
                local setting_value = fields["setting_" .. setting_name]
                core.settings:set(setting_name, setting_value)
                core.settings:write()

                show_admin_panel_mods(player, fields.selected_mod_name, scroll_pos, formname == "nebula_admin:admin_panel_server")
            end

            if string.match(field_name, "^reset_") then
                local setting_name = string.sub(field_name, 7) -- Removes the reset_ prefix
                core.settings:remove(setting_name)
                core.settings:write()

                show_admin_panel_mods(player, fields.selected_mod_name, scroll_pos, formname == "nebula_admin:admin_panel_server")
            end
        end
    end

    return true
end)


local function show_admin_panel(player)
    local formspec =
        "formspec_version[7]" ..
        "size[8,2]" ..
        "button[0.5,0.5;3.25,1;mods;Mods]" ..
        "button[4.25,0.5;3.25,1;server;Server]"

    core.show_formspec(player:get_player_name(), "nebula_admin:admin_panel", formspec)
end



core.register_on_player_receive_fields(function(player, formname, fields)
    if formname ~= "nebula_admin:admin_panel" then return false end
    if not core.check_player_privs(player:get_player_name(), { nebula_admin_panel = true }) then return false end

    if fields.mods then
        show_admin_panel_mods(player)
    elseif fields.server then
        show_admin_panel_mods(player, nil, nil, true)
    end

    return true
end)



core.register_privilege("nebula_admin_panel", {
    description = "Allows access to the admin panel"
})


core.register_chatcommand("admin_panel", {
    description = "Shows the admin panel",
    privs = { nebula_admin_panel = true },
    func = function(name, param)
        if not ie then
            return false, "Nebula Admin needs to be trusted to use the admin panel."
        end

        show_admin_panel(core.get_player_by_name(name))
        return true
    end,
})

core.register_on_joinplayer(function(player, last_login)
    player_info[player:get_player_name()] = {
        admin_panel_mods_scroll_pos = 0,
        admin_panel_mods_last_selected = nil,
        admin_panel_server_last_selected = nil
    }
end)

core.register_on_leaveplayer(function(player, timed_out)
    player_info[player:get_player_name()] = nil
end)
