nebula_anticheat = {}

local history_length = 200
local player_info = {}
local checks = {}

-- TODO: Maybe rename info to data in the checks
-- and player_info to player_data

nebula_anticheat.engine_version_data = {
    ["5.10.0"] = {
        protocol_version = 46,
        formspec_version = 8,
        serialization_version = 29
    },
    ["5.9.1"] = {
        protocol_version = 45,
        formspec_version = 7,
        serialization_version = 29
    },
    ["5.9.0"] = {
        protocol_version = 44,
        formspec_version = 7,
        serialization_version = 29
    },
    ["5.8.1"] = {
        protocol_version = 43,
        formspec_version = 7,
        serialization_version = 29
    },
    ["5.8.0"] = {
        protocol_version = 43,
        formspec_version = 7,
        serialization_version = 29
    },
    ["5.7.0"] = {
        protocol_version = 42,
        formspec_version = 6,
        serialization_version = 29
    },
    ["5.6.1"] = {
        protocol_version = 41,
        formspec_version = 6,
        serialization_version = 29,
    },
    ["5.6.0"] = {
        protocol_version = 41,
        formspec_version = 6,
        serialization_version = 29
    },
    ["5.5.1"] = {
        protocol_version = 40,
        formspec_version = 5,
        serialization_version = 29
    },
    ["5.5.0"] = {
        protocol_version = 40,
        formspec_version = 5,
        serialization_version = 29
    },
    ["5.4.2"] = {
        protocol_version = 39,
        formspec_version = 4,
        serialization_version = 28
    },
    ["5.4.1"] = {
        protocol_version = 39,
        formspec_version = 4,
        serialization_version = 28
    },
    ["5.4.0"] = {
        protocol_version = 39,
        formspec_version = 4,
        serialization_version = 28
    },
    ["5.3.0"] = {
        protocol_version = 39,
        formspec_version = 3,
        serialization_version = 28
    }
}

function nebula_anticheat.get_engine_data(major, minor, patch)
    local engine_version = ""
    if patch ~= nil then
        engine_version = major .. "." .. minor .. "." .. patch
    elseif major ~= nil then
        engine_version = major
    else
        engine = core.get_version().string
    end

    return nebula_anticheat.engine_version_data[engine_version]
end

function nebula_anticheat.get_matching_engine_versions(protocol_version, formspec_version, serialization_version)
    local engine_versions = {}
    for key, value in pairs(nebula_anticheat.engine_version_data) do
        if (not protocol_version or protocol_version == value.protocol_version)
            and (not formspec_version or formspec_version == value.formspec_version)
            and (not serialization_version or serialization_version == value.serialization_version)
            then
            table.insert(engine_versions, key)
        end
    end

    return engine_versions
end

function nebula_anticheat.get_highest_known_protocol_version()
    local highest_protocol_version = -1
    for key, value in pairs(nebula_anticheat.engine_version_data) do
        if value.protocol_version > highest_protocol_version then
            highest_protocol_version = value.protocol_version
        end
    end

    return highest_protocol_version
end

core.register_privilege("nebula_anticheat_alert", {
    description = "Allows the player to view anti-cheat alerts"
})

local function split_command(input)
    local space_pos = input:find(" ")

    if space_pos then
        local command = input:sub(1, space_pos - 1)
        local param = input:sub(space_pos + 1)
        return command, param
    else
        return input, ""
    end
end

function nebula_anticheat.flag(player, check_name)
    local check = checks[check_name]
    local time = core.get_server_uptime()
    local player_name = player:get_player_name()
    if not player_info[player_name].violations[check_name] then player_info[player_name].violations[check_name] = {} end
    local violations = player_info[player_name].violations[check_name]

    -- Prevents flags from happening too fast
    if check.violation_delay and check.violation_delay >= 0 and #violations > 0 and time - violations[#violations] < check.violation_delay then
        return
    end

    -- Iterate backwards and remove anything too old
    for i = #violations, 1, -1 do
        if check.violation_period and check.violation_period >= 0 and time - violations[i] > check.violation_period then
            table.remove(violations, i)
        end
    end

    -- Add the violation/flag
    table.insert(violations, time)

    if check.violation_punish and check.violation_punish >= 1 and #violations >= check.violation_punish then
        -- The punishment command might not disconnect the player, so we reset their violations
        player_info[player_name].violations[check_name] = {}

        local punishment_command, punishment_params = split_command(core.settings:get("nebula_anticheat_punishment_command") or "kick @player Cheating detected by Nebula Anti-Cheat (NAC). If this is incorrect, you may rejoin.")
        if #punishment_command > 0 then
            local cmd = core.registered_chatcommands[punishment_command]
            if cmd then
                -- Runs the command as the admin user
                cmd.func(core.settings:get("name"), punishment_params:gsub("@player", player_name):gsub("@check", check.title))
            end
        end

        for _, loop_player in ipairs(core.get_connected_players()) do
            local loop_player_name = loop_player:get_player_name()
            if core.check_player_privs(loop_player_name, { nebula_anticheat_alert = true }) then
                core.chat_send_player(loop_player_name, core.colorize("#FF4B4B", "[") .. core.colorize("#FF9F1C", "NAC") .. core.colorize("#FF4B4B", "] ")
                    .. core.colorize("#FFFFFF", player_name)
                    .. core.colorize("#CCCCCC", " was punished for using ")
                    .. core.colorize("#FFFFFF", check.title)
                    .. core.colorize("#CCCCCC", "."))
            end
        end
    elseif (check.violation_alert and check.violation_alert >= 1 and #violations >= check.violation_alert)
        or core.settings:get_bool("nebula_anticheat_verbose_enabled", false) then
        for _, loop_player in ipairs(core.get_connected_players()) do
            local loop_player_name = loop_player:get_player_name()
            if core.check_player_privs(loop_player_name, { nebula_anticheat_alert = true }) then
                core.chat_send_player(loop_player_name, core.colorize("#FF4B4B", "[") .. core.colorize("#FF9F1C", "NAC") .. core.colorize("#FF4B4B", "] ")
                    .. core.colorize("#FFFFFF", player_name)
                    .. core.colorize("#CCCCCC", " was caught using ")
                    .. core.colorize("#FFFFFF", check.title)
                    .. core.colorize("#FF9F1C", " #" .. #violations))
            end
        end
    end
end

function nebula_anticheat.register_check(name, info)
    checks[name] = info
end

function nebula_anticheat.clear_history(player)
    player_info[player:get_player_name()].position_history = {}
    player_info[player:get_player_name()].velocity_history = {}
    player_info[player:get_player_name()].ground_history = {}
    player_info[player:get_player_name()].in_wall_history = {}
    player_info[player:get_player_name()].max_speed_history = {}
end

core.register_on_joinplayer(function(player, last_login)
    player_info[player:get_player_name()] = {
        violations = {}
    }
    nebula_anticheat.clear_history(player)
end)

core.register_on_leaveplayer(function(player, timed_out)
    player_info[player:get_player_name()] = nil
end)

core.register_on_respawnplayer(function(player)
    nebula_anticheat.clear_history(player)
end)

function nebula_anticheat.is_setback_enabled()
    return core.settings:get_bool("nebula_anticheat_setback_enabled", true)
end

function nebula_anticheat.setback(player, pos)
    player:set_pos(pos)
    nebula_anticheat.clear_history(player)
end

local function update_history(history_table, new_value)
    table.insert(history_table, 1, new_value)
    if #history_table > history_length then
        table.remove(history_table)
    end
end

local function on_ground(player)
    local pos = player:get_pos()
    local allowed_dist = 0.4
    local points = {
        {x = pos.x, y = pos.y - 0.5, z = pos.z},
        {x = pos.x + allowed_dist, y = pos.y - 0.5, z = pos.z},
        {x = pos.x - allowed_dist, y = pos.y - 0.5, z = pos.z},
        {x = pos.x, y = pos.y - 0.5, z = pos.z + allowed_dist},
        {x = pos.x, y = pos.y - 0.5, z = pos.z - allowed_dist},
    }
    
    for _, check_pos in ipairs(points) do
        local node = core.get_node(check_pos)
        local node_def = core.registered_nodes[node.name]
        if node_def and (node_def.walkable or node_def.liquidtype ~= "none" or node_def.climbable) then
            return true
        end
    end
    
    return false
end

local function eq_for(arr, prop)
    local last_value = nil
    for i, v in ipairs(arr) do
        if i > 1 then
            if prop then
                if v[prop] ~= last_value then
                    return i - 1
                end
            else
                if v ~= last_value then
                    return i - 1
                end
            end
        end
        if prop then
            last_value = v[prop]
        else
            last_value = v
        end
    end
    return #arr
end

local function ge_for(arr, prop, margin)
    if not margin then margin = 0 end
    local last_value = nil
    for i, v in ipairs(arr) do
        if i > 1 then
            if prop then
                if v[prop] + margin < last_value then
                    return i - 1
                end
            else
                if v + margin < last_value then
                    return i - 1
                end
            end
        end
        if prop then
            last_value = v[prop] + margin
        else
            last_value = v + margin
        end
    end
    return #arr
end

local function le_for(arr, prop, margin)
    if not margin then margin = 0 end
    local last_value = nil
    for i, v in ipairs(arr) do
        if i > 1 then
            if prop then
                if v[prop] - margin > last_value then
                    return i - 1
                end
            else
                if v - margin > last_value then
                    return i - 1
                end
            end
        end
        if prop then
            last_value = v[prop] - margin
        else
            last_value = v - margin
        end
    end
    return #arr
end

nebula_anticheat.register_check("fly_a", {
    title = "Fly (A)",
    violation_delay = 2,
    violation_period = 60 * 5,
    violation_alert = 5,
    violation_punish = 10,
    globalstep = function(player, info, flag)
        if not core.settings:get_bool("nebula_anticheat_fly_a_enabled", true) then return end
        if core.check_player_privs(player, { fly = true }) then return end

        local y_margin = 0.5
        if info.ground_history[1] == false and eq_for(info.ground_history) >= 70 and info.velocity_history[1].y >= -y_margin and le_for(info.velocity_history, "y", y_margin) >= 30 then
            if nebula_anticheat.is_setback_enabled() then
                local old_pos = info.position_history[eq_for(info.ground_history)+1] or info.position_history[#info.position_history]
                nebula_anticheat.setback(player, old_pos)
            end

            flag()
        end
    end
})

nebula_anticheat.register_check("fly_b", {
    title = "Fly (B)",
    violation_delay = 1,
    violation_period = 60 * 3,
    violation_alert = 5,
    violation_punish = 10,
    globalstep = function(player, info, flag)
        if not core.settings:get_bool("nebula_anticheat_fly_b_enabled", true) then return end
        if core.check_player_privs(player, { fly = true }) then return end
        if #info.position_history < 2 then return end
        if #info.velocity_history < 2 then return end

        local vel = info.velocity_history[1]
        local last_vel = info.velocity_history[2]
        local last_last_vel = info.velocity_history[2]
        local vel_y_diff = vel.y - last_vel.y
        local last_vel_y_diff = last_vel.y - last_last_vel.y
        local vel_y_diffs = vel_y_diff - last_vel_y_diff

        if not info.ground_history[1] and eq_for(info.ground_history) >= 70 and vel_y_diff >= 0 and last_vel_y_diff >= 0 and vel_y_diffs > 0 then
            if nebula_anticheat.is_setback_enabled() then
                local old_pos = info.position_history[eq_for(info.ground_history)+1] or info.position_history[#info.position_history]
                nebula_anticheat.setback(player, old_pos)
            end

            flag()
        end
    end
})

nebula_anticheat.register_check("speed_a", {
    title = "Speed (A)",
    violation_delay = 1,
    violation_period = 60 * 5,
    violation_alert = 5,
    violation_punish = 10,
    globalstep = function(player, info, flag)
        if not core.settings:get_bool("nebula_anticheat_speed_a_enabled", false) then return end
        if core.check_player_privs(player, { fast = true }) then return end
        if #info.position_history < 60 then return end
        if eq_for(info.max_speed_history) < 20 then return end

        -- the +1 is to give some breathing room
        local max_speed = info.max_speed_history[1] + 1
        local vel_speed = math.hypot(player:get_velocity().x, player:get_velocity().z)
        local pos_speed = math.hypot(info.position_history[1].x - info.position_history[2].x, info.position_history[1].z - info.position_history[2].z) * 10

        -- Maybe also check if the speed is increasing
        if pos_speed > max_speed or vel_speed > max_speed then
            if nebula_anticheat.is_setback_enabled() then
                local old_pos = info.position_history[2]
                nebula_anticheat.setback(player, old_pos)
            end

            flag()
        end
    end
})

nebula_anticheat.register_check("motion_a", {
    title = "Motion (A)",
    violation_delay = 1,
    violation_period = 60 * 5,
    violation_alert = 5,
    violation_punish = 10,
    globalstep = function(player, info, flag)
        if not core.settings:get_bool("nebula_anticheat_motion_a_enabled", true) then return end
        if core.check_player_privs(player, { fast = true }) then return end
        if #info.position_history < 2 then return end

        -- Luanti's built-in anti-cheat allows for infinite downward motion, and ~65 blocks upwards

        local dY = info.position_history[1].y - info.position_history[2].y
        if math.abs(dY) > 10 then
            if nebula_anticheat.is_setback_enabled() then
                local old_pos = info.position_history[2]
                nebula_anticheat.setback(player, old_pos)
            end

            flag()
        end
    end
})

nebula_anticheat.register_check("motion_b", {
    title = "Motion (B)",
    violation_delay = 1,
    violation_period = 60 * 3,
    violation_alert = 5,
    violation_punish = 10,
    globalstep = function(player, info, flag)
        if not core.settings:get_bool("nebula_anticheat_motion_b_enabled", true) then return end
        if core.check_player_privs(player, { fly = true }) then return end

        if info.ground_history[1] == true and eq_for(info.ground_history) >= 10 then
            local last_on_ground_pos_y = math.abs(info.position_history[2].y  % 1) == 0.5
            if last_on_ground_pos_y and info.velocity_history[1].y < 0 and not info.in_wall_history[1] then
                if nebula_anticheat.is_setback_enabled() then
                    local old_pos = info.position_history[2]
                    nebula_anticheat.setback(player, old_pos)
                end

                flag()
            end
        end
    end
})

nebula_anticheat.register_check("bad_client_a", {
    title = "Bad Client (A)",
    violation_punish = 1,
    joinplayer = function(player, info, flag)
        if not core.settings:get_bool("nebula_anticheat_bad_client_a_enabled", true) then return end

        local core_player_info = core.get_player_information(player:get_player_name())

        -- Only available when the server is in debug mode
        if core_player_info.version_string then
            local bad_clients = {
                ["5.9.0-dev-196a7d3"] = "Otter Client (v1.0.1)",
                ["5.9.0-dev-4b62aed"] = "Otter Client (v1.0.0)",
                ["5.6.0-350b6d175"] = "Dragonfire (2022.05)",
                ["5.5.0-b7abc8df2"] = "Dragonfire (2021.05)",
                ["5.5.0-c47eae316"] = "Dragonfire (2021.03)",
                ["5.4.0--dragonfire"] = "waspsaliva"
            }

            if bad_clients[core_player_info.version_string] then
                flag()
            end
        end
    end
})



nebula_anticheat.register_check("invalid_version_a", {
    title = "Invalid Version (A)",
    violation_punish = 1,
    joinplayer = function(player, info, flag)
        if not core.settings:get_bool("nebula_anticheat_invalid_version_a_enabled", true) then return end

        -- Checks for an unexpected protocol, formspec, and serialization version based on the client version
        local core_player_info = core.get_player_information(player:get_player_name())

        -- Only available when the server is in debug mode
        if core_player_info.major and core_player_info.minor and core_player_info.patch then
            local engine_data = nebula_anticheat.get_engine_data(core_player_info.major, core_player_info.minor, core_player_info.patch)
            if engine_data ~= nil then
                if core_player_info.protocol_version ~= engine_data.protocol_version
                    or core_player_info.formspec_version ~= engine_data.formspec_version
                    or (core_player_info.serialization_version ~= nil and core_player_info.serialization_version ~= engine_data.serialization_version)
                    then
                    flag()
                end
            end
        else
            -- Fallback version

            -- If their protocol version is equal to the highest we know about, then we shouldn't do the check
            -- Since a newer version may have released with the same protocol version
            -- But we should have all data about previous releases
            -- And we might as well not check newer versions for optimization
            if core_player_info.protocol_version < nebula_anticheat.get_highest_known_protocol_version() then
                local engine_versions = nebula_anticheat.get_matching_engine_versions(
                    core_player_info.protocol_version,
                    core_player_info.formspec_version,
                    core_player_info.serialization_version
                )

                if #engine_versions == 0 then
                    flag()
                end
            end
        end
    end
})

nebula_anticheat.register_check("invalid_version_b", {
    title = "Invalid Version (B)",
    violation_punish = 1,
    joinplayer = function(player, info, flag)
        if not core.settings:get_bool("nebula_anticheat_invalid_version_b_enabled", true) then return end

        -- Compares the client version string to the client version
        local core_player_info = core.get_player_information(player:get_player_name())

        -- Only available when the server is in debug mode
        if core_player_info.version_string and core_player_info.major and core_player_info.minor and core_player_info.patch then
            local pos = string.find(core_player_info.version_string, "-")
            local result = pos and string.sub(core_player_info.version_string, 1, pos-1) or core_player_info.version_string
            if result ~= core_player_info.major .. "." .. core_player_info.minor .. "." .. core_player_info.patch then
                flag()
            end
        end
    end
})

-- There are many reasons this can occur without cheating, so this will only setback and not punish
nebula_anticheat.register_check("noclip_a", {
    title = "Noclip (A)",
    violation_delay = 0.5, -- For verbose mode
    globalstep = function(player, info, flag)
        if not core.settings:get_bool("nebula_anticheat_noclip_a_enabled", false) then return end
        if core.check_player_privs(player, { noclip = true }) then return end

        local in_wall_for = eq_for(info.in_wall_history)
        if info.in_wall_history[1] == true and in_wall_for >= 70 then
            if nebula_anticheat.is_setback_enabled() then
                local old_pos = info.position_history[in_wall_for+1] or info.position_history[#info.position_history]
                nebula_anticheat.setback(player, old_pos)
            end

            flag()
        end
    end
})

core.register_on_joinplayer(function(player, last_login)
    local info = player_info[player:get_player_name()]

    -- Checks
    for name, check in pairs(checks) do
        if check.joinplayer then
            local function flag()
                nebula_anticheat.flag(player, name)
            end

            check.joinplayer(player, info, flag)
        end
    end
end)

core.register_globalstep(function(dtime)
    local players = core.get_connected_players()
    for _, player in ipairs(players) do
        local info = player_info[player:get_player_name()]

        -- History
        local pos = player:get_pos()
        update_history(info.position_history, pos)
        update_history(info.velocity_history, player:get_velocity())
        update_history(info.ground_history, on_ground(player))

        local passable_nodes = core.find_nodes_in_area({x = pos.x-1, y = pos.y-1, z = pos.z-1}, {x = pos.x+1, y = pos.y+1, z = pos.z+1}, {"air", "group:liquid", "group:climbable"})
        update_history(info.in_wall_history, #passable_nodes == 0)

        update_history(info.max_speed_history, player:get_physics_override().speed * player:get_physics_override().speed_walk * 4)

        -- Checks
        for name, check in pairs(checks) do
            if check.globalstep then
                local function flag()
                    nebula_anticheat.flag(player, name)
                end

                check.globalstep(player, info, flag)
            end
        end
    end
end)
