nebula_combat = {}

-- Defaults must be put here due to the issue below
-- https://github.com/minetest/minetest/issues/6711

local config_custom_knockback_enabled = core.settings:get_bool("nebula_combat_custom_knockback_enabled", true)
local config_custom_attack_distance_enabled = core.settings:get_bool("nebula_combat_custom_attack_distance_enabled", false)

-- Knockback - Horizontal
local config_horizontal_momentum = tonumber(core.settings:get("nebula_combat_horizontal_momentum") or 0.5)
local config_horizontal_force = tonumber(core.settings:get("nebula_combat_horizontal_force") or 8.0)

-- Knockback - Horizontal - Sprinting
local config_horizontal_different_when_sprinting = core.settings:get_bool("nebula_combat_horizontal_different_when_sprinting", false)
local config_horizontal_momentum_sprinting = tonumber(core.settings:get("nebula_combat_horizontal_momentum_sprinting") or 0.5)
local config_horizontal_force_sprinting = tonumber(core.settings:get("nebula_combat_horizontal_force_sprinting") or 10.0)

-- Knockback - Vertical
local config_vertical_momentum = tonumber(core.settings:get("nebula_combat_vertical_momentum") or 0.5)
local config_vertical_force = tonumber(core.settings:get("nebula_combat_vertical_force") or 6.0)

-- Knockback - Vertical - Sprinting
local config_vertical_different_when_sprinting = core.settings:get_bool("nebula_combat_vertical_different_when_sprinting", false)
local config_vertical_momentum_sprinting = tonumber(core.settings:get("nebula_combat_vertical_momentum_sprinting") or 0.5)
local config_vertical_force_sprinting = tonumber(core.settings:get("nebula_combat_vertical_force_sprinting") or 6.0)

-- Knockback - Physics
local config_physics_duration = tonumber(core.settings:get("nebula_combat_knockback_physics_duration") or 0.25)
local config_speed_mult = tonumber(core.settings:get("nebula_combat_knockback_speed_mult") or 1.0)
local config_gravity_mult = tonumber(core.settings:get("nebula_combat_knockback_gravity_mult") or 1.0)
local config_air_accel_mult = tonumber(core.settings:get("nebula_combat_knockback_air_accel_mult") or 1.0)

-- Misc
local config_trust_player_monoids = core.settings:get_bool("nebula_combat_trust_player_monoids", false)
local config_use_player_monoids_for_physics = core.settings:get_bool("nebula_combat_use_player_monoids_for_physics", false)
local config_use_pova_for_physics = core.settings:get_bool("nebula_combat_use_pova_for_physics", false)
local config_interact_distance = tonumber(core.settings:get("nebula_combat_interact_distance") or 3.0)

local player_info = {}

-- Checks if a player is sprinting
function nebula_combat.is_sprinting(player)
    -- This checks if any mods claim the player is sprinting

    -- Some sprint mods allow sprinting while standing still, which shouldn't be the case
    -- We could also verify that they are moving forwards but we don't incase sideways/backwards sprint is desired
    -- If that's not the case, that should be fixed in the sprint mod
    local control = player:get_player_control()
    if control.movement_x and control.movement_x == 0 and control.movement_y == 0 then
        return false
    elseif not control.up and not control.down and not control.left and not control.right then
        -- movement_x/y is new so we have this fallback
        return false
    end

    -- For Stamina (Fork), not regular Stamina
    if stamina.players then
        if stamina.players[player:get_player_name()].sprint then
            return true
        end
    end

    if minetest_wadsprint then
        local stats = minetest_wadsprint.api.stats(player:get_player_name())
        if stats.is_sprinting then
            return true
        end
    end

    if player:get_meta():get_int("real_stamina:sprinting") == 1 then
        return true
    end

    if player_monoids and config_trust_player_monoids then
        local speed = player_monoids.speed:value(player)
        if speed > 1 then
            return true
        end
    end

    -- Covers mods that don't have a direct way to query sprinting, but also falls back to false
    return player_info[player:get_player_name()].sprinting
end


-- Sets a player to sprinting (or not sprinting)
-- ID might be used in the future to support multiple sprint mods at once, so they don't conflict
function nebula_combat.set_sprinting(player, sprinting, id)
    player_info[player:get_player_name()].sprinting = sprinting
end

-- Allows custom knockback from other mods
local registered_on_knockback = {}
function nebula_combat.register_on_knockback(cb)
	table.insert(registered_on_knockback, cb)
end

-- Stamina doesn't have an api to check if a player is sprinting, but it has a callback for it
-- The stamina.players check is for Stamina (Fork)
if stamina and not stamina.players then
    stamina.register_on_sprinting(function(player, sprinting)
        nebula_combat.set_sprinting(player, sprinting, "stamina")
    end)
end


core.register_on_joinplayer(function(player, last_login)
    player_info[player:get_player_name()] = {
        sprinting = false
    }
end)


core.register_on_leaveplayer(function(player, timed_out)
    player_info[player:get_player_name()] = nil
end)


-- Replaces the built-in knockback
local old_calculate_knockback = core.calculate_knockback
function core.calculate_knockback(player, hitter, time_from_last_punch, tool_capabilities, dir, distance, damage)
    if damage == 0 or player:get_armor_groups().immortal then return 0 end

    local horizontal_momentum = 1.0
    local horizontal_force = 0.0
    local vertical_momentum = 1.0
    local vertical_force = 0.0

    if config_custom_knockback_enabled then
        if config_horizontal_different_when_sprinting and nebula_combat.is_sprinting(hitter) then
            horizontal_momentum = config_horizontal_momentum_sprinting
            horizontal_force = config_horizontal_force_sprinting
        else
            horizontal_momentum = config_horizontal_momentum
            horizontal_force = config_horizontal_force
        end

        if config_vertical_different_when_sprinting and nebula_combat.is_sprinting(hitter) then
            vertical_momentum = config_vertical_momentum_sprinting
            vertical_force = config_vertical_force_sprinting
        else
            vertical_momentum = config_vertical_momentum
            vertical_force = config_vertical_force
        end
    end

    for _, cb in ipairs(registered_on_knockback) do
        local value = cb(player, hitter, time_from_last_punch, tool_capabilities, dir, distance, damage)
        if value then
            horizontal_momentum = value.horizontal_momentum or horizontal_momentum
            horizontal_force = value.horizontal_force or horizontal_force
            vertical_momentum = value.vertical_momentum or vertical_momentum
            vertical_force = value.vertical_force or vertical_force
        end
    end

    if horizontal_momentum == 1.0 and horizontal_force == 0.0
        and vertical_momentum == 1.0 and vertical_force == 0.0 then
        if config_custom_knockback_enabled then
            return 0
        else
            return old_calculate_knockback(player, hitter, time_from_last_punch, tool_capabilities, dir, distance, damage)
        end
    end

    local dist_x = -(dir.x * distance)
    local dist_z = -(dir.z * distance)
    local dist_horizontal = math.sqrt(dist_x * dist_x + dist_z * dist_z) -- The argument-provided distance includes the vertical axis
    local velo = player:get_velocity()

    velo.x = (velo.x * horizontal_momentum) - (dist_x / dist_horizontal * horizontal_force)
    velo.z = (velo.z * horizontal_momentum) - (dist_z / dist_horizontal * horizontal_force)

    velo.y = (velo.y * vertical_momentum) + vertical_force

    if config_physics_duration > 0 and (config_speed_mult ~= 1 or config_gravity_mult ~= 1 or config_air_accel_mult ~= 1) then
        if playerphysics then
            playerphysics.add_physics_factor(player, "speed", "nebula_combat:knockback_physics", config_speed_mult)
            playerphysics.add_physics_factor(player, "gravity", "nebula_combat:knockback_physics", config_gravity_mult)
            playerphysics.add_physics_factor(player, "acceleration_air", "nebula_combat:knockback_physics", config_air_accel_mult)

            core.after(config_physics_duration, function()
                playerphysics.remove_physics_factor(player, "speed", "nebula_combat:knockback_physics")
                playerphysics.remove_physics_factor(player, "gravity", "nebula_combat:knockback_physics")
                playerphysics.remove_physics_factor(player, "acceleration_air", "nebula_combat:knockback_physics")
            end)
        elseif pova and config_use_pova_for_physics then
            pova.add_override(name, "nebula_combat:knockback_physics", {
                speed = config_speed_mult,
                gravity = config_gravity_mult,
                acceleration_air = config_air_accel_mult -- Doesn't support air accel but left here incase a future version does
            })
            pova.do_override(player)

            core.after(config_physics_duration, function()
                pova.del_override(name, "nebula_combat:knockback_physics")
                pova.do_override(player)
            end)
        elseif player_monoids and config_use_player_monoids_for_physics then
            player_monoids.speed:add_change(player, config_speed_mult, "nebula_combat:knockback_physics")
            player_monoids.gravity:add_change(player, config_gravity_mult, "nebula_combat:knockback_physics")

            core.after(config_physics_duration, function()
                player_monoids.speed:del_change(player, "nebula_combat:knockback_physics")
                player_monoids.gravity:del_change(player, "nebula_combat:knockback_physics")
            end)
        else
            local orig_physics = player:get_physics_override()
            player:set_physics_override({
                speed = orig_physics.speed * config_speed_mult,
                gravity = orig_physics.gravity * config_gravity_mult,
                acceleration_air = orig_physics.acceleration_air * config_air_accel_mult
            })

            core.after(config_physics_duration, function()
                -- We don't just set it to the old table since it may have changed during the duration time
                orig_physics = player:get_physics_override()
                if config_speed_mult == 0 or config_gravity_mult == 0 or config_air_accel_mult == 0 then
                    -- More of a work-around than a fix, but not much we can do
                    player:set_physics_override({
                        speed = 1,
                        gravity = 1,
                        acceleration_air = 1
                    })
                else
                    player:set_physics_override({
                        speed = orig_physics.speed / config_speed_mult,
                        gravity = orig_physics.gravity / config_gravity_mult,
                        acceleration_air = orig_physics.acceleration_air / config_air_accel_mult
                    })
                end
            end)
        end
    end

    player:add_velocity(-player:get_velocity() + velo)
    return 0
end


if config_custom_attack_distance_enabled then
    -- Overrides the hand to change the attack distance
    core.override_item("", {
        range = config_interact_distance
    })
end
