--====
-- Func
--====

jukesounds = {}
njukesounds = {}
music_player = {}

music_player.register_sound = function(id, name, sound)
    -- Function
    jukesounds[id] = sound
    -- Name for wheel
    table.insert(njukesounds, {id, name})
end

music_player.split_string = function(s,c)
	local list = {}
	if s then
		if c and string.match(s, c) then
			for word in string.gmatch(s, '([^' .. c .. ']+)') do
			    table.insert(list, word)
			end
		else
			table.insert(list, s)
		end
	end
	return list
end

--====
-- Sound
--====

local music_path = minetest.get_modpath("music_player")

-- Piano sounds
dofile(music_path.."/soundpacks/piano_notes.lua")

-- Sounds from mt_game
dofile(music_path.."/soundpacks/default_sounds.lua")

-- Mesecon sounds
dofile(music_path.."/soundpacks/mesecon_sounds.lua")

--====
-- Recipes
--====

local rw = "group:wood"
local rg = "default:glass"
local rm = "default:mese_crystal"

minetest.register_craft({
	output = "music_player:music_player",
	recipe = {
		{rm, rg, rm},
		{rw, "music_player:disc", rw},
		{rm, rw, rm}
	}
})

minetest.register_craft({
	output = "music_player:disc",
	recipe = {
		{"", rg, ""},
		{rg, rm, rg},
		{"", rg, ""}
	}
})

--====
-- GUI
--====

local esc = minetest.formspec_escape

--====
-- ITEM
--====

minetest.register_craftitem("music_player:disc", {
	description = "Empty Disc",
	inventory_image = "music_player_disc.png",
	groups = {disc = 1},
})

minetest.register_tool("music_player:music_disc", {
	description = "Music Disc",
	inventory_image = "music_player_disc.png",
	groups = {disc = 1, music_disc = 1, not_in_creative_inventory = 1},
})

--====
-- FORMSPEC MAKER
--====

local createmain = function(meta, sets, page)
    -- Build Guide
    local formpart = "label[7,1.0;Sounds]"
			
    formpart = formpart .. "tablecolumns[text]"
    formpart = formpart .. "table[7,1.5;7,11.5;jukeguide;"
    formpart = formpart .. "Code"
			
    for i in ipairs(njukesounds) do
        local id = njukesounds[i][1]
        formpart = formpart .. "," .. id
    end
			
    formpart = formpart .. "]"
    -- Give Formspec
    page = page or 1
    local ex = ((tonumber(page) - 1) * 4)
    
    local ops = "On"
    if meta:get_string("status") ~= "on" then
        ops = "Off"
    end
			
    local pla = "From Disc"
    if meta:get_string("from") ~= "disc" then
        pla = "From Box"
    end
    
    local form = "size[15,15]" .. formpart
    
        for i = 1 + ex, 4 + ex do
            local h = 1.5 + (((i - ex) - 1) * 3)
            
            sets = sets or {music = {}, mods = {}}
            sets.music[i] = sets.music[i] or ""
            
            form = form .. "textarea[0.5,".. h ..";6.5,3;set_".. i ..";" .. esc("Set ".. i) .. ";".. esc(sets.music[i]) .."]"..
                "field_enter_after_edit[set_".. i ..";true]"..
		        "field_close_on_enter[set_".. i ..";false]"
        end
        
    	form = form .. "button[0.5,12;5,4;disc;".. esc("Disc") .."]" ..
    	"button[5.0,12;5,4;turn_on;".. esc(ops) .."]" ..
    	"button[9.5,12;5,4;play_from;".. esc(pla) .."]" ..
    	"button[5.0,13;5,4;config;" .. esc("Config") .. "]" ..
    	"button[0.5,13;5,4;guide;" .. esc("Guide") .. "]" ..
   	 "button_exit[9.5,13;5,4;save;" .. esc("Save") .. "]" ..
        "button[5.0,-1;5,4;ppage_main;".. esc("Page " .. page) .. "]"
   
   meta:set_string("stored_formspec", form)
end

local createconfig = function(meta, sets, page)
    page = page or 1
    local ex = ((page - 1) * 4)
    
    local form = "size[8,9]"
    
        for i = 1 + ex, 4 + ex do
            local h = 0.5 + (((i - ex) - 1) * 1.5)
            
            local nd = -0.25 + (i * 0.25)
            if i > 4 then
                nd = 0.125 + ((i - 5) * 0.25)
            end
            
            form = form .. "textarea[0.5,".. h ..";1.75,1;gain_".. i .. ";" .. esc("Gain ".. i) .. ";".. esc(sets.mods[i].g) .."]" ..
	        	"textarea[2.4,".. h ..";1.75,1;fade_".. i .. ";" .. esc("Fade ".. i) .. ";".. esc(sets.mods[i].f) .."]" ..
	        	"textarea[4.3,".. h ..";1.75,1;pitch_".. i .. ";" .. esc("Pitch ".. i) .. ";".. esc(sets.mods[i].p) .."]" ..
	        	"textarea[6.2,".. h ..";1.75,1;delay_".. i .. ";" .. esc("Delay " .. i) .. ";".. esc(sets.mods[i].d) .."]" ..
	            "field_enter_after_edit[gain_".. i ..";true]"..
		        "field_close_on_enter[gain_".. i ..";false]"..
		        "field_enter_after_edit[fade_".. i ..";true]"..
		        "field_close_on_enter[fade_".. i ..";false]"..
		        "field_enter_after_edit[pitch_".. i ..";true]"..
		        "field_close_on_enter[pitch_".. i ..";false]"..
		        "field_enter_after_edit[delay_".. i ..";true]"..
		        "field_close_on_enter[delay_".. i ..";false]"
		end
		
		form = form .. "button[2.5,7.0;3,1;csave;" .. esc("Save") .. "]" ..
        "button[2.5,8.0;3,1;jukebox;" .. esc("Music Player") .. "]"..
        -- Page
        "button[2.5,6.0;3,1;ppage_config;" .. esc("Page ".. page) .. "]"
        
    meta:set_string("stored_formspec", form)
end

--====
--NODE
--====

minetest.register_node("music_player:music_player", {
	description = "Music Player",
	groups = {choppy = 2, oddly_breakable_by_hand = 1},
	paramtype = "light",
	drawtype = "nodebox",
	tiles = {
		"default_wood.png^music_player_disc.png^default_glass.png",
		"default_wood.png",
		"default_wood.png",
		"default_wood.png",
		"default_wood.png",
		"default_wood.png",
	},
	node_box = {
		type = "fixed",
		fixed = {
		    {-0.45, -0.5, -0.45,  0.45, -0.3,  0.45},
		}
	},
	-- Mesecons support
	mesecons = {effector = {
		action_on = function(pos)
			local meta = minetest.get_meta(pos)
			meta:set_string("status", "on")
		end,
		action_off = function(pos)
			local meta = minetest.get_meta(pos)
			meta:set_string("status", "off")
		end
	}},
	--
	on_construct = function(pos)
	    local meta = minetest.get_meta(pos)
	    local inv = meta:get_inventory()
	
	    meta:set_string("status", "off")
	    createmain(meta, sets)
	
	    inv:set_size("disc", 1*1)
	end,
	after_place_node = function(pos, placer)
	    local meta = minetest.get_meta(pos)
	    meta:set_string("owner", placer:get_player_name() or "")
	end,
	can_dig = function(pos,player)
		local meta = minetest.get_meta(pos);
		local inv = meta:get_inventory()
	    return inv:is_empty("disc")
	end,
	on_receive_fields = function(pos, formname, fields, player)
	    local meta = minetest.get_meta(pos)
		local name = player:get_player_name()
		local owner = meta:get_string("owner")
		if owner == name then
	    	-- Initialize sounds
	    	local sets = minetest.deserialize(meta:get_string("sets")) or {music = {}, mods = {}}
	        if meta:get_string("page") == "" then meta:set_string("page", 1) end
	        local page = tonumber(meta:get_string("page"))
	
	        -- Always try to save
	        for i = 1, 8 do
	            if fields["set_"..i] then
		        	sets.music[i] = (fields["set_"..i]:sub(1, 2000)) or ""
	            end
	        end
	        local darinum = function(num)
	            if not num then return nil end
	            if num >= 1 then
	                return 1
	            elseif 0 >= num then
	                return 0
	            end
	            return math.floor(num*100)/100
	        end
		
	        local verinum = function(num)
	            if not num then return nil end
	            if num >= 1.5 then
	                return 1.5
	            elseif -1.5 >= num then
	                return -1.5
 	           end
	            return math.floor(num*100)/100
	        end
	
	        local perinum = function(num)
	            if not num then return nil end
	            if num >= 2.5 then
	                return 2.5
	            elseif -2.5 >= num then
	                return -2.5
 	           end
	            return math.floor(num*100)/100
	        end
	
	        for i = 1, 8 do
	            if fields["gain_"..i] then
	                local nd = -0.25 + (i * 0.25)
	                if i > 4 then
	                    nd = 0.125 + ((i - 5) * 0.25)
	                end
	        	
	                sets.mods[i] = {
		            	g = darinum(tonumber(fields["gain_".. i])) or 1,
		            	f = verinum(tonumber(fields["fade_".. i])) or 0,
		            	p = perinum(tonumber(fields["pitch_".. i])) or 1,
		            	d = darinum(tonumber(fields["delay_".. i])) or nd
		            }
		        end
		    end
		
	        -- Now go
	    	if fields.save then
		    	local ops = "On"
		    	if meta:get_string("status") ~= "on" then
		        	ops = "Off"
		    	end
		    	local pla = "From Disc"
		    	if meta:get_string("from") ~= "disc" then
		        	pla = "From Box"
		    	end
		
		    	createmain(meta, sets, page)
	        -- Go to guide
	    	elseif fields.guide then
		        -- Build Formspec 1
		    	local formparti = "label[3.5,0;Help]"
			
		    	formparti = formparti .. "textarea[0.3,0.5;8,4;jukeguidet;;"
		    	formparti = formparti .. "Basics\nYou have created/bought/found your first music player to make unique music of your own! The process is very simple: Read the sounds \"Codes\" from the panels aside the sets and separate every code you use with a space!\nExamples:\n1: c1 c2 d1\n2: c1 c1 c2\n3: c2 c2 c1\nRemember to save your music before closing the formspec or changing of tabs!\n=====\n"
		    	formparti = formparti .. "Sets\nYou may have noticed there 4 sets, every set plays at different times, after a note of Set 1 plays, a note from Set 2 will play 0.25 seconds after it and so on, you can modify delays in config.\n=====\n"
		    	formparti = formparti .. "Tempo\nWant to create a tempo where no noise is played for a bit without the box resetting? Just use a non-existent code, those don't make sounds! Examples:\n1: c1 l c1\n2: l l c2\nAs it can be seen, the \"Code\" l is mute because it doesn\'t exist and won\'t generate any sound unless provided by a mod\n=====\n"
	    		formparti = formparti .. "Config\nWant to make your music even MORE unique? Try editing some of the configurations! Things like Pitch can change how your music feels completely! the best of all? You can edit every set individually.\n=====\n"
		    	formparti = formparti .. "Discs\nOne of the best parts, the disc, these are stackeable but will become unstackeable as soon as they are modified by the music player! So, please, don't put your whole stack of discs at once.\nTaking that out of the way, discs are the way composed music can be distributed among players, you can give them a title and only you (the producer) will be able to edit or decompile the disc you wrote.\nDiscs will copy everything and store it in themselves, they will copy the sets and the configs.\n=====\n"
			
		    	formparti = formparti .. "Tips\nMusic Players can be triggered by mesecons, however they need a bit more than a second to start producing sound, which means buttons are useless.\nIf you play sound from disc and there no disc then sound from the box will be played instead.\n"
	    		formparti = formparti .. "Not sure how a specific code sounds? Try tapping it! There no need to check the origin, codes usually play their respective sound when you tap on them, something worth noting is that they are the \"original\" which means they are completely unconfigured and will sound differently in a configured set."
			
                formparti = formparti .. "]"
			
		        -- Build Formspec 2
		    	local formpart = "label[0,4.5;Sounds & Their ID]"
			
	   	 	formpart = formpart .. "tablecolumns[text;text]"
		    	formpart = formpart .. "table[0,5.0;8,3;jukeguide;"
                formpart = formpart .. "Code, Sound"
			
		    	for i in ipairs(njukesounds) do
	        		local id = njukesounds[i][1]
		        	local na = njukesounds[i][2]
	    	    	formpart = formpart .. "," .. id ..", "..na
		    	end
			
		    	formpart = formpart .. "]"
		        -- Give Formspec
		        meta:set_string("stored_formspec", "size[8,9]" ..
			        formparti..
			    	formpart..
	                "button[2.5,8.0;3,1;jukebox;" .. esc("Music Player") .. "]")
	    	elseif fields.jukebox then
		        local ops = "On"
		    	if meta:get_string("status") ~= "on" then
		        	ops = "Off"
		    	end
		    	local pla = "From Disc"
	    		if meta:get_string("from") ~= "disc" then
		        	pla = "From Box"
		    	end
		
		        createmain(meta, sets, page)
	        elseif fields.turn_on or fields.play_from then
	
		    	if fields.turn_on then
	                if meta:get_string("status") == "on" then
	                    meta:set_string("status", "off")
	                else
		            	meta:set_string("status", "on")
	                end
		    	elseif fields.play_from then
		        	if meta:get_string("from") == "disc" then
	                    meta:set_string("from", "self")
	                else
		            	meta:set_string("from", "disc")
	                end
		    	end
	
	            for i = 1, 8 do
	                if fields["set_"..i] then
		    			sets.music[i] = (fields["set_"..i]:sub(1, 2000)) or ""
	                end
	            end
			
	            createmain(meta, sets, page)
	        elseif fields.config then
	            for i = 1, 8 do
	                local nd = -0.25 + (i * 0.25)
	                if i > 4 then
	                    nd = 0.125 + ((i - 5) * 0.25)
	                end
	
	                sets.mods[i] = sets.mods[i] or {g = 1, f = 0, p = 1, d = nd}
	            end
	
	            createconfig(meta, sets, page)
	        elseif fields.disc then
	    		meta:set_string("stored_formspec", "size[8,9]" ..
		    	    "textarea[3,0.5;2.5,1;name;" .. esc("Track Name") .. ";]" ..
	    			"list[current_name;disc;3.5,1.5;1,1;]" ..
		    		"button[4.5,3.0;3,1;decompile;" .. esc("Decompile") .. "]" ..
		    		"button[0.5,3.0;3,1;dsave;" .. esc("Save") .. "]" ..
	                "button[2.5,4.0;3,1;jukebox;" .. esc("Music Player") .. "]" ..
		    		"list[current_player;main;0,5;8,1;]" ..
		    		"list[current_player;main;0,6.25;8,3;8]")
	    	elseif fields.decompile then
	    		local inv = meta:get_inventory()
		    	local stack = inv:get_stack("disc", 1)
		    	if stack then
		    		local sdef = stack:get_definition()
		    		local cmeta = stack:get_meta()
		    		local smeta = cmeta
				
	    			if cmeta:get_string("owner") == "" or cmeta:get_string("owner") == name then
		    	    	if sdef and sdef.groups and sdef.groups.disc then
				            sets = minetest.deserialize(smeta:get_string("sets"))
				            minetest.chat_send_player(name, "Decompiled Sucessfully!")
			        	end
			    	else
		        		minetest.chat_send_player(name, "You need to be the composer to decompile the disc!")
			    	end
		    	end
	        elseif fields.dsave then
	            local inv = meta:get_inventory()
		    	local stack = inv:get_stack("disc", 1)
		    	if stack then
		    		local sdef = stack:get_definition()
		    		local cmeta = stack:get_meta()
				
		    		if cmeta:get_string("owner") == "" or cmeta:get_string("owner") == name then
		    	    	if sdef and sdef.groups and sdef.groups.disc then
		    	        	-- Transform into a music disc
			            	local nstack = ItemStack(minetest.registered_items["music_player:music_disc"])
			            	local smeta = nstack:get_meta()
				
			            	smeta:set_string("sets", meta:get_string("sets"))
				
	    	                smeta:set_string("description", "\"" .. fields.name .. "\" By " .. meta:get_string("owner"))
		                    smeta:set_string("owner", meta:get_string("owner"))
		
		                    inv:set_stack("disc", 1, nstack)
			        	end
		    		else
		            	minetest.chat_send_player(name, "You need to be the composer to edit the disc!")
		    		end
		    	end
	        elseif fields.csave then
	        	createconfig(meta, sets, page)
	        elseif fields.jukeguide then
		        local jukevent = minetest.explode_table_event(fields.jukeguide)
		    	if jukevent.type == "CHG" and jukevent.row > 0 then
		            local row = tonumber(jukevent.row)
		        	if row > 1 then -- ignore clicks on the header
			    		local noise = njukesounds[row - 1][2]
		    	    	minetest.sound_play(noise, {
                            pos = pos,
                            max_hear_distance = 25,
                            -- Default noise!
                            gain = 1,
                            fade = 0,
                            pitch = 1})
			    	end
	    		end
	    	elseif fields.ppage_main or fields.ppage_config then
	            local cpage = meta:get_string("page")
	
	            if cpage == "1" then
	                meta:set_string("page", 2)
	            else
	                meta:set_string("page", 1)
	            end
	
	            if fields.ppage_main then
	                createmain(meta, sets, tonumber(meta:get_string("page")))
	            elseif fields.ppage_config then
	                createconfig(meta, sets, tonumber(meta:get_string("page")))
	            end
            end
	        meta:set_string("sets", minetest.serialize(sets))
	        -- Update
	        if not fields.quit then
	            minetest.show_formspec(player:get_player_name(),
                    pos.x .. "," .. pos.y .. "," .. pos.z .. "-,-Music Player",
                    meta:get_string("stored_formspec"))
            end
	    end
	end,
	on_rightclick = function(pos, node, player)
        local meta = minetest.get_meta(pos)
        if player and player:get_player_name() then
            if meta:get_string("owner") == player:get_player_name() then
                -- Recreate main
                local sets = minetest.deserialize(meta:get_string("sets")) or {music = {}, mods = {}}
	            if meta:get_string("page") == "" then meta:set_string("page", 1) end
	            local page = tonumber(meta:get_string("page"))
                createmain(meta, sets, page)
                -- Show
                minetest.show_formspec(player:get_player_name(),
                    pos.x .. "," .. pos.y .. "," .. pos.z .. "-,-Music Player",
                    meta:get_string("stored_formspec"))
            end
        end
    end
})

-- Function that set sets.
local function get_wsets(meta)
    local sets = {
    	music = {},
        mods = {}
    }
    
    local metaset = minetest.deserialize(meta:get_string("sets"))
    for i = 1, 8 do
        local nd = -0.25 + (i * 0.25)
        if i > 4 then
            nd = 0.125 + ((i - 5) * 0.25)
        end
    
        if metaset and metaset.music and metaset.music[i] then
            sets["music"][i] = (music_player.split_string(metaset.music[i], " ")) or {}
            sets["mods"][i] = metaset.mods[i] or {g = 1, f = 0, p = 1, d = nd}
        end
    end

    return sets
end

-- Function that plays jukeboxes
minetest.register_abm({
	label = "Play Music Players",
	nodenames = {"music_player:music_player"},
	interval = 1,
	chance = 1,
	catch_up = false,
	
	action = function(pos, node)
	    local nn = minetest.get_node(pos)
        local meta = minetest.get_meta(pos)
        local cs = meta:get_int("current")
        
        -- Only bother if it is playing
        if meta:get_string("status") == "on" then
            local wsets = get_wsets(meta)
        
            if meta:get_string("from") == "disc" then
                local inv = meta:get_inventory()
	            local stack = inv:get_stack("disc", 1)
	            if stack then
	                local sdef = stack:get_definition()
	                if sdef and sdef.groups and sdef.groups.disc then
	                    local smeta = stack:get_meta()
	
                        wsets = get_wsets(smeta)
                    end
                end
            end
        
            local sh = wsets.music
            local sh2 = wsets.mods
            for i = 1, 8 do
                if sh[i] and sh[i][cs] and jukesounds[sh[i][cs]] then
                    minetest.after(sh2[i].d or 0, function()
                        local noise = jukesounds[sh[i][cs]]
                        minetest.sound_play(noise.s, {
                            pos = pos,
                            max_hear_distance = 25,
                            gain = (noise.g or 1)*(sh2[i].g + 1),
                            fade = (noise.f or 1)*(sh2[i].f + 1),
                            pitch = (noise.p or 1)*(sh2[i].p + 1),
                        })
                    end)
                end
            end
            
            local max = 0
            for i = 1, 8 do
                if sh[i] and #sh[i] > max then
                    max = #sh[i]
                end
            end
        
            if cs >= max then
                meta:set_int("current", 0)
            else
                meta:set_int("current", cs + 1)
            end
         else
            meta:set_int("current", 0)
        end
    end
})

-- Receive

minetest.register_on_player_receive_fields(function(player, formname, fields)
    local fsplit = formname:split("-,-") or nil
    if not fields.quit and fsplit and fsplit[2] and fsplit[2] == "Music Player" then
        local posd = fsplit[1]:split(",")
        
        local npos = {
        	x = tonumber(posd[1]),
            y = tonumber(posd[2]),
            z = tonumber(posd[3])
        }
        
        local node = minetest.get_node(npos)
        local nodef = minetest.registered_nodes[node.name]
        if nodef and nodef.on_receive_fields then
            nodef.on_receive_fields(npos, formname, fields, player)
        end
    end
end)
