-- Mod name: item_hover_display

local hud = {}
local update_interval = 0.5
local next_update = 0
local get_us_time = minetest.get_us_time

-- Caching frequently used functions
local vector_add = vector.add
local vector_multiply = vector.multiply
local floor = math.floor
local format = string.format

-- Function to update the HUD
local function update_hud(player, text)
    local player_name = player:get_player_name()
    local player_hud = hud[player_name]
    if player_hud then
        if player_hud.last_text ~= text then
            player:hud_change(player_hud.id, "text", text)
            player_hud.last_text = text
        end
    else
        hud[player_name] = {
            id = player:hud_add({
                type = "text",  -- Changed from hud_elem_type to type
                position = {x = 0.5, y = 0},
                offset = {x = 0, y = 20},
                text = text,
                scale = {x = 100, y = 100},
                alignment = {x = 0, y = 1},
                number = 0xFFFFFF,
            }),
            last_text = text
        }
    end
end

-- Function to get a friendly name for an entity
local function get_entity_name(ent)
    if not ent then return "Unknown Entity" end
    return ent.description or
           (ent.nametag and ent.nametag ~= "" and ent.nametag) or
           (ent.name and ent.name:gsub("^mobs_mc:", ""):gsub("^%l", string.upper):gsub("_", " ")) or
           "Unknown Entity"
end

-- Function to get entity health percentage
local function get_entity_health(obj)
    local ent = obj:get_luaentity()
    if ent and ent.health and ent.hp_max then
        return floor((ent.health / ent.hp_max) * 100)
    end
    return nil
end

-- Function to get the pointed thing's name and additional info
local function get_pointed_thing_info(player)
    local eye_pos = player:get_pos()
    eye_pos.y = eye_pos.y + player:get_properties().eye_height
    local look_dir = player:get_look_dir()
    local end_pos = vector_add(eye_pos, vector_multiply(look_dir, 5))
    
    local ray = minetest.raycast(eye_pos, end_pos, true, false)
    for pointed_thing in ray do
        if pointed_thing.type == "node" then
            local node = minetest.get_node(pointed_thing.under)
            local node_def = minetest.registered_nodes[node.name]
            return node_def and node_def.description or node.name
        elseif pointed_thing.type == "object" then
            local obj = pointed_thing.ref
            if obj:is_player() and obj ~= player then
                return "Player: " .. obj:get_player_name()
            elseif not obj:is_player() then
                local ent_name = get_entity_name(obj:get_luaentity())
                local health_percent = get_entity_health(obj)
                if health_percent then
                    return format("%s (Health: %d%%)", ent_name, health_percent)
                else
                    return ent_name
                end
            end
        end
    end
    return ""
end

-- Main update function
local function update_all_huds()
    local current_time = get_us_time()
    if current_time < next_update then return end
    next_update = current_time + update_interval * 1000000

    for _, player in ipairs(minetest.get_connected_players()) do
        local info = get_pointed_thing_info(player)
        update_hud(player, info)
    end
end

minetest.register_globalstep(function(dtime)
    update_all_huds()
end)

minetest.register_on_leaveplayer(function(player)
    hud[player:get_player_name()] = nil
end)

minetest.log("action", "[item_hover_display] Mod loaded successfully")