-- Return a configuration option
--
-- The function takes an uprefixed configuration option and a default value.
-- When the option exists in the world-specific configuration object located
-- in `imese.world_config` then this value is returned.
--
-- If it does not exist there, then the global configuration option value is
-- returned, if this is not set either then the given default value is returned.
--
-- @param option  The unprefixed option (gets autoprefixed with `imese_`)
-- @param default The default value to return if the option isn’t set anywhere
-- @return mixed  The configuration value or the default value
imese.get_config = function (option, default)
    local parameter = 'imese_'..option
    local global_setting =  core.settings:get(parameter)
    local world_specific_setting = imese.world_config:get(parameter)
    return world_specific_setting or global_setting or default
end


-- Log a message to the server log
--
-- @see core.log
-- @param message The unprefixed message to log
-- @param level   Optional loglevel, defaults to imese.default_loglevel
imese.log = function (message, level)
    local loglevel = level and level or imese.default_loglevel
    core.log(loglevel, '[imese] '..message)
end


-- Convert a recipe items list into a two-dimensional crafting recipe table
-- for further processing.
--
-- The recipe shape is determined by the width value.
--
-- @see https://github.com/minetest/minetest/issues/7429
-- @see https://github.com/minetest/minetest/issues/8402
-- @see https://forum.luanti.org/viewtopic.php?p=370187#p370187
-- @param items  The items list
-- @param width  The width of the recipe
-- @param method The method the recipe is for
-- @return table The converted recipe
imese.recipe_converter = function (items, width, method)
    local usable_recipe = {}
    local last_index = 0

    -- Get the last index of the recipes table by simple counting
    for index,_ in pairs(items) do
        if last_index < index then
            last_index = index
        end
    end

    -- Return early if the recipe is shapeless, has no items, or is a
    -- cooking recipe.
    if width == 0 or last_index == 0 or method == 'cooking' then
        return items
    end

    -- “Convert” the items table to the usable recipe table.
    for index = 1, math.ceil(last_index / width) * width, 1 do
        local row = (math.ceil(index / width))
        local column = math.fmod(index - 1, width) + 1
        usable_recipe[row] = usable_recipe[row] or {}
        usable_recipe[row][column] = items[index] or ''
    end

    return usable_recipe
end


-- Get a compatibility item
--
-- Some mods might already use a specific recipe so registering recipes might
-- cause some issues with this mods by using the same recipe as those.
--
-- The compatibility item settings allow altering all possibly critical base
-- recipes. Settings are loaded based on the provided variable.
--
--     imese.get_compatibility_item('mese_crystal_mold')
--
-- The above example looks for `imese_compatibility_item_mese_crystal_mold` in
-- `./worlds/worldname/imese.conf` and the server’s configuration file If the
-- setting is found in either of the files it will be used. If the setting is
-- found in both of the files, the world-specific file takes precedence.
--
-- The string set will then be validated to be either an item, a craftitem or
-- a node. Groups are explicitly not allowed here because a group name cannot
-- be used for the replacement table. The item used for compatibility reasons
-- is returned 1:1 on crafting.
--
-- If validation was passed the item will be used on the returned table.
--
--     {
--         replacements = REPLACEMENT_TABLE
--         item = VALID_ITEM_ID
--     }
--
-- The returned `REPLACEMENT_TABLE` is a replacement tables as per Luanti API
-- and the `VALID_ITEM_ID` is the ID of the replacement item as detected by the
-- validation process of this function. If no valid item was found then `item`
-- is an empty string and `replacements` is an empty table. This causes
-- Luanti not to use those when registering the items or nodes.
--
-- @see Luanti API on the built-in Settings() interface
-- @see core.register_node() and core.register_item()
-- @param for_recipe Used to build the configuration option name
-- @return table     The table as described.
imese.get_compatibility_item = function (for_recipe)
    local ci_config = imese.get_config('compatibility_item_'..for_recipe, '')
    local ci_valid_name = ''
    local ci_replacement = {}

    if ci_config ~= '' then
        local compatibility_item_is = {
            item = core.registered_items[ci_config],
            node = core.registered_nodes[ci_config],
            craftitem = core.registered_craftitems[ci_config]
        }

        for _,valid_def in pairs(compatibility_item_is) do
            ci_replacement = {{ valid_def.name, valid_def.name}}
            ci_valid_name = valid_def.name
        end
    end

    if ci_valid_name ~= '' then
        local text = 'Compatibility item for `+t` type recipes is used: +ci'
        imese.log(text:gsub('%+%w+', {
            ['+t'] = for_recipe,
            ['+ci'] = ci_valid_name
        }), 'info')
    end

    return { replacements = ci_replacement, item = ci_valid_name }
end
